/*
 *========================================================================
 * $Id: stat.c 88 2004-09-28 22:49:38Z rgb $
 *
 * See copyright in copyright.h and the accompanying file COPYING
 *========================================================================
 */

#include <wulfware/libwulf.h>

/* id buffer */
static char idbuf[K];

/*
 * <stat> tag init and update (same thing)
 */
void init_stat(Host *hostptr)
{

 int i,j;
 unsigned long seconds,useconds,new_ivalue;

 if((verbose == D_ALL) || (verbose == D_VALUES)){
   fprintf(OUTFP,"D_VALUES: Starting init_stat().  Use -v %d to focus.\n",D_VALUES);
   
 }

 /* 
  * stat - tv_sec and tv_usec timestamps.
  */
 /* <stat tv_sec=?> */
 if(xtract_attribute(UNSIGNED_LONG,&seconds,value_xpath[STAT],
                      "tv_sec",hostptr->xp_doc) > 1){
   fprintf(OUTFP,"fill_values() Warning: %s not unique content tag.\n",
         value_xpath[STAT]);
 }
 if((verbose == D_ALL) || (verbose == D_VALUES)){
   fprintf(OUTFP,"D_VALUES: seconds = %d\n",seconds);
 }
 hostptr->val.stat_tv_sec = seconds;

 /* <stat tv_usec=?> */
 if(xtract_attribute(UNSIGNED_LONG,&useconds,value_xpath[STAT],
                      "tv_usec",hostptr->xp_doc) > 1){
   fprintf(OUTFP,"fill_values() Warning: %s not unique content tag.\n",
         value_xpath[STAT]);
 }
 if((verbose == D_ALL) || (verbose == D_VALUES)){
   fprintf(OUTFP,"D_VALUES: useconds = %d\n",useconds);
 }
 hostptr->val.stat_tv_usec = useconds;
 if((verbose == D_ALL) || (verbose == D_VALUES)){
   fprintf(OUTFP,"D_VALUES: stat_tv_sec = %lu stat_tv_usec = %ul\n",hostptr->val.stat_tv_sec,hostptr->val.stat_tv_usec);
 }

 /* Count the cpus on this host */
 hostptr->val.num_cpus = xtract(COUNT,NULL,value_xpath[CPU],hostptr->xp_doc);
 if((verbose == D_ALL) || (verbose == D_VALUES)){
   fprintf(OUTFP,"D_VALUES: Found %d cpus on %s\n",hostptr->val.num_cpus,hostptr->hostname);
   
 }
 /* Allocate space for them (this needs to be done safely!) */
 hostptr->val.cpu = (Cpu *) ( malloc( (size_t) ( (hostptr->val.num_cpus+1)*sizeof(Cpu) ) ) );
 if((verbose == D_ALL) || (verbose == D_VALUES)){
   fprintf(OUTFP,"D_VALUES: Need %d bytes per cpu\n",sizeof(Cpu));
   fprintf(OUTFP,"D_VALUES: Malloc'd %d bytes in cpu struct vector\n",hostptr->val.num_cpus*sizeof(Cpu));
   fprintf(OUTFP,"D_VALUES: base address is %x\n",&hostptr->val.cpu[0]);
 }
 hostptr->val.cpu++;
 if((verbose == D_ALL) || (verbose == D_VALUES)){
   fprintf(OUTFP,"D_VALUES: displaced base address is now %x\n",hostptr->val.cpu);
   fprintf(OUTFP,"D_VALUES: address of cpu[-1] is %x\n",&hostptr->val.cpu[-1]);
 }

 /* Do all cpus */
 for(j=0;j<hostptr->val.num_cpus;j++){
   if((verbose == D_ALL) || (verbose == D_VALUES)){
     fprintf(OUTFP,"D_VALUES: About to fill host %s: val.cpu[%d].\n",hostptr->hostname,j);
   }

   /* cpu_user */
   sprintf(idbuf,"%s[@id=\"%d\"]/user",value_xpath[CPU],j);
   if( xtract(LONG,&new_ivalue,idbuf,hostptr->xp_doc) > 1 ){
     fprintf(stderr,"init_stat() Warning: %s not unique content tag.\n",
         idbuf);
   }
   hostptr->val.cpu[j].user = new_ivalue;
   if((verbose == D_ALL) || (verbose == D_VALUES)){
     fprintf(OUTFP,"D_VALUES: cpu[%d].user = %d\n",j,hostptr->val.cpu[j].user);
   }
   /* cpu_nice */
   sprintf(idbuf,"%s[@id=\"%d\"]/nice",value_xpath[CPU],j);
   if(xtract(LONG,&new_ivalue,idbuf,hostptr->xp_doc) > 1){
     fprintf(stderr,"init_stat() Warning: %s not unique content tag.\n",
         idbuf);
   }
   hostptr->val.cpu[j].nice = new_ivalue;
   if((verbose == D_ALL) || (verbose == D_VALUES)){
     fprintf(OUTFP,"D_VALUES: cpu[%d].nice = %d\n",j,hostptr->val.cpu[j].nice);
   }
   /* cpu_sys */
   sprintf(idbuf,"%s[@id=\"%d\"]/sys",value_xpath[CPU],j);
   if(xtract(LONG,&new_ivalue,idbuf,hostptr->xp_doc) > 1){
     fprintf(stderr,"init_stat() Warning: %s not unique content tag.\n",
         idbuf);
   }
   hostptr->val.cpu[j].sys = new_ivalue;
   if((verbose == D_ALL) || (verbose == D_VALUES)){
     fprintf(OUTFP,"D_VALUES: cpu[%d].sys = %d\n",j,hostptr->val.cpu[j].sys);
   }
   /* cpu_idle */
   sprintf(idbuf,"%s[@id=\"%d\"]/idle",value_xpath[CPU],j);
   if(xtract(LONG,&new_ivalue,idbuf,hostptr->xp_doc) > 1){
     fprintf(stderr,"On host %s:\n",hostptr->hostname);
     fprintf(stderr,"init_stat() Warning: %s not unique content tag.\n",
         idbuf);
     exit(0);
   }
   hostptr->val.cpu[j].idle = new_ivalue;
   if((verbose == D_ALL) || (verbose == D_VALUES)){
     fprintf(OUTFP,"D_VALUES: cpu[%d].idle = %d\n",j,hostptr->val.cpu[j].idle);
   }
 }

 /* page in, page out */
 if(xtract(LONG,&new_ivalue,value_xpath[PAGE_IN],hostptr->xp_doc) > 1){
   fprintf(stderr,"init_stat() Warning: %s not unique content tag.\n",
       value_xpath[PAGE_IN]);
 }
 hostptr->val.page_in = new_ivalue;
 hostptr->val.page_in_rate = 0.0;
 if((verbose == D_ALL) || (verbose == D_VALUES)){
   fprintf(OUTFP,"D_VALUES: page_in = %d\n",hostptr->val.page_in);
   fprintf(OUTFP,"D_VALUES: page_in_rate = %f\n",hostptr->val.page_in_rate);
 }
 if(xtract(LONG,&new_ivalue,value_xpath[PAGE_OUT],hostptr->xp_doc) > 1){
   fprintf(stderr,"init_stat() Warning: %s not unique content tag.\n",
       value_xpath[PAGE_OUT]);
 }
 hostptr->val.page_out = new_ivalue;
 hostptr->val.page_out_rate = 0.0;
 if((verbose == D_ALL) || (verbose == D_VALUES)){
   fprintf(OUTFP,"D_VALUES: page_out = %d\n",hostptr->val.page_out);
   fprintf(OUTFP,"D_VALUES: page_out_rate = %f\n",hostptr->val.page_out_rate);
 }
 /* swap in, swap out */
 if(xtract(LONG,&new_ivalue,value_xpath[SWAP_IN],hostptr->xp_doc) > 1){
   fprintf(stderr,"init_stat() Warning: %s not unique content tag.\n",
       value_xpath[SWAP_IN]);
 }
 hostptr->val.swap_in = new_ivalue;
 hostptr->val.swap_in_rate = 0.0;
 if((verbose == D_ALL) || (verbose == D_VALUES)){
   fprintf(OUTFP,"D_VALUES: swap_in = %d\n",hostptr->val.swap_in);
   fprintf(OUTFP,"D_VALUES: swap_in_rate = %f\n",hostptr->val.swap_in_rate);
 }
 if(xtract(LONG,&new_ivalue,value_xpath[SWAP_OUT],hostptr->xp_doc) > 1){
   fprintf(stderr,"init_stat() Warning: %s not unique content tag.\n",
       value_xpath[SWAP_OUT]);
 }
 hostptr->val.swap_out = new_ivalue;
 hostptr->val.swap_out_rate = 0.0;
 if((verbose == D_ALL) || (verbose == D_VALUES)){
   fprintf(OUTFP,"D_VALUES: swap_out = %d\n",hostptr->val.swap_out);
   fprintf(OUTFP,"D_VALUES: swap_out_rate = %f\n",hostptr->val.swap_out_rate);
 }

 /* intr, ctxt, processes */
 if(xtract(LONG,&new_ivalue,value_xpath[INTR],hostptr->xp_doc) > 1){
   fprintf(stderr,"init_stat() Warning: %s not unique content tag.\n",
       value_xpath[INTR]);
 }
 hostptr->val.intr = new_ivalue;
 hostptr->val.intr_rate = 0.0;
 if((verbose == D_ALL) || (verbose == D_VALUES)){
   fprintf(OUTFP,"D_VALUES: intr = %d\n",hostptr->val.intr);
 }
 if(xtract(LONG,&new_ivalue,value_xpath[CTXT],hostptr->xp_doc) > 1){
   fprintf(stderr,"init_stat() Warning: %s not unique content tag.\n",
       value_xpath[CTXT]);
 }
 hostptr->val.ctxt = new_ivalue;
 hostptr->val.ctxt_rate = 0.0;
 if((verbose == D_ALL) || (verbose == D_VALUES)){
   fprintf(OUTFP,"D_VALUES: ctxt = %d\n",hostptr->val.ctxt);
 }

 if(xtract(LONG,&new_ivalue,value_xpath[PROCESSES],hostptr->xp_doc) > 1){
   fprintf(stderr,"init_stat() Warning: %s not unique content tag.\n",
       value_xpath[PROCESSES]);
 }
 hostptr->val.processes = new_ivalue;
 hostptr->val.processes_rate = 0.0;
 if((verbose == D_ALL) || (verbose == D_VALUES)){
   fprintf(OUTFP,"D_VALUES: processes = %d\n",hostptr->val.processes);
 }

 if(xtract(LONG,&new_ivalue,value_xpath[PROCS_RUNNING],hostptr->xp_doc) > 1){
   fprintf(stderr,"init_stat() Warning: %s not unique content tag.\n",
       value_xpath[PROCS_RUNNING]);
 }
 hostptr->val.procs_running = new_ivalue;
 if((verbose == D_ALL) || (verbose == D_VALUES)){
   fprintf(OUTFP,"D_VALUES: procs_running = %d\n",hostptr->val.procs_running);
 }

 if(xtract(LONG,&new_ivalue,value_xpath[PROCS_BLOCKED],hostptr->xp_doc) > 1){
   fprintf(stderr,"init_stat() Warning: %s not unique content tag.\n",
       value_xpath[PROCS_BLOCKED]);
 }
 hostptr->val.procs_blocked = new_ivalue;
 if((verbose == D_ALL) || (verbose == D_VALUES)){
   fprintf(OUTFP,"D_VALUES: procs_blocked = %d\n",hostptr->val.procs_blocked);
 }

}


void update_stat(Host *hostptr)
{

 int i,j;
 /* 
  * scratch: current timestamp and delta (to microsecond resolution) 
  */
 double timestamp, delta_time, delta_value;
 unsigned long seconds,useconds,new_ivalue;

 if((verbose == D_ALL) || (verbose == D_VALUES)){
   fprintf(OUTFP,"D_VALUES: Starting init_stat().  Use -v %d to focus.\n",D_VALUES);
   
 }

 /* 
  * stat - tv_sec and tv_usec timestamps.
  */
 /* <stat tv_sec=?> */
 if(xtract_attribute(UNSIGNED_LONG,&seconds,value_xpath[STAT],
                      "tv_sec",hostptr->xp_doc) > 1){
   fprintf(OUTFP,"fill_values() Warning: %s not unique content tag.\n",
         value_xpath[STAT]);
 }
 if((verbose == D_ALL) || (verbose == D_VALUES)){
   fprintf(OUTFP,"D_VALUES: old seconds = %ld\n",hostptr->val.stat_tv_sec);
   fprintf(OUTFP,"D_VALUES: new seconds = %ld\n",seconds);
 }

 /* <stat tv_usec=?> */
 if(xtract_attribute(UNSIGNED_LONG,&useconds,value_xpath[STAT],
                      "tv_usec",hostptr->xp_doc) > 1){
   fprintf(OUTFP,"fill_values() Warning: %s not unique content tag.\n",
         value_xpath[STAT]);
 }
 if((verbose == D_ALL) || (verbose == D_VALUES)){
   fprintf(OUTFP,"D_VALUES: old useconds = %ld\n",hostptr->val.stat_tv_usec);
   fprintf(OUTFP,"D_VALUES: new useconds = %ld\n",useconds);
 }

 /* 
  * There is a small chance that delta_time is negative - if the clock
  * is reset between cycles, for example - and it COULD be zero.  We
  * fix whatever garbage we might find here which will affect only one
  * cycle anyway.  Note that since seconds and useconds are unsigned
  * long, the following MUST be done with the 1.0* or casts or the
  * arithmetic won't come out at all right when a subtraction of two
  * unsigned int's turns out negative.
  */
 delta_time = 1.0*seconds - 1.0*hostptr->val.stat_tv_sec + 1.e-6*useconds - 1.e-6*hostptr->val.stat_tv_usec;
 if(delta_time <= 0.0) delta_time = 1.e-8;	/* avoid divergence */
 timestamp = (double) seconds + 1.e-6*useconds;

 hostptr->val.stat_tv_sec = seconds;
 hostptr->val.stat_tv_usec = useconds;

 if((verbose == D_ALL) || (verbose == D_VALUES)){
   fprintf(OUTFP,"D_VALUES: stat_tv_sec = %lu stat_tv_usec = %lu\n",hostptr->val.stat_tv_sec,hostptr->val.stat_tv_usec);
   fprintf(OUTFP,"D_VALUES: stat_time = %.6f\n",timestamp);
   fprintf(OUTFP,"D_VALUES: delta_time = %.6f\n",delta_time);
 }

 /* 
  * For all values below, save delta_value in rate slot and normalize to
  * rate percentages only at the very end.
  */
 hostptr->val.cpu[-1].user_rate = 0.0;
 hostptr->val.cpu[-1].nice_rate = 0.0;
 hostptr->val.cpu[-1].sys_rate = 0.0;
 hostptr->val.cpu[-1].idle_rate = 0.0;
 hostptr->val.cpu[-1].tot_rate = 0.0;
 for(j=0;j<hostptr->val.num_cpus;j++){
   if((verbose == D_ALL) || (verbose == D_VALUES)){
     fprintf(OUTFP,"D_VALUES: About to fill host %s val.cpu[%d].\n",hostptr->hostname,j);
   }

   /* cpu_user */
   sprintf(idbuf,"%s[@id=\"%d\"]/user",value_xpath[CPU],j);
   if(xtract(LONG,&new_ivalue,idbuf,hostptr->xp_doc) > 1){
     fprintf(OUTFP,"fill_values() Warning: %s not unique content tag.\n",
         idbuf);
   }
   delta_value = (new_ivalue - hostptr->val.cpu[j].user);
   hostptr->val.cpu[j].user_rate = delta_value;
   hostptr->val.cpu[j].user = new_ivalue;
   if((verbose == D_ALL) || (verbose == D_VALUES)){
     fprintf(OUTFP,"D_VALUES: cpu[%d].user = %d, rate = %f\n",j,hostptr->val.cpu[j].user,hostptr->val.cpu[j].user_rate);
   }

   /* cpu_nice */
   sprintf(idbuf,"%s[@id=\"%d\"]/nice",value_xpath[CPU],j);
   if(xtract(LONG,&new_ivalue,idbuf,hostptr->xp_doc) > 1){
     fprintf(OUTFP,"fill_values() Warning: %s not unique content tag.\n",
         idbuf);
   }
   delta_value = (new_ivalue - hostptr->val.cpu[j].nice);
   hostptr->val.cpu[j].nice_rate = delta_value;
   hostptr->val.cpu[j].nice = new_ivalue;
   if((verbose == D_ALL) || (verbose == D_VALUES)){
     fprintf(OUTFP,"D_VALUES: cpu[%d].nice = %d, rate = %f\n",j,hostptr->val.cpu[j].nice,hostptr->val.cpu[j].nice_rate);
   }

   /* cpu_sys */
   sprintf(idbuf,"%s[@id=\"%d\"]/sys",value_xpath[CPU],j);
   if(xtract(LONG,&new_ivalue,idbuf,hostptr->xp_doc) > 1){
     fprintf(OUTFP,"fill_values() Warning: %s not unique content tag.\n",
         idbuf);
   }
   delta_value = (new_ivalue - hostptr->val.cpu[j].sys);
   hostptr->val.cpu[j].sys_rate = delta_value;
   hostptr->val.cpu[j].sys = new_ivalue;
   if((verbose == D_ALL) || (verbose == D_VALUES)){
     fprintf(OUTFP,"D_VALUES: cpu[%d].sys = %d, rate = %f\n",j,hostptr->val.cpu[j].sys,hostptr->val.cpu[j].sys_rate);
   }

   /* cpu_idle */
   sprintf(idbuf,"%s[@id=\"%d\"]/idle",value_xpath[CPU],j);
   if(xtract(LONG,&new_ivalue,idbuf,hostptr->xp_doc) > 1){
     fprintf(OUTFP,"fill_values() Warning: %s not unique content tag.\n",
         idbuf);
   }
   delta_value = (new_ivalue - hostptr->val.cpu[j].idle);
   hostptr->val.cpu[j].idle_rate = delta_value;
   hostptr->val.cpu[j].idle = new_ivalue;
   if((verbose == D_ALL) || (verbose == D_VALUES)){
     fprintf(OUTFP,"D_VALUES: cpu[%d].idle = %d, rate = %f\n",j,hostptr->val.cpu[j].idle,hostptr->val.cpu[j].idle_rate);
   }

   /*
    * I should probably document here how the cpu fields work.
    * First we add up the total number of cycles in the previous
    * delta_time.   These cycles could have been
    * used (as user, nice, or system time) or idle.  To get the FRACTION
    * of available cycles used we therefore divide used cycles by the
    * total.  "tot_rate" is eventually reset to be the sum of the used
    * rates, and "idle_rate" is 1 - tot_rate (all times 100 to get percent).
    * Note that we DO NOT USE delta_time because tot_rate contains the
    * relevant "time" (number of cycles) already.
    */
    /* make cpu_tot and for rate percentages */
   hostptr->val.cpu[j].tot_rate = hostptr->val.cpu[j].user_rate + 
     hostptr->val.cpu[j].nice_rate + hostptr->val.cpu[j].sys_rate +
       hostptr->val.cpu[j].idle_rate;
   hostptr->val.cpu[j].user_rate = 100.0*hostptr->val.cpu[j].user_rate/hostptr->val.cpu[j].tot_rate;
   hostptr->val.cpu[j].nice_rate = 100.0*hostptr->val.cpu[j].nice_rate/hostptr->val.cpu[j].tot_rate;
   hostptr->val.cpu[j].sys_rate = 100.0*hostptr->val.cpu[j].sys_rate/hostptr->val.cpu[j].tot_rate;
   hostptr->val.cpu[j].idle_rate = 100.0*hostptr->val.cpu[j].idle_rate/hostptr->val.cpu[j].tot_rate;
   hostptr->val.cpu[j].tot_rate = 100 - hostptr->val.cpu[j].idle_rate;
   hostptr->val.cpu[-1].user_rate += hostptr->val.cpu[j].user_rate;
   hostptr->val.cpu[-1].nice_rate += hostptr->val.cpu[j].nice_rate;
   hostptr->val.cpu[-1].sys_rate += hostptr->val.cpu[j].sys_rate;
   hostptr->val.cpu[-1].idle_rate += hostptr->val.cpu[j].idle_rate;
   hostptr->val.cpu[-1].tot_rate += hostptr->val.cpu[j].tot_rate;
   if((verbose == D_ALL) || (verbose == D_VALUES)){
     fprintf(OUTFP,"D_VALUES: user_rate nice_rate sys_rate tot_rate idle_rate \n");
     fprintf(OUTFP,"D_VALUES: %6.2f %6.2f %6.2f %6.2f %6.2f \n",
         hostptr->val.cpu[j].user_rate,
         hostptr->val.cpu[j].nice_rate,
         hostptr->val.cpu[j].sys_rate,
         hostptr->val.cpu[j].tot_rate,
         hostptr->val.cpu[j].idle_rate);
   }
 }
 if((verbose == D_ALL) || (verbose == D_VALUES)){
   fprintf(OUTFP,"D_VALUES: TOTAL: user_rate nice_rate sys_rate tot_rate idle_rate \n");
   fprintf(OUTFP,"D_VALUES: %6.2f %6.2f %6.2f %6.2f %6.2f \n",
       hostptr->val.cpu[-1].user_rate,
       hostptr->val.cpu[-1].nice_rate,
       hostptr->val.cpu[-1].sys_rate,
       hostptr->val.cpu[-1].tot_rate,
       hostptr->val.cpu[-1].idle_rate);
 }

 /* page in, page out rates */
 if(xtract(LONG,&new_ivalue,value_xpath[PAGE_IN],hostptr->xp_doc) > 1){
   fprintf(OUTFP,"fill_values() Warning: %s not unique content tag.\n",
       value_xpath[PAGE_IN]);
 }
 delta_value = (new_ivalue - hostptr->val.page_in);
 hostptr->val.page_in_rate = delta_value/delta_time;
 hostptr->val.page_in = new_ivalue;
 if((verbose == D_ALL) || (verbose == D_VALUES)){
   fprintf(OUTFP,"D_VALUES: page_in = %d\n",hostptr->val.page_in);
   fprintf(OUTFP,"D_VALUES: page_in_rate = %f\n",hostptr->val.page_in_rate);
 }

 if(xtract(LONG,&new_ivalue,value_xpath[PAGE_OUT],hostptr->xp_doc) > 1){
   fprintf(OUTFP,"fill_values() Warning: %s not unique content tag.\n",
       value_xpath[PAGE_OUT]);
 }
 delta_value = (new_ivalue - hostptr->val.page_out);
 hostptr->val.page_out_rate = delta_value/delta_time;
 hostptr->val.page_out = new_ivalue;
 if((verbose == D_ALL) || (verbose == D_VALUES)){
   fprintf(OUTFP,"D_VALUES: page_out = %d\n",hostptr->val.page_out);
   fprintf(OUTFP,"D_VALUES: page_out_rate = %f\n",hostptr->val.page_out_rate);
 }
      
/* swap_in, swap_out rates */
 if(xtract(LONG,&new_ivalue,value_xpath[SWAP_IN],hostptr->xp_doc) > 1){
   fprintf(OUTFP,"fill_values() Warning: %s not unique content tag.\n",
       value_xpath[SWAP_IN]);
 }
 delta_value = (new_ivalue - hostptr->val.swap_in);
 hostptr->val.swap_in_rate = delta_value/delta_time;
 hostptr->val.swap_in = new_ivalue;
 if((verbose == D_ALL) || (verbose == D_VALUES)){
   fprintf(OUTFP,"D_VALUES: swap_in = %d\n",hostptr->val.swap_in);
   fprintf(OUTFP,"D_VALUES: swap_in_rate = %f\n",hostptr->val.swap_in_rate);
 }

 if(xtract(LONG,&new_ivalue,value_xpath[SWAP_OUT],hostptr->xp_doc) > 1){
   fprintf(OUTFP,"fill_values() Warning: %s not unique content tag.\n",
       value_xpath[SWAP_OUT]);
 }
 delta_value = (new_ivalue - hostptr->val.swap_out);
 hostptr->val.swap_out_rate = delta_value/delta_time;
 hostptr->val.swap_out = new_ivalue;
 if((verbose == D_ALL) || (verbose == D_VALUES)){
   fprintf(OUTFP,"D_VALUES: swap_out = %d\n",hostptr->val.swap_out);
   fprintf(OUTFP,"D_VALUES: swap_out_rate = %f\n",hostptr->val.swap_out_rate);
 }
      
 /* intr, ctxt, processes */
 if(xtract(UNSIGNED_LONG,&new_ivalue,value_xpath[INTR],hostptr->xp_doc) > 1){
   fprintf(OUTFP,"fill_values() Warning: %s not unique content tag.\n",
       value_xpath[INTR]);
 }
 delta_value = (double)(new_ivalue - hostptr->val.intr);
 if((verbose == D_ALL) || (verbose == D_VALUES)){
   fprintf(OUTFP,"D_VALUES: old intr = %lu\n",hostptr->val.intr);
   fprintf(OUTFP,"D_VALUES: intr = %lu\n",new_ivalue);
   fprintf(OUTFP,"D_VALUES: delta_value = %f\n",delta_value);
 }
 hostptr->val.intr_rate = delta_value/delta_time;
 hostptr->val.intr = new_ivalue;
 if((verbose == D_ALL) || (verbose == D_VALUES)){
   fprintf(OUTFP,"D_VALUES: intr_rate = %f\n",hostptr->val.intr_rate);
 }

 if(xtract(UNSIGNED_LONG,&new_ivalue,value_xpath[CTXT],hostptr->xp_doc) > 1){
   fprintf(OUTFP,"fill_values() Warning: %s not unique content tag.\n",
       value_xpath[PROCESSES]);
 }
 if((verbose == D_ALL) || (verbose == D_VALUES)){
   fprintf(OUTFP,"D_VALUES: old ctxt = %d\n",hostptr->val.ctxt);
 }
 delta_value = (new_ivalue - hostptr->val.ctxt);
 hostptr->val.ctxt_rate = delta_value/delta_time;
 hostptr->val.ctxt = new_ivalue;
 if((verbose == D_ALL) || (verbose == D_VALUES)){
   fprintf(OUTFP,"D_VALUES: ctxt = %d\n",hostptr->val.ctxt);
   fprintf(OUTFP,"D_VALUES: ctxt_rate = %f\n",hostptr->val.ctxt_rate);
 }
      
 if(xtract(UNSIGNED_LONG,&new_ivalue,value_xpath[PROCESSES],hostptr->xp_doc) > 1){
   fprintf(OUTFP,"fill_values() Warning: %s not unique content tag.\n",
       value_xpath[PROCESSES]);
 }
 if((verbose == D_ALL) || (verbose == D_VALUES)){
   fprintf(OUTFP,"D_VALUES: old processes = %d\n",hostptr->val.processes);
 }
 delta_value = (new_ivalue - hostptr->val.processes);
 hostptr->val.processes_rate = delta_value/delta_time;
 hostptr->val.processes = new_ivalue;
 if((verbose == D_ALL) || (verbose == D_VALUES)){
   fprintf(OUTFP,"D_VALUES: processes = %d\n",hostptr->val.processes);
   fprintf(OUTFP,"D_VALUES: processes_rate = %f\n",hostptr->val.processes_rate);
 }

 if(xtract(LONG,&new_ivalue,value_xpath[PROCS_RUNNING],hostptr->xp_doc) > 1){
   fprintf(stderr,"init_stat() Warning: %s not unique content tag.\n",
       value_xpath[PROCS_RUNNING]);
 }
 hostptr->val.procs_running = new_ivalue;
 if((verbose == D_ALL) || (verbose == D_VALUES)){
   fprintf(OUTFP,"D_VALUES: procs_running = %d\n",hostptr->val.procs_running);
 }

 if(xtract(LONG,&new_ivalue,value_xpath[PROCS_BLOCKED],hostptr->xp_doc) > 1){
   fprintf(stderr,"init_stat() Warning: %s not unique content tag.\n",
       value_xpath[PROCS_BLOCKED]);
 }
 hostptr->val.procs_blocked = new_ivalue;
 if((verbose == D_ALL) || (verbose == D_VALUES)){
   fprintf(OUTFP,"D_VALUES: procs_blocked = %d\n",hostptr->val.procs_blocked);
 }

}
