% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/smqr.R
\name{conquer}
\alias{conquer}
\title{Convolution-Type Smoothed Quantile Regression}
\usage{
conquer(
  X,
  Y,
  tau = 0.5,
  kernel = c("Gaussian", "logistic", "uniform", "parabolic", "triangular"),
  h = 0,
  checkSing = FALSE,
  tol = 1e-04,
  iteMax = 5000,
  ci = c("none", "bootstrap", "asymptotic", "both"),
  alpha = 0.05,
  B = 1000
)
}
\arguments{
\item{X}{An \eqn{n} by \eqn{p} design matrix. Each row is a vector of observations with \eqn{p} covariates. Number of observations \eqn{n} must be greater than number of covariates \eqn{p}.}

\item{Y}{An \eqn{n}-dimensional response vector.}

\item{tau}{(\strong{optional}) The desired quantile level. Default is 0.5. Value must be between 0 and 1.}

\item{kernel}{(\strong{optional})  A character string specifying the choice of kernel function. Default is "Gaussian". Choices are "Gaussian", "logistic", "uniform", "parabolic" and "triangular".}

\item{h}{(\strong{optional}) Bandwidth/smoothing parameter. Default is \eqn{\max\{((log(n) + p) / n)^{0.4}, 0.05\}}. The default will be used if the input value is less than or equal to 0.05.}

\item{checkSing}{(\strong{optional}) A logical flag. Default is FALSE. If \code{checkSing = TRUE}, then it will check if the design matrix is singular before running conquer.}

\item{tol}{(\strong{optional}) Tolerance level of the gradient descent algorithm. The iteration will stop when the maximum magnitude of all the elements of the gradient is less than \code{tol}. Default is 1e-04.}

\item{iteMax}{(\strong{optional}) Maximum number of iterations. Default is 5000.}

\item{ci}{(\strong{optional}) A character string specifying methods to construct confidence intervals. Choices are "none" (default), "bootstrap", "asymptotic" and "both". If \code{ci = "none"}, then confidence intervals will not be constructed. 
If \code{ci = "bootstrap"}, then three types of confidence intervals (percentile, pivotal and normal) will be constructed via multiplier bootstrap. 
If \code{ci = "asymptotic"}, then confidence intervals will be constructed based on estimated asymptotic covariance matrix. 
If \code{ci = "both"}, then confidence intervals from both bootstrap and asymptotic covariance will be returned.}

\item{alpha}{(\strong{optional}) Miscoverage level for each confidence interval. Default is 0.05.}

\item{B}{(\strong{optional}) The size of bootstrap samples. Default is 1000.}
}
\value{
An object containing the following items will be returned:
\describe{
\item{\code{coeff}}{A \eqn{(p + 1)}-vector of estimated quantile regression coefficients, including the intercept.}
\item{\code{ite}}{Number of iterations until convergence.}
\item{\code{residual}}{An \eqn{n}-vector of fitted residuals.}
\item{\code{bandwidth}}{Bandwidth value.}
\item{\code{tau}}{Quantile level.}
\item{\code{kernel}}{Kernel function.}
\item{\code{n}}{Sample size.}
\item{\code{p}}{Number of covariates.}
\item{\code{perCI}}{The percentile confidence intervals for regression coefficients. Only available if \code{ci = "bootstrap"} or \code{ci = "both"}.}
\item{\code{pivCI}}{The pivotal confidence intervals for regression coefficients. Only available if \code{ci = "bootstrap"} or \code{ci = "both"}.}
\item{\code{normCI}}{The normal-based confidence intervals for regression coefficients. Only available if \code{ci = "bootstrap"} or \code{ci = "both"}.}
\item{\code{asyCI}}{The asymptotic confidence intervals for regression coefficients. Only available if \code{ci = "asymptotic"} or \code{ci = "both"}.}
}
}
\description{
Estimation and inference for conditional linear quantile regression models using a convolution smoothed approach. Efficient gradient-based methods are employed for fitting both a single model and a regression process over a quantile range. 
Normal-based and (multiplier) bootstrap confidence intervals for all slope coefficients are constructed.
}
\examples{
n = 500; p = 10
beta = rep(1, p)
X = matrix(rnorm(n * p), n, p)
Y = X \%*\% beta + rt(n, 2)

## Smoothed quantile regression with Gaussian kernel
fit.Gauss = conquer(X, Y, tau = 0.5, kernel = "Gaussian")
beta.hat.Gauss = fit.Gauss$coeff

## Smoothe quantile regression with uniform kernel
fit.unif = conquer(X, Y, tau = 0.5, kernel = "uniform")
beta.hat.unif = fit.unif$coeff

## Construct three types of confidence intervals via multiplier bootstrap
fit = conquer(X, Y, tau = 0.5, kernel = "Gaussian", ci = "bootstrap")
ci.per = fit$perCI
ci.piv = fit$pivCI
ci.norm = fit$normCI
}
\references{
Barzilai, J. and Borwein, J. M. (1988). Two-point step size gradient methods. IMA J. Numer. Anal., 8, 141–148.

Fernandes, M., Guerre, E. and Horta, E. (2021). Smoothing quantile regressions. J. Bus. Econ. Statist., 39, 338-357.

He, X., Pan, X., Tan, K. M., and Zhou, W.-X. (2022+). Smoothed quantile regression for large-scale inference. J. Econometrics, in press.

Koenker, R. and Bassett, G. (1978). Regression quantiles. Econometrica, 46, 33-50.
}
\seealso{
See \code{\link{conquer.process}} for smoothed quantile regression process.
}
