/*--------------------------------------------------------------------------*/
/* ALBERTA:  an Adaptive multi Level finite element toolbox using           */
/*           Bisectioning refinement and Error control by Residual          */
/*           Techniques for scientific Applications                         */
/*                                                                          */
/* file:     traverse_r_3d.c                                                */
/*                                                                          */
/* description:                                                             */
/*           recursive mesh traversal - 3d routines:                        */
/*           fill_macro_info_3d(), fill_elinfo_3d()                         */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/*  authors:   Alfred Schmidt                                               */
/*             Zentrum fuer Technomathematik                                */
/*             Fachbereich 3 Mathematik/Informatik                          */
/*             Universitaet Bremen                                          */
/*             Bibliothekstr. 2                                             */
/*             D-28359 Bremen, Germany                                      */
/*                                                                          */
/*             Kunibert G. Siebert                                          */
/*             Institut fuer Mathematik                                     */
/*             Universitaet Augsburg                                        */
/*             Universitaetsstr. 14                                         */
/*             D-86159 Augsburg, Germany                                    */
/*                                                                          */
/*  http://www.mathematik.uni-freiburg.de/IAM/ALBERTA                       */
/*                                                                          */
/*  (c) by A. Schmidt and K.G. Siebert (1996-2003)                          */
/*                                                                          */
/*--------------------------------------------------------------------------*/

/*--------------------------------------------------------------------------*/
/*   Refinement/Coarsening vertex numbering             		    */
/*   like Baensch / Kossaczky                                               */
/*--------------------------------------------------------------------------*/

/*--------------------------------------------------------------------------*/
/*   fill_macro_info_3d:                              		            */
/*   ----------------                                   		    */
/*   fill EL_INFO structure for a macro element         		    */
/*--------------------------------------------------------------------------*/

static void fill_macro_info_3d(MESH *mesh, const MACRO_EL *mel,
			       EL_INFO *elinfo)
{
  FUNCNAME("fill_macro_info_3d");
  EL       *nb;
  MACRO_EL *mnb;
  int      i, j, k;
  FLAGS    fill_opp_coords;

  elinfo->mesh     = mesh;
  elinfo->macro_el = mel;
  elinfo->el       = mel->el;
  elinfo->parent   = nil;
  elinfo->level    = 0;
  elinfo->el_type  = mel->el_type;

  if (elinfo->fill_flag & FILL_COORDS) {

    for (i=0; i<N_VERTICES_3D; i++) {
      DEBUG_TEST_EXIT(mel->coord[i], "no mel->coord[%d]\n",i);
      
      for (j=0;j<DIM_OF_WORLD;j++) elinfo->coord[i][j] = mel->coord[i][j];
    }
  }

  if (elinfo->fill_flag & (FILL_OPP_COORDS | FILL_NEIGH)) {

    fill_opp_coords = (elinfo->fill_flag & FILL_OPP_COORDS);
    for (i=0; i < N_NEIGH_3D; i++) {
      if ((mnb = mel->neigh[i])) {
	nb = elinfo->neigh[i]      = mel->neigh[i]->el;
	k  = elinfo->opp_vertex[i] = mel->opp_vertex[i];

	if (nb->child[0] && (k < 2)) {   /*make nb nearest el.*/
	  if (k==1) {
	    nb = elinfo->neigh[i]      = nb->child[0];
	  } else {
	    nb = elinfo->neigh[i]      = nb->child[1];
	  }
	  k  = elinfo->opp_vertex[i] = 3;
	  if  (fill_opp_coords)
	  {
	    /* always edge between vertices 0 and 1 is bisected! */
	    if (mnb->el->new_coord)
	      for (j=0;j<DIM_OF_WORLD;j++)
		elinfo->opp_coord[i][j] = mnb->el->new_coord[j];
	    else
	      for (j=0;j<DIM_OF_WORLD;j++) elinfo->opp_coord[i][j] =
			 (mnb->coord[0][j] + mnb->coord[1][j]) / 2;
	  }
	}
	else {
	  if  (fill_opp_coords) {
	    for (j=0;j<DIM_OF_WORLD;j++)
	      elinfo->opp_coord[i][j] = mnb->coord[k][j];
	  }
	}
      }
      else {
	elinfo->neigh[i] = nil;
      }
    }
  }

  if (elinfo->fill_flag & FILL_BOUND)
  {
    for (i = 0; i < N_VERTICES_3D; i++)
      elinfo->vertex_bound[i] = mel->vertex_bound[i];

    for (i = 0; i < N_EDGES_3D; i++) 
      elinfo->edge_bound[i] = mel->edge_bound[i];

    for (i = 0; i < N_FACES_3D; i++) 
      elinfo->face_bound[i] = mel->face_bound[i];
  }

  if(elinfo->fill_flag & FILL_PROJECTION) {
    for(i = 0; i < N_FACES_3D + 1; i++)
      elinfo->projections[i] = mel->projection[i];

    elinfo->active_projection = mel->projection[0];

    /* If we have projection functions for faces, these take precedence! */
    if(mel->projection[3])
      elinfo->active_projection = mel->projection[3];
    else if(mel->projection[4])
      elinfo->active_projection = mel->projection[4];
  }

  if (elinfo->fill_flag & FILL_ORIENTATION) 
    elinfo->orientation = mel->orientation;

  return;
}

/*--------------------------------------------------------------------------*/
/*   fill_elinfo_3d:                                   		            */
/*   ------------                                       		    */
/*   fill EL_INFO structure for one child of an element   		    */
/*--------------------------------------------------------------------------*/

static void fill_elinfo_3d(int ichild, const EL_INFO *elinfo_old,
			   EL_INFO *elinfo)
{
  FUNCNAME("fill_elinfo_3d");
  int     i,j,k;
  int     el_type = 0;                             /* el_type in {0,1,2}    */
  int     ochild = 0;                 /* index of other child = 1-ichild    */
  int     *cv = NULL;              /* cv = child_vertex_3d[el_type][ichild] */
  int     (*cvg)[4] = {NULL};             /* cvg = child_vertex_3d[el_type] */
  int     *ce;                       /* ce = child_edge_3d[el_type][ichild] */
  int     iedge;
  EL      *nb, *nbk, **neigh_old;
  EL      *el_old = elinfo_old->el;
  FLAGS   fill_flag = elinfo_old->fill_flag;
  DOF    *dof;
  int     ov;
  EL      **neigh;
  FLAGS   fill_opp_coords;
  U_CHAR  *opp_vertex;

  DEBUG_TEST_EXIT(el_old->child[0], "missing child?\n");  /* Kuni 22.08.96 */

  elinfo->el        = el_old->child[ichild];
  elinfo->macro_el  = elinfo_old->macro_el;
  elinfo->fill_flag = fill_flag;
  elinfo->mesh      = elinfo_old->mesh;
  elinfo->parent    = el_old;
  elinfo->level     = elinfo_old->level + 1;
  elinfo->el_type   = (elinfo_old->el_type + 1) % 3;

  DEBUG_TEST_EXIT(elinfo->el, "missing child %d?\n", ichild);

  if (fill_flag) {
    el_type = elinfo_old->el_type;
    cvg = child_vertex_3d[el_type];
    cv = cvg[ichild];
    ochild = 1-ichild;
  }

  if (fill_flag & FILL_COORDS)
  {
    for (i=0; i<3; i++) {
      for (j = 0; j < DIM_OF_WORLD; j++) {
	elinfo->coord[i][j] = elinfo_old->coord[cv[i]][j];
      }
    }
    if (el_old->new_coord)
      for (j = 0; j < DIM_OF_WORLD; j++) 
	elinfo->coord[3][j] = el_old->new_coord[j];
    else
      for (j = 0; j < DIM_OF_WORLD; j++)
	elinfo->coord[3][j] =
	  (elinfo_old->coord[0][j] + elinfo_old->coord[1][j]) / 2;
  }


  if (fill_flag & (FILL_NEIGH | FILL_OPP_COORDS))
  {
    neigh      = elinfo->neigh;
    neigh_old  = (EL **) elinfo_old->neigh;
    opp_vertex = elinfo->opp_vertex;
    fill_opp_coords = (fill_flag & FILL_OPP_COORDS);

/*----- nb[0] is other child --------------------------------------------*/

/*    if (nb = el_old->child[ochild])   old version  */
    if (el_old->child[0]  &&  (nb = el_old->child[ochild])) /*Kuni 22.08.96*/
    {
      if (nb->child[0])
      {     /* go down one level for direct neighbour */
	if (fill_opp_coords)
	{
	  if (nb->new_coord)
	  {
	    for (j = 0; j < DIM_OF_WORLD; j++)
	      elinfo->opp_coord[0][j] = nb->new_coord[j];
	  }
	  else
	  {
	    k = cvg[ochild][1];
	    for (j = 0; j < DIM_OF_WORLD; j++)
	      elinfo->opp_coord[0][j] =
		(elinfo_old->coord[ochild][j] + elinfo_old->coord[k][j]) / 2;
	  }
	}
	neigh[0]      = nb->child[1];
	opp_vertex[0] = 3;
      }
      else {
	if (fill_opp_coords) {
	  for (j = 0; j < DIM_OF_WORLD; j++) {
	    elinfo->opp_coord[0][j] = elinfo_old->coord[ochild][j];
	  }
	}
	neigh[0]      = nb;
	opp_vertex[0] = 0;
      }
    }
    else {
      ERROR_EXIT("no other child");
      neigh[0] = nil;
    }


/*----- nb[1],nb[2] are childs of old neighbours nb_old[cv[i]] ----------*/

    for (i=1; i<3; i++)
    { 
      if ((nb = neigh_old[cv[i]])) 
      {
	DEBUG_TEST_EXIT(nb->child[0], "nonconforming triangulation\n");

	for (k=0; k<2; k++)  /* look at both childs of old neighbour */
	{       
	  nbk = nb->child[k];
	  if (nbk->dof[0] == el_old->dof[ichild]) {
	    dof = nb->dof[elinfo_old->opp_vertex[cv[i]]]; /* opp. vertex */
	    if (dof == nbk->dof[1]) 
	    {
	      ov = 1;
	      if (nbk->child[0])
	      {
		if (fill_opp_coords)
		{
		  if (nbk->new_coord)
		    for (j = 0; j < DIM_OF_WORLD; j++)
		      elinfo->opp_coord[i][j] = nbk->new_coord[j];
		  else
		    for (j = 0; j < DIM_OF_WORLD; j++)
		      elinfo->opp_coord[i][j] =
			(elinfo_old->opp_coord[cv[i]][j]
			 + elinfo_old->coord[ichild][j]) / 2;
		}
		neigh[i]      = nbk->child[0];
		opp_vertex[i] = 3;
		break;
	      }
	    }
	    else
	    {
	      DEBUG_TEST_EXIT(dof == nbk->dof[2], "opp_vertex not found\n");
	      ov = 2;
	    }

	    if (fill_opp_coords)
	    {
	      for (j = 0; j < DIM_OF_WORLD; j++)
	      {
		elinfo->opp_coord[i][j] = elinfo_old->opp_coord[cv[i]][j];
	      }
	    }
	    neigh[i]      = nbk;
	    opp_vertex[i] = ov;
	    break;
	  }
	  
	} /* end for k */
	DEBUG_TEST_EXIT(k<2, "child not found with vertex\n");
      }
      else 
      {
	neigh[i] = nil;
      }
    }  /* end for i */


/*----- nb[3] is old neighbour neigh_old[ochild] ------------------------*/

    if ((neigh[3] = neigh_old[ochild]))
    {
      opp_vertex[3] = elinfo_old->opp_vertex[ochild];
      if (fill_opp_coords) {
	for (j = 0; j < DIM_OF_WORLD; j++) {
	  elinfo->opp_coord[3][j] = elinfo_old->opp_coord[ochild][j];
	}
      }
    }
  }

  if(fill_flag & FILL_PROJECTION) {
    elinfo->projections[0] = elinfo_old->projections[0];

    elinfo->projections[1] = elinfo_old->projections[0];
    elinfo->projections[2] = elinfo_old->projections[cv[1]+1];
    elinfo->projections[3] = elinfo_old->projections[cv[2]+1];
    elinfo->projections[4] = elinfo_old->projections[ochild+1];

    if(elinfo->projections[3])
      elinfo->active_projection = elinfo->projections[3];
    else if(elinfo->projections[4])
      elinfo->active_projection = elinfo->projections[4];
    else
      elinfo->active_projection = elinfo->projections[0];
  }

  if (fill_flag & FILL_BOUND)
  {
    for (i = 0; i < 3; i++)
    {
      elinfo->vertex_bound[i] = elinfo_old->vertex_bound[cv[i]];
    }
    elinfo->vertex_bound[3] = elinfo_old->edge_bound[0];

    elinfo->face_bound[0] = INTERIOR;
    elinfo->face_bound[1] = elinfo_old->face_bound[cv[1]];
    elinfo->face_bound[2] = elinfo_old->face_bound[cv[2]];
    elinfo->face_bound[3] = elinfo_old->face_bound[ochild];

    ce = child_edge_3d[el_type][ichild];
    for (iedge=0; iedge<4; iedge++) {
      elinfo->edge_bound[iedge] = elinfo_old->edge_bound[ce[iedge]];
    }
    for (iedge=4; iedge<6; iedge++) {
      i = 5 - cv[iedge-3];                /* old vertex opposite new edge */
      elinfo->edge_bound[iedge] = elinfo_old->face_bound[i];
    }
  }

  if (elinfo->fill_flag & FILL_ORIENTATION) {
    elinfo->orientation =
      elinfo_old->orientation * child_orientation_3d[el_type][ichild];
  }

}

/*--------------------------------------------------------------------------*/
/*   AI_update_elinfo_3d:                             		            */
/*   --------------                                       		    */
/*   update EL_INFO structure after refinement (of some neighbours)	    */
/*--------------------------------------------------------------------------*/

void AI_update_elinfo_3d(EL_INFO *elinfo)
{
  FUNCNAME("AI_update_elinfo_3d");
  EL      *el = elinfo->el;

  DEBUG_TEST_EXIT(el, "missing element?\n");

  if (elinfo->fill_flag & (FILL_NEIGH | FILL_OPP_COORDS))
  {
    EL    *nb;
    int   ineigh, ov, j, k;
    FLAGS fill_opp_coords = (elinfo->fill_flag & FILL_OPP_COORDS);
    
    for (ineigh=0; ineigh < N_NEIGH_3D; ineigh++) {
      if ((nb = elinfo->neigh[ineigh])) {
	ov = elinfo->opp_vertex[ineigh];
	if (ov < 2 && nb->child[0]) {
	  if (fill_opp_coords) {
	    k = -1;
	    for (j=0; j<N_VERTICES_3D; j++)
	      if (el->dof[j] == nb->dof[1-ov]) k = j;
	    DEBUG_TEST_EXIT(k >= 0, "neighbour dof not found\n");

	    if (nb->new_coord)
	      for (j = 0; j < DIM_OF_WORLD; j++)
		elinfo->opp_coord[ineigh][j] = nb->new_coord[j];
	    else
	      for (j = 0; j < DIM_OF_WORLD; j++)
		elinfo->opp_coord[ineigh][j] = 
		  (elinfo->opp_coord[ineigh][j] + elinfo->coord[k][j]) / 2;
	  }
	  elinfo->neigh[ineigh]      = nb->child[1-ov];
	  elinfo->opp_vertex[ineigh] = 3;
	}
      }
    }
  }
}
