/*******************************************************************************
 *  PROJECT: Agave
 *
 *  AUTHOR: Jonathon Jongsma
 *
 *  Copyright (c) 2006 Jonathon Jongsma
 *
 *  License:
 *    This program is free software; you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation; either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the 
 *    Free Software Foundation, Inc., 59 Temple Place, Suite 330, 
 *    Boston, MA  02111-1307  USA
 *
 *******************************************************************************/
#include <sstream>
#include <cstring>  // for strlen
#include "palettetreemodel.h"
#include <gtkmm/treepath.h>

namespace gcs
{

    std::string PaletteTreeModel::m_dragDataType = "x-paletteline";

    Glib::RefPtr<PaletteTreeModel> PaletteTreeModel::create(PaletteTreeModel::value_type& palette)
    {
        return Glib::RefPtr<PaletteTreeModel>(new PaletteTreeModel(palette));
    }


    PaletteTreeModel::PaletteTreeModel(PaletteTreeModel::value_type& palette) :
        Glib::ObjectBase(typeid(PaletteTreeModel)), // register custom GType
        Glib::Object(),  // the GType is actually registered here
        ContainerTreeModel<PaletteTreeModel::value_type>(palette)
    {
        m_columnRecord.add(m_text_column);
        m_columnRecord.add(m_color_column);
    }


    int PaletteTreeModel::get_n_columns_vfunc(void) const
    {
        return 2;
    }

    GType PaletteTreeModel::get_column_type_vfunc(int index) const
    {
        if (index == 0)
        {
            return m_text_column.type();
        }
        else if (index == 1)
        {
            return m_color_column.type();
        }
        else
        {
            return 0;
        }
    }


    void PaletteTreeModel::get_value_vfunc(const Gtk::TreeModel::iterator& iter, int column, Glib::ValueBase& value) const
    {
        if (iter_is_valid(iter))
        {
            GlueItem* pItem = (GlueItem*) iter.gobj()->user_data;
            ColorPtr color = *(pItem->iter);
            if (column == 0)
            {
                name_column_type::ValueType val;
                val.init(name_column_type::ValueType::value_type());
                val.set(color->get_name());
                value.init(name_column_type::ValueType::value_type());
                value = val;
            }
            else if (column == 1)
            {
                color_column_type::ValueType val;
                val.init(color_column_type::ValueType::value_type());
                val.set(color);
                value.init(color_column_type::ValueType::value_type());
                value = val;
            }
            else
            {
                g_assert_not_reached();
            }
        }
        else
        {
            switch (column)
            {
                case 0:
                    value.init(name_column_type::ValueType::value_type());
                    break;
                case 1:
                    value.init(color_column_type::ValueType::value_type());
                    break;
            }
        }
    }


    Gtk::TreeModel::Row PaletteTreeModel::append(ColorPtr& clr)
    {
        // add the color to the underlying container
        m_pContainer->colors().push_back(clr);
        Gtk::TreeModel::iterator iter;
        // get the last iter in the list and the path
        iter_nth_root_child_vfunc(m_pContainer->size() - 1, iter);
        Gtk::TreePath path = get_path(iter);
        // emit the row_inserted signal so the view knows that the model has
        // changed and can update itself
        row_inserted(path, iter);
        // increment the stamp to note that the model has changed, so iterators
        // might not be valid anymore
        m_stamp++;
        return *iter;
    }

    Gtk::TreeModel::Row PaletteTreeModel::insert(const Gtk::TreeModel::iterator&
            iter, ColorPtr& color)
    {
        GlueItem* pItem = (GlueItem*) iter.gobj()->user_data;
        value_type::iterator std_iter = pItem->iter;
        m_pContainer->colors().insert(std_iter, color);
        Gtk::TreePath path = get_path(iter);
        row_inserted(path, iter);
        m_stamp++;
        return *iter;
    }


    void PaletteTreeModel::clear(void)
    {
        while (!children().empty())
        {
            // keep erasing the first element in the list until the list is
            // empty
            PaletteTreeModel::iterator iter = children().begin();
            (void) erase(iter);
        }
    }


    PaletteTreeModel::iterator PaletteTreeModel::erase(iterator& iter)
    {
        // next iterator, invalid by default
        iterator iter_next;
        if (iter_is_valid(iter))
        {
            // get a reference to the next iter
            iter_next_vfunc(iter, iter_next);
            // get a path for the iterator we're about to delete, so we can tell
            // the treeview that the path was deleted
            Gtk::TreePath path = get_path(iter);

            GlueItem* pItem = (GlueItem*) iter.gobj()->user_data;
            m_pContainer->colors().erase(pItem->iter);
            // FIXME: delete this?
            delete pItem;
            row_deleted(path);

            // increment the stamp to note that the model has changed, so iterators
            // might not be valid anymore
            ++m_stamp;

        }
        //Gtk::TreeModel::iterator next_iter(iter);
        return iter_next;
    }


void PaletteTreeModel::set_value_impl(const iterator& row, int column,
        const Glib::ValueBase& value)
{
    if (iter_is_valid(row))
    {
        ColorPtr color = row->get_value(m_color_column);
        if (column == 0)
        {
            color->set_name(static_cast<const name_column_type::ValueType&>(value).get());
        }
        /*
            row->set_value(m_text_column, value
            name_column_type::ValueType val;
            val.init(name_column_type::ValueType::value_type());
            val.set(color->get_name());
            value.init(name_column_type::ValueType::value_type());
            value = val;
        }
        else if (column == 1)
        {
            color_column_type::ValueType val;
            val.init(color_column_type::ValueType::value_type());
            val.set(color);
            value.init(color_column_type::ValueType::value_type());
            value = val;
        }
        else
        {
            g_assert_not_reached();
        }
        */

    }
    Gtk::TreePath path = get_path(row);
    row_changed(path, row);
}


bool PaletteTreeModel::drag_data_get_vfunc(const Gtk::TreeModel::Path& path, Gtk::SelectionData& selection_data) const
{
    Gtk::TreeModel::iterator iter;
    get_iter_vfunc(path, iter);
    GlueItem* pItem = (GlueItem*) iter.gobj()->user_data;
    ColorPtr color = *(pItem->iter);
    std::stringstream stream;
    // convert the color object to a string so we can send it with the
    // selection_data
    stream << color;
    // use a custom type identifier
    selection_data.set(m_dragDataType, stream.str());
    return true;
}


bool PaletteTreeModel::drag_data_delete_vfunc(const Gtk::TreeModel::Path& path)
{
    Gtk::TreeModel::iterator iter;
    get_iter_vfunc(path, iter);
    erase(iter);
    return true;
}

bool PaletteTreeModel::drag_data_received_vfunc(const Gtk::TreeModel::Path&
        dest, const Gtk::SelectionData& selection_data)
{
    if (selection_data.get_data_type() != m_dragDataType)
    {
        return false;
    }

    ColorPtr c;
    try
    {
        c = pp::Palette::parse_color(selection_data.get_data_as_string().c_str());
    }
    catch (pp::ParseError& e)
    {
        std::cout << "exception: " << e.what() << std::endl;
        return false;
    }

    Gtk::TreeModel::iterator iter;
    get_iter_vfunc(dest, iter);
    insert(iter, c);
    return true;
}

} // namespace gcs
