/*
    $Id: xwork.c,v 1.33 2001/08/23 04:56:20 belyi Exp $

    xkeysw - window bound/multi code keyboard switch
    Copyright (C) 1999  Dima Barsky <dima@debian.org>,
                        Igor Belyi  <ibelyi@yahoo.com>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/

#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/Xatom.h>
#include <X11/keysym.h>

/* Copied from autoconf info for AC_HEADER_SYS_WAIT macro definition */
#include <sys/types.h>
#if HAVE_SYS_WAIT_H
# include <sys/wait.h>
#endif
#ifndef WEXITSTATUS
# define WEXITSTATUS(stat_val) ((unsigned)(stat_val) >> 8)
#endif
#ifndef WIFEXITED
# define WIFEXITED(stat_val) (((stat_val) & 255) == 0)
#endif

#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <malloc.h>
#include <signal.h>
#include <unistd.h>
#include <string.h>
#include "init.h"
#include "debug.h"

#define XKEYSW_SELECTION "XKeySwSelection"

typedef struct windowList_t {
    Window win;
    codeRecord_t* code;
    struct windowList_t* next;
    struct windowList_t* prev;
} windowList_t;

/*
 * This is a type of WM_STATE property. The value of the 'state' field has
 * the same meaning as the 'initial_state' field in WM_HINTS property.
 * Read ICCCM specification for details.
 */
typedef struct WM_STATE_T {
    unsigned long state;
    Window icon;
} WM_STATE_T;

static void AddWindow(Window w);
static void AddChildren(Window w);
static void DelWindow(Window w);
static Window FindWmAncestor(Window w);
static int window_is_top_level(Window w);
static void add_prefix(Window win, char *prefix);
static void remove_prefix(Window win, char *prefix);
static windowList_t* WinToRecord(Window w);
static codeRecord_t* NewCode(codeRecord_t* currCode, KeyCode kc);
static windowList_t* AddWinRecord(Window w, codeRecord_t* code);
static void DelWinRecord(windowList_t* currRec);
static void ResetAllWindows();

windowList_t* WindowList = NULL;
char* displayname = NULL;
Display* dpy = NULL;
keyMap_t originalKeyMap = {0, NULL};
keyMap_t originalKeyMap2 = {0, NULL};
int max_keycode = -1, min_keycode = 0;
static KeyCode *Modifiers = NULL;
static int MaxKeyPerMod=0;
static int nMods = 0, ShiftLockSep = 0;

static int CallXModMap(char *command[])
{
    char* param[64];
    int i, j=0, status;

    param[j++] = "xmodmap";
    if(displayname) {
	param[j++] = "-display";
	param[j++] = displayname;
    }

    for(i=0; command[i] && j<63; i++)
	param[j++] = command[i];

    param[j] = NULL;

    XSync(dpy, False);

    fflush(stderr);
    switch(fork()){
    case(-1):
#ifdef HAVE_STRERROR
	fprintf(stderr,"fork() failed - %s\n", strerror(errno));
#else
	fprintf(stderr,"fork() failed - %d\n", errno);
#endif
	return -1;
    case(0):
	/* child */
	execvp("xmodmap", param);
#ifdef HAVE_STRERROR
	fprintf(stderr,"execvp(xmodmap, ...) failed - %s\n", strerror(errno));
#else
	fprintf(stderr,"execvp(xmodmap, ...) failed - %d\n", errno);
#endif
	fflush(stderr);
	_exit(1);
    default:
	/* parent */
	;
    }

    wait(&status);
 
    if( !WIFEXITED(status) || WEXITSTATUS(status) )
	return -1;
 
    return 0;
}

keyMap_t GetKeyMap()
{
    keyMap_t resMap;

    resMap.keysyms = XGetKeyboardMapping(dpy,min_keycode,
					 max_keycode-min_keycode+1,
					 &(resMap.keysyms_per_keycode));

    DBG_PRINT1(DBG_PARSING, "GetKeyMap returns '%p'.\n", resMap.keysyms);

    return resMap;
}

int SetXMMKeyMap(char* xmmfile)
{
    DBG_PRINT1(DBG_PARSING, "SetXMMKeyMap is called for '%s'\n", xmmfile);

    if(xmmfile) {
	char* command[2] = {NULL, NULL};
	command[0] = xmmfile;
	return CallXModMap(command);
    }

    return 0;
}
    
void SetKeyMap(keyMap_t keymap)
{
    DBG_PRINT1(DBG_SWITCHKEY, "SetKeyMap is called for %p\n", keymap.keysyms);

    if(keymap.keysyms) {
	XChangeKeyboardMapping(dpy, min_keycode, keymap.keysyms_per_keycode,
			       keymap.keysyms, max_keycode-min_keycode+1);
	/* Make sure that keymap make it there */
	XSync(dpy, False);
    }
}

KeyCode GetKeyCode(char *keyname)
{
    return XKeysymToKeycode(dpy, XStringToKeysym(keyname));
}

int SetNewKeySyms(char* keyName, char* newName)
{
    DBG_PRINT2(DBG_PARSING, "SetNewKeySyms(\"%s\", \"%s\")\n",
	       keyName, newName);

    if( newName && keyName ) {
	char buff[256];
	char* command[3] = {"-e", NULL, NULL};
	command[1] = buff;

	sprintf(buff, "keysym %s = %s", keyName, newName);
	return CallXModMap(command);
    }

    return 0;
}

int   (*old_error_handler)(Display *d,XErrorEvent *ev);

int   X_error_handler(Display *d,XErrorEvent *ev)
{
   /* A window can be deleted between CreateNotify and our request */
    DBG_PRINT1(DBG_SWITCHKEY, "X Error %d is reported.\n", ev->error_code);

   if(ev->error_code==BadWindow
   || ev->error_code==BadDrawable
   || ev->error_code==BadAccess)
      return(-1);
   return(old_error_handler(d,ev));
}

/* RETSIGTYPE: read autoconf info for AC_TYPE_SIGNAL macro definition */
RETSIGTYPE
Interrupt(int x)
{
    DBG_PRINT1(DBG_SWITCHKEY, "Interrupt %d is issued.\n", x);

    ResetAllWindows();
    SetKeyMap(originalKeyMap);
    /* Make sure that everything is reset before we close Display */
    XSync(dpy, False);
    XCloseDisplay (dpy);
    exit(1);
}

void
UpdateModifiersTable()
{
    XModifierKeymap *mmap = XGetModifierMapping(dpy);
    int i;
    if( MaxKeyPerMod != mmap->max_keypermod )
    {
        if( Modifiers != NULL )
            free( Modifiers );

        Modifiers = malloc( sizeof(KeyCode) * mmap->max_keypermod * 8 );
        MaxKeyPerMod = mmap->max_keypermod;
    }

    /* Store Shift/Lock modifiers first */
    for( i = 0, nMods = 0; i < 2*MaxKeyPerMod; i++ )
	if(mmap->modifiermap[i])
	    Modifiers[nMods++] = mmap->modifiermap[i];

    /* Store separator index */
    ShiftLockSep = nMods;

    /* Continue with the rest of modifiers */
    for( i = 2*MaxKeyPerMod; i < 8*MaxKeyPerMod; i++ )
	if(mmap->modifiermap[i])
	    Modifiers[nMods++] = mmap->modifiermap[i];

    XFreeModifiermap( mmap );
}


void InitializeX(char* d, int columns)
{
    int scr, i, nkeys;
    Atom XKeySwSelection;
    Window id;
    KeySym *ptr;

    displayname = d;
    dpy = XOpenDisplay (displayname);
    if (!dpy)
    {
	fprintf (stderr, "InitializeX: unable to open display '%s'\n",
		 XDisplayName (displayname));
	exit (1);
    }

    /* Create an Atom which xkeysw will need to grab before it does anything else */
    XKeySwSelection = XInternAtom(dpy, XKEYSW_SELECTION, False);
    /* Create an invisible window which will be the owner of the Atom */
    id = XCreateSimpleWindow(dpy, DefaultRootWindow(dpy),
			     0, 0, 1, 1, 0, 0L, 0L);

    /* Grab Xserver so that taking Atom/Lock is an atomic procedure */
    XGrabServer(dpy);
    XSync(dpy, False);

    /* Check that there's no owner for the Atom */
    if(XGetSelectionOwner(dpy, XKeySwSelection) != None) {
	fprintf(stderr, "Another xkeysw is already running.\n");
	XUngrabServer(dpy);
	exit (1);
    }

    /* Take the ownership of the Atom. Now, we are in charge! */
    XSetSelectionOwner(dpy, XKeySwSelection, id, CurrentTime);

    /* Release the Xserver. From this point we are sure no other xkeysw will interfere. */
    XUngrabServer(dpy);


    /* Get a copy of original keymap. It will be used as a default code */
    XDisplayKeycodes(dpy,&min_keycode,&max_keycode);
    originalKeyMap.keysyms = XGetKeyboardMapping
	(dpy, min_keycode, max_keycode-min_keycode+1,
	 &(originalKeyMap.keysyms_per_keycode));

    /*
     * Get a second copy of the original keymap.
     * This one will be used when we convert Keysyms to KeyCodes for switching keys.
     * There's a problem when the same Keysym may be assigned to multiple KeyCodes
     * and with this trick we ensure that only the first 'columns' rows of Keysyms is used
     * during the conversion.
     */
    originalKeyMap2.keysyms = XGetKeyboardMapping
	(dpy, min_keycode, max_keycode-min_keycode+1,
	 &(originalKeyMap2.keysyms_per_keycode));

    nkeys = originalKeyMap2.keysyms_per_keycode;
    for( ptr = originalKeyMap2.keysyms;
         ptr < originalKeyMap2.keysyms +
             (max_keycode-min_keycode+1) * nkeys;
         ptr += nkeys )
    {
        for( i = columns; i < nkeys; i++ )
            ptr[i] = 0;
    }

    UpdateModifiersTable();

    /* Handle interrupt signals with our Interrupt handle... */
    signal( SIGINT,  &Interrupt );
    signal( SIGQUIT, &Interrupt );
    signal( SIGTERM, &Interrupt );
    signal( SIGHUP,  &Interrupt );

    /* ...and error interrupts with our X_error_handler */
    old_error_handler=XSetErrorHandler(X_error_handler);

    /*
     * Add root windows from all screens to our record table.
     * This way we'll get notified of all window creations */
    for(scr=0; scr<ScreenCount(dpy); scr++)
      AddWindow(RootWindowOfScreen(ScreenOfDisplay(dpy,scr)));
}

/* Auxilary type to keep set of interesting change masks */
typedef struct mask_t {
    char *name;
    unsigned long mask;
} mask_t;

/* For now we're only interested in FocusChangeMask and PropertyChangeMask */
static mask_t maskArray[] = {
    {"FocusChangeMask", FocusChangeMask},
    {"PropertyChangeMask", PropertyChangeMask},
    {NULL, 0L}
};

static void XKeySwPrintMask(char* msg, Window win)
{
    XWindowAttributes wattr;
    mask_t *maskP;

    if ( win && XGetWindowAttributes(dpy,win,&wattr) ) {
	DBG_PRINT1(DBG_CHANGEMASK, "%s Mask(", msg);
	for(maskP=maskArray; maskP->name; maskP++)
	    if(wattr.your_event_mask & maskP->mask) {
		DBG_PRINT2(DBG_CHANGEMASK, "%s%s",
			   maskP==maskArray?"":"|", maskP->name);
	    }
	DBG_PRINT(DBG_CHANGEMASK, ")\n");
    }
}

void XKeySwMainLoop()
{
    XWindowAttributes wattr;
    Window win;
    XEvent event;
    XKeyEvent *ke = NULL;
    codeRecord_t *newCode;
    windowList_t *currWinRec;
    int i, isDefaultMap = 1, isModifierKey;
    Window focus_win = 0;
    Atom WM_STATE = XInternAtom(dpy, "WM_STATE", 1);
    
    while(1) {
	XNextEvent (dpy, &event);
	switch (event.type) {
	case MappingNotify:
	    DBG_PRINT1(DBG_NOTIFYEVENT, "Event (MappingNotify): request(%s)\n",
		       event.xmapping.request == MappingModifier ?
		       "MappingModifier" :
		       (event.xmapping.request == MappingKeyboard ?
		       "MappingKeyboard" :
		       (event.xmapping.request == MappingPointer ?
		       "MappingPointer" : "(Unknown)")));
            if( event.xmapping.request == MappingModifier )
                UpdateModifiersTable();
            break;
	case MapNotify:
	    win = event.xcreatewindow.window;

	    if (IS_DBG(DBG_NOTIFYEVENT)) {
		DBG_PRINT1(DBG_NOTIFYEVENT,
			   "Event (MapNotify): window(%#lx)\n",
			   win);

		if (IS_DBG(DBG_CHANGEMASK))
		    XKeySwPrintMask("Event (MapNotify):", win);
	    }

	    AddWindow(win);
	    break;
	case UnmapNotify:
	    win = event.xcreatewindow.window;

	    if (IS_DBG(DBG_NOTIFYEVENT)) {
		DBG_PRINT1(DBG_NOTIFYEVENT,
			   "Event (UnmapNotify): window(%#lx)\n",
			   win);

		if (IS_DBG(DBG_CHANGEMASK))
		    XKeySwPrintMask("Event (UnmapNotify):", win);
	    }

	    break;
        case DestroyNotify:
	    win = event.xdestroywindow.window;

	    if (IS_DBG(DBG_NOTIFYEVENT)) {
		DBG_PRINT1(DBG_NOTIFYEVENT,
			   "Event (DestroyNotify): window(%#lx)\n",
			   win);

		if (IS_DBG(DBG_CHANGEMASK))
		    XKeySwPrintMask("Event (DestroyNotify):", win);
	    }

            DelWindow(win);
            break;
        case KeyPress:
            ke = (XKeyEvent *) &event;
            win = FindWmAncestor( ke->window );

	    if (IS_DBG(DBG_KEYEVENT)) {
		DBG_PRINT2(DBG_KEYEVENT,
			   "Event (KeyPress) key(%d), window(%#lx)\n",
			   ke->keycode, win);

		if (IS_DBG(DBG_CHANGEMASK))
		    XKeySwPrintMask("Event (KeyPress):", win);
	    }

            if( win == 0 || !XGetWindowAttributes(dpy,win,&wattr) )
                continue;

	    currWinRec = WinToRecord(win);

#define XKEYSYMCODE(keymap) (keymap.keysyms[keymap.keysyms_per_keycode * \
				    (ke->keycode - min_keycode)])
#define XKEYSYM(keymap) XKeysymToString(XKEYSYMCODE(keymap))

	    DBG_PRINT2(DBG_KEYEVENT, "\tKey corresponds to '%s(%#lx)' "
		       "on display.\n", XKeysymToString(XLookupKeysym(ke, 0)),
		       XLookupKeysym(ke, 0));

	    DBG_PRINT2(DBG_KEYEVENT, "\tKey corresponds to '%s(%#lx)' "
		       "in records.\n", currWinRec ?
		       XKEYSYM(currWinRec->code->keymap) :
		       XKEYSYM(defaultCode->keymap),
		       currWinRec ?
		       XKEYSYMCODE(currWinRec->code->keymap) :
		       XKEYSYMCODE(defaultCode->keymap));

#undef XKEYSYM
#undef XKEYSYMCODE

	    /* Test if pressed key is a modifier */
	    isModifierKey = 0;
            if( currWinRec != NULL )
            {
		/* We ignore Shift and Lock modifiers here */
                for( i=ShiftLockSep; i<nMods; i++ )  
                    if( ke->keycode == Modifiers[i] )
                    {
			isModifierKey = 1;
                        break;
                    }
            }

	    /* Based on the current code and the pressed key find what will be the new code */
	    newCode = NewCode(currWinRec ? currWinRec->code : defaultCode,
			      ke->keycode);

	    /* 
	     * If
	     *     there's no new code defined or
	     *     there's no record for current window and new code should be the original code or
	     *     there's a record for the current window but the new code is the same
	     * then
	     *     we should do nothing with the current code
	     * except that
	     *     if it was a modifier key and window has non-default code which is active
	     *     then we need to activate default code and remember that so that when the modifier
	     *     key get released we will be able to restore the necessary code.
	     */
	    if(newCode == NULL || (newCode==defaultCode && currWinRec==NULL) ||
	       (currWinRec!=NULL && currWinRec->code==newCode)) {
		if ((currWinRec == NULL && !isDefaultMap) || isModifierKey) {
		    isDefaultMap = 1;
		    SetKeyMap( defaultCode->keymap );
		}
		continue;
	    }

	    /*
	     * If
	     *     there was a record for the current window
	     * then
	     *     we need to clean up its prefix from the current window and
	     *     if new code is the default code we are free to delete the window from our records
	     *     and cleanup event triggers for FocusChange and PropertyChange which were
	     *     set when we added the window to our records.
	     * else
	     *     Add the window to our records and setup event triggers for FocusChange and
	     *     PropertyChange events.
	     */
	    if(currWinRec)
            {
		remove_prefix( win, currWinRec->code->prefix);
		
                if(newCode == defaultCode)
                {
                    DelWinRecord(currWinRec);
                    wattr.your_event_mask &=
                        ~(FocusChangeMask|PropertyChangeMask);
                    XSelectInput(dpy,win,wattr.your_event_mask);
                }
                else
                    currWinRec->code = newCode;
            }
	    else
            {
		AddWinRecord(win, newCode);
                wattr.your_event_mask |=
                    (FocusChangeMask|PropertyChangeMask);
                XSelectInput(dpy,win,wattr.your_event_mask);
            }

	    /* Enforce  default code to be active if the pressed key is a modifier. */
	    isDefaultMap = isModifierKey;
	    SetKeyMap(isDefaultMap ? defaultCode->keymap : newCode->keymap);

	    /* Update the window title if newcode is not the default one */
	    if(newCode != defaultCode)
                add_prefix( win, newCode->prefix);

            break;
        case KeyRelease:
            ke = (XKeyEvent *) &event;
            win = FindWmAncestor( ke->window );

	    if (IS_DBG(DBG_KEYEVENT)) {
		DBG_PRINT2(DBG_KEYEVENT,
			   "Event (KeyRelease) key(%d), window(%#lx)\n",
			   ke->keycode, win);

		if (IS_DBG(DBG_CHANGEMASK))
		    XKeySwPrintMask("Event (KeyRelease):", win);
	    }

            if( win == 0 || !XGetWindowAttributes(dpy,win,&wattr) )
                continue;

	    /*
	     * If this is a window which we do not trace then ensure that
	     * the default code is active
	     */
	    if((currWinRec = WinToRecord(win)) == NULL) {
		if (!isDefaultMap) {
		    isDefaultMap = 1;
		    SetKeyMap( defaultCode->keymap );
		}
		continue;
	    }

	    /*
	     * If the released key is a modifier then restore code corresponding
	     * to our record for the window
	     */
	    /* We ignore Shift and Lock modifiers here */
            for( i=ShiftLockSep; i<nMods; i++ )  
                if( ke->keycode == Modifiers[i] )
                {
                    isDefaultMap = 0;
                    SetKeyMap( currWinRec->code->keymap );
                    break;
                }
	    break;

        case FocusIn:
	    /*
	     * We want to know all pressed keys when we enter a window because
	     * we tracking KeyPress/KeyRelease events only for windows with
	     * non default code assigned. For this we need Keymap event which has
	     * key_vector[] containing depressed keys information. The problem is
	     * it does not contains window information. Since the order of events is
	     * FocusIn and then immidiately KeymapNotify we use FocusIn event
	     * to store the windows id which we use in KeymapNotify event
	     * afterwards.
	     */

            focus_win = event.xfocus.window;

	    if (IS_DBG(DBG_FOCUSEVENT)) {
		DBG_PRINT1(DBG_FOCUSEVENT,"Event (FocusIn): window(%#lx)\n",
			   focus_win);

		if (IS_DBG(DBG_CHANGEMASK))
		    XKeySwPrintMask("Event (FocusIn):", focus_win);
	    }

            break;
        case KeymapNotify:
            win = focus_win;
            focus_win = 0;

	    if (IS_DBG(DBG_KEYMAPEVENT)) {
		DBG_PRINT(DBG_KEYMAPEVENT, "Event (KeymapNotify)\n");

		if (IS_DBG(DBG_CHANGEMASK))
		    XKeySwPrintMask("Event (KeymapNotify):", win);
	    }

            if( win == 0 || !XGetWindowAttributes(dpy,win,&wattr) )
                continue;

	    /*
	     * We've enteried a window which we do not track.
	     * Use this oportunity to restore default code.
	     */
	    if((currWinRec = WinToRecord(win)) == NULL) {
		if (!isDefaultMap) {
		    isDefaultMap = 1;
		    SetKeyMap( defaultCode->keymap );
		}
		continue;
	    }

#define KEY_IS_PRESSED(kc) (event.xkeymap.key_vector[(kc)/8]&(1<<((kc)%8)))
	    /* Find out if a modifier key is pressed */
	    /* We ignore Shift and Lock modifiers here */
            for( i=ShiftLockSep; i<nMods; i++ )  
                if( KEY_IS_PRESSED(Modifiers[i]) )
                    break;

#undef KEY_IS_PRESSED

	    /*
	     * !!! I really not quite understand this.... !!!
	     * !!! So, we may have a window in a record with no FocusChangeMask set?! !!!
	     * !!! At what point we can get this? !!!
	     * !!! Besides, how do we get into this event without the bit? !!!
	     */
            if( i == nMods &&
                (wattr.your_event_mask & FocusChangeMask ) != 0 ) {
                isDefaultMap = 0;
                SetKeyMap(currWinRec->code->keymap);
	    }
	    break;
        case FocusOut:
            win = event.xfocus.window;

	    if (IS_DBG(DBG_FOCUSEVENT)) {
		DBG_PRINT1(DBG_FOCUSEVENT, "Event (FocusOut): window(%#lx)\n",
			   win);

		if (IS_DBG(DBG_CHANGEMASK))
		    XKeySwPrintMask("Event (FocusOut):", win);
	    }

            if( win == 0 || !XGetWindowAttributes(dpy,win,&wattr) )
                continue;

	    /* !!! The same questions as in the end of FocusIn event handler. !!! */
            if( (wattr.your_event_mask & FocusChangeMask ) != 0 ) {
                isDefaultMap = 1;
                SetKeyMap(defaultCode->keymap);
	    }
	    break;
        case PropertyNotify:
            win = event.xproperty.window;

	    if (IS_DBG(DBG_NOTIFYEVENT)) {
		DBG_PRINT3(DBG_NOTIFYEVENT, "Event (PropertyNotify): "
			   "window(%#lx) property %s(%#lx)\n",
			   win, XGetAtomName(dpy, event.xproperty.atom),
			   event.xproperty.atom);

		if (IS_DBG(DBG_CHANGEMASK))
		    XKeySwPrintMask("Event (PropertyNotify):", win);
	    }

	    /* Restore FocusChange and PropertyChange masks when
	     * the window is restored from the iconified state.
	     */
	    if((event.xproperty.atom == WM_STATE) &&
	       (currWinRec = WinToRecord(win)) != NULL &&
	       currWinRec->code && currWinRec->code != defaultCode) {
		WM_STATE_T *prop = NULL;
		Atom type;
		int format;
		unsigned long nitems;
		unsigned long bytes_after;
	
		if ( XGetWindowProperty(dpy, win, WM_STATE, 0, 16,
					False, WM_STATE, &type,
					&format, &nitems, &bytes_after,
					(unsigned char **)&prop) == Success &&
		     type == WM_STATE && format == 32 &&
		     nitems == 2 && bytes_after == 0 &&
		     prop->state == NormalState) {
		    wattr.your_event_mask |=
			(FocusChangeMask|PropertyChangeMask);
		    XSelectInput(dpy,win,wattr.your_event_mask);
		}
	    }

	    /*
	     * Implement persistent prefix in the window title.
	     * If window has title changed we add our prefix
	     * !!! Hopefully we are the only persistent guys in the system. :o) !!!
	     */
            if( (event.xproperty.atom == XA_WM_NAME ||
                 event.xproperty.atom == XA_WM_ICON_NAME ) &&
                (currWinRec = WinToRecord(win)) != NULL )
                add_prefix( win, currWinRec->code->prefix );
            break;
	}
    }
}

static windowList_t* WinToRecord(Window w)
{
    windowList_t* ptrW;

    for(ptrW=WindowList; ptrW; ptrW=ptrW->next)
	if(ptrW->win == w)
	    break;
    
    return ptrW;
}

static codeRecord_t* NewCode(codeRecord_t* currCode, KeyCode kc)
{
    switchKey_t* ptrK;
    int i;
    
    for(i=0,ptrK=currCode->switchKeys; i<currCode->switchKeysSize; i++,ptrK++)
	if (ptrK->switchCode == kc)
	    return ptrK->code;

    return NULL;
}

static windowList_t* AddWinRecord(Window w, codeRecord_t* code)
{
    windowList_t* newRec;

    newRec = (windowList_t*)malloc(sizeof(windowList_t));
    if (newRec == NULL) {
	fprintf (stderr, "Cannot allocate another window.\n");
	return NULL;
    }
    newRec->win = w;
    newRec->code = code;
    newRec->next = WindowList;
    newRec->prev = NULL;
    if(WindowList) WindowList->prev = newRec;
    WindowList = newRec;

    return newRec;
}

static void DelWinRecord(windowList_t* currRec)
{
    if (currRec) {
	if(currRec->next)
	    currRec->next->prev = currRec->prev;
	if(currRec->prev)
	    currRec->prev->next = currRec->next;
	else
	    WindowList = currRec->next;
	free(currRec);
    }
}

static void ResetAllWindows()
{
    windowList_t* ptrW;

    for(ptrW=WindowList; ptrW; ptrW=WindowList) {
	WindowList = ptrW->next;
	remove_prefix( ptrW->win, ptrW->code->prefix);
	free(ptrW);
    }

}

static void AddWindow(Window w)
{
   XWindowAttributes wa;

   XSelectInput(dpy,w,0);
       
   if(!XGetWindowAttributes(dpy,w,&wa)) {
       return;
   }

   /* we want to be notified of subwindows creation/destruction etc */
   wa.your_event_mask|=SubstructureNotifyMask;

   /* we request events only if
      1. the window already requested this kind of events
      2. the window is blocking the events from propagation
   */
   wa.your_event_mask|=(wa.all_event_masks|wa.do_not_propagate_mask)
       &(KeyPressMask|KeyReleaseMask);
   
   /* we always want both Press and Release events */
   /* also we are interested in KeymapState to keep track of keys pressed */
   if(wa.your_event_mask&(KeyPressMask|KeyReleaseMask))
       wa.your_event_mask|=(KeyPressMask|KeyReleaseMask|KeymapStateMask);
   
   XSelectInput(dpy,w,wa.your_event_mask);
   
   AddChildren(w);
}

static void AddChildren(Window w)
{
   Window   root, parent, *children;
   unsigned children_num, i;

   if(XQueryTree(dpy,w,&root,&parent,&children,&children_num))
   {
      for(i=0; i<children_num; i++)
         AddWindow(children[i]);
      XFree(children);
   }
}

static void DelWindow(Window w)
{
    DelWinRecord(WinToRecord(w));
}

static Window FindWmAncestor(Window w)
{
    Window   root = 0,*children;
    unsigned children_num;

    while( w != root && w != 0 )
    {
        if( window_is_top_level(w) )
            return w;
        else if( XQueryTree(dpy,w,&root,&w,&children,&children_num) )
            XFree( children );
        else
            break;
    }
    return 0;
}

static int window_is_top_level(Window w)
{
    Atom type_ret;
    int  format_ret;
    unsigned long nitems_ret,bytes_after_ret;
    unsigned char *prop;

   XGetWindowProperty(dpy,w,XA_WM_CLASS,0L,0L,0,XA_STRING,
         &type_ret,&format_ret,&nitems_ret,&bytes_after_ret,&prop);
   return(type_ret!=None);
}

/**************************************************************************
 *  Add a prefix to the window name displayed in the title bar.
 **************************************************************************/
static void add_prefix(Window win, char *prefix)
{
    char *buffer;
    char *win_name = NULL;
    XTextProperty name;

    DBG_PRINT2(DBG_WINPROPERTY, "add_prefix(%#lx, \"%s\")\n", win, prefix);

    if(prefix == NULL || *prefix == '\0')
	return;

    XFetchName( dpy, win, &win_name );

    DBG_PRINT1(DBG_WINPROPERTY, "XFetchName retrieved \"%s\" name.\n",
	       win_name);

    if( win_name != NULL && strncmp(win_name,prefix,strlen(prefix)) != 0 )
    {
        buffer = malloc( strlen(win_name) + strlen(prefix) + 1 );
	if (buffer == NULL) {
	    fprintf(stderr, "Cannot allocate buffer for window name.\n");
	    return;
	}
        strcpy( buffer, prefix );
        strcat( buffer, win_name );

        if (XStringListToTextProperty(&buffer,1,&name) == 0)
        {
            fprintf(stderr,"Can not allocate window name");
            return;
        }
        XSetWMName(dpy,win,&name);
        XSetWMIconName(dpy,win,&name);
        XFree(name.value);
        free( buffer );
    }

    XFree(win_name);
}

/**************************************************************************
 *  Remove a prefix from the window name displayed in the title bar.
 **************************************************************************/
static void remove_prefix(Window win, char *prefix)
{
    char *win_name = NULL;
    XTextProperty name;

    DBG_PRINT2(DBG_WINPROPERTY, "remove_prefix(%#lx, \"%s\")\n", win, prefix);

    if(prefix == NULL || *prefix == '\0')
	return;

    XFetchName( dpy, win, &win_name );

    DBG_PRINT1(DBG_WINPROPERTY, "XFetchName retrieved \"%s\" name.\n",
	       win_name);

    if( win_name != NULL && strncmp(win_name, prefix, strlen(prefix)) == 0 )
    {
        char *new_name = win_name + strlen(prefix);
        if (XStringListToTextProperty(&new_name,1,&name) == 0)
        {
            fprintf(stderr,"Can not allocate window name");
            return;
        }
        XSetWMName(dpy,win,&name);
        XSetWMIconName(dpy,win,&name);
        XFree(name.value);
    }

    XFree(win_name);
}
