/* printpkt.c
 *
 * build something looking like a iptables LOG message
 *
 * (C) 2000-2003 by Harald Welte <laforge@gnumonks.org>
 * (C) 2004 by Michal Kwiatkowski <ruby@joker.linuxstuff.pl>
 */

/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 2 
 *  as published by the Free Software Foundation
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


#include <stdio.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <time.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <arpa/inet.h>
#include <netinet/ip.h>
#include <netinet/ip_icmp.h>
#include <specter/specter.h>
#include "printpkt.h"
#include "lret.h"

/* we rely on arrays, not pointers, so we need different macro here */
#undef IS_VALID
#define IS_VALID(ret_arr,x)	(ret_arr[x].p->flags & SPECTER_RETF_VALID)


#define OTHER_RET_NUM 	7
static specter_local_ret_t ret_other[OTHER_RET_NUM] = {
/*  0 */{ "oob.time_sec", NULL },
	{ "oob.prefix", NULL },
	{ "oob.in", NULL },
	{ "oob.out", NULL },
	{ "raw.mac", NULL },
/*  5 */{ "local.hostname", NULL },
	{ "local.time", NULL },
};
#define IP_RET_NUM	13
static specter_local_ret_t ret_ip[IP_RET_NUM] = {
/*  0 */{ "ip.saddr", NULL },
	{ "ip.daddr", NULL },
	{ "ip.totlen", NULL },
	{ "ip.tos", NULL },
	{ "ip.ttl", NULL },
/*  5 */{ "ip.id", NULL },
	{ "ip.fragoff", NULL },
	{ "ip.protocol", NULL },
	{ "ip.version", NULL },
	{ "ip.rf", NULL },
/* 10 */{ "ip.df", NULL },
	{ "ip.mf", NULL },
	{ "ip.opt", NULL },
};
#define TCP_RET_NUM	16
static specter_local_ret_t ret_tcp[TCP_RET_NUM] = {
/*  0 */{ "tcp.sport", NULL },
	{ "tcp.dport", NULL },
	{ "tcp.seq", NULL },
	{ "tcp.ackseq", NULL },
	{ "tcp.window", NULL },
/*  5 */{ "tcp.res", NULL },
	{ "tcp.urg", NULL },
	{ "tcp.ack", NULL },
	{ "tcp.psh", NULL },
	{ "tcp.rst", NULL },
/* 10 */{ "tcp.syn", NULL },
	{ "tcp.fin", NULL },
	{ "tcp.urgp", NULL },
	{ "tcp.cwr", NULL },
	{ "tcp.ece", NULL },
/* 15 */{ "tcp.opt", NULL },
};
#define UDP_RET_NUM	3
static specter_local_ret_t ret_udp[UDP_RET_NUM] = {
/*  0 */{ "udp.sport", NULL },
	{ "udp.dport", NULL },
	{ "udp.len", NULL },
};
#define ICMP_RET_NUM	6
static specter_local_ret_t ret_icmp[ICMP_RET_NUM] = {
/*  0 */{ "icmp.type", NULL },
	{ "icmp.code", NULL },
	{ "icmp.echoid", NULL },
	{ "icmp.echoseq", NULL },
	{ "icmp.gateway", NULL },
/*  5 */{ "icmp.fragmtu", NULL },
};
#define AHESP_RET_NUM	1
static specter_local_ret_t ret_ahesp[AHESP_RET_NUM] = {
	{ "ahesp.spi", NULL },
};


static int print_dev(char *buf, int flags)
{
	char *buf_cur = buf;

	buf_cur += sprintf(buf_cur, " IN=%s OUT=%s ", 
		(char *) GET_VALUE(ret_other,2).ptr, 
		(char *) GET_VALUE(ret_other,3).ptr);

	/* optional logging of mac header */
	if (flags & PRINTPKT_MAC_HDR)
		buf_cur += sprintf(buf_cur, "MAC=%s ",
			IS_VALID(ret_other,4) ? (char *) GET_VALUE(ret_other,4).ptr : "");

	return (buf_cur - buf);
}

static int print_ipv4(char *buf, int flags)
{
	char *buf_cur = buf;

	buf_cur += sprintf(buf_cur, "SRC=%s ",
			inet_ntoa((struct in_addr) {htonl(GET_VALUE(ret_ip,0).ui32)}));
	buf_cur += sprintf(buf_cur, "DST=%s ",
			inet_ntoa((struct in_addr) {htonl(GET_VALUE(ret_ip,1).ui32)}));

	buf_cur += sprintf(buf_cur, "LEN=%u TOS=0x%02X PREC=0x%02X TTL=%u ID=%u ", 
			GET_VALUE(ret_ip,2).ui16, GET_VALUE(ret_ip,3).ui8 & IPTOS_TOS_MASK, 
			GET_VALUE(ret_ip,3).ui8 & IPTOS_PREC_MASK, GET_VALUE(ret_ip,4).ui8,
			GET_VALUE(ret_ip,5).ui16);

	if (GET_VALUE(ret_ip,9).b) 
		buf_cur += sprintf(buf_cur, "CE ");

	if (GET_VALUE(ret_ip,10).b)
		buf_cur += sprintf(buf_cur, "DF ");

	if (GET_VALUE(ret_ip,11).b)
		buf_cur += sprintf(buf_cur, "MF ");

	if (GET_VALUE(ret_ip,6).ui16)
		buf_cur += sprintf(buf_cur, "FRAG:%u ",
				GET_VALUE(ret_ip,6).ui16);

	/* optional logging of ip options */
	if ((flags & PRINTPKT_IP_OPT) && IS_VALID(ret_ip,12))
		buf_cur += sprintf(buf_cur, "OPT (%s) ", (char *)GET_VALUE(ret_ip,12).ptr);

	switch (GET_VALUE(ret_ip,7).ui8) {

		case IPPROTO_TCP:
			buf_cur += sprintf(buf_cur, "PROTO=TCP ");
			buf_cur += sprintf(buf_cur, "SPT=%u DPT=%u ",
					GET_VALUE(ret_tcp,0).ui16, GET_VALUE(ret_tcp,1).ui16);
			/* optional logging of tcp seq numbers */
			if (flags & PRINTPKT_TCP_SEQ)
				buf_cur += sprintf(buf_cur, "SEQ=%u ACK=%u ",
					GET_VALUE(ret_tcp,2).ui32, GET_VALUE(ret_tcp,3).ui32);

			buf_cur += sprintf(buf_cur, "WINDOW=%u ", GET_VALUE(ret_tcp,4).ui16);

			buf_cur += sprintf(buf_cur, "RES=0x%02x ", GET_VALUE(ret_tcp,5).ui8);

			if (GET_VALUE(ret_tcp,13).b)
				buf_cur += sprintf(buf_cur, "CWR ");

			if (GET_VALUE(ret_tcp,14).b)
				buf_cur += sprintf(buf_cur, "ECE ");

			if (GET_VALUE(ret_tcp,6).b)
				buf_cur += sprintf(buf_cur, "URG ");

			if (GET_VALUE(ret_tcp,7).b)
				buf_cur += sprintf(buf_cur, "ACK ");

			if (GET_VALUE(ret_tcp,8).b)
				buf_cur += sprintf(buf_cur, "PSH ");

			if (GET_VALUE(ret_tcp,9).b)
				buf_cur += sprintf(buf_cur, "RST ");

			if (GET_VALUE(ret_tcp,10).b)
				buf_cur += sprintf(buf_cur, "SYN ");

			if (GET_VALUE(ret_tcp,11).b)
				buf_cur += sprintf(buf_cur, "FIN ");

			buf_cur += sprintf(buf_cur, "URGP=%u ", GET_VALUE(ret_tcp,12).ui16);

			/* optional logging of tcp options */
			if ((flags & PRINTPKT_TCP_OPT) && IS_VALID(ret_tcp,15))
				buf_cur += sprintf(buf_cur, "OPT (%s) ",
						(char *)GET_VALUE(ret_tcp,15).ptr);

			break;

		case IPPROTO_UDP:

			buf_cur += sprintf(buf_cur, "PROTO=UDP ");

			buf_cur += sprintf(buf_cur, "SPT=%u DPT=%u LEN=%u ", 
					GET_VALUE(ret_udp,0).ui16, GET_VALUE(ret_udp,1).ui16,
					GET_VALUE(ret_udp,2).ui16);
			break;

		case IPPROTO_ICMP:

			buf_cur += sprintf(buf_cur, "PROTO=ICMP ");

			buf_cur += sprintf(buf_cur, "TYPE=%u CODE=%u ",
					GET_VALUE(ret_icmp,0).ui8, GET_VALUE(ret_icmp,1).ui8);

			switch (GET_VALUE(ret_icmp,0).ui8) {
				case ICMP_ECHO:
				case ICMP_ECHOREPLY:
					buf_cur += sprintf(buf_cur, "ID=%u SEQ=%u ",
						GET_VALUE(ret_icmp,2).ui16,
						GET_VALUE(ret_icmp,3).ui16);
					break;
				case ICMP_PARAMETERPROB:
					buf_cur += sprintf(buf_cur, "PARAMETER=%u ",
						GET_VALUE(ret_icmp,4).ui32 >> 24);
					break;
				case ICMP_REDIRECT:
					buf_cur += sprintf(buf_cur, "GATEWAY=%s ",
							inet_ntoa((struct in_addr) {htonl(GET_VALUE(ret_icmp,4).ui32)}));
					break;
				case ICMP_DEST_UNREACH:
					if (GET_VALUE(ret_icmp,1).ui8 == ICMP_FRAG_NEEDED)
						buf_cur += sprintf(buf_cur, "MTU=%u ",
							GET_VALUE(ret_icmp,5).ui16);
					break;
			}
			break;

		case IPPROTO_ESP:
		case IPPROTO_AH:
			buf_cur += sprintf(buf_cur, "PROTO=%s ",
					GET_VALUE(ret_ip,7).ui8 == IPPROTO_ESP ? "ESP" : "AH");
			buf_cur += sprintf(buf_cur, "SPI=0x%x ", GET_VALUE(ret_ahesp,0).ui32);
			break;

		default:

			buf_cur += sprintf(buf_cur, "PROTO=%u ", GET_VALUE(ret_ip,7).ui8);
	}

	return (buf_cur - buf);
}

int printpkt_print(char *buf, int flags)
{
	char *buf_cur = buf;

	if (flags & PRINTPKT_PREFIX) {
		char *timestr;
		char *tmp;
		time_t now;

		/* use time from ULOG if possible, otherwise log current time */
		if (IS_VALID(ret_other,0))
			now = (time_t) GET_VALUE(ret_other,0).ui32;
		else
			now = (time_t) GET_VALUE(ret_other,6).ui32;

		timestr = ctime(&now) + 4;

		/* truncate time */
		if ((tmp = strchr(timestr, '\n')))
			*tmp = '\0';

		/* print time and hostname */
		buf_cur += sprintf(buf_cur, "%.15s %s:", timestr, (char *)GET_VALUE(ret_other,5).ptr);

		/* print prefix */
		if (*(char *) GET_VALUE(ret_other,1).ptr)
			buf_cur += sprintf(buf_cur, " %s", (char *) GET_VALUE(ret_other,1).ptr);
	}

	buf_cur += print_dev(buf_cur, flags);

	if (GET_VALUE(ret_ip,8).ui8 == 4) {
		buf_cur += print_ipv4(buf_cur, flags);
	}
	else
		return 0;

	strcat(buf_cur++, "\n");

	return (buf_cur - buf);
}

int printpkt_init(void)
{
	if (fill_local_ret(ret_other, OTHER_RET_NUM) == -1)
		return -1;
	if (fill_local_ret(ret_ip, IP_RET_NUM) == -1)
		return -1;
	if (fill_local_ret(ret_tcp, TCP_RET_NUM) == -1)
		return -1;
	if (fill_local_ret(ret_udp, UDP_RET_NUM) == -1)
		return -1;
	if (fill_local_ret(ret_icmp, ICMP_RET_NUM) == -1)
		return -1;
	if (fill_local_ret(ret_ahesp, AHESP_RET_NUM) == -1)
		return -1;

	return 0;
}

