/* some code borrowed from "Making application programming easy with GNOME libraries"
 * by George Lebl
 * http://www-4.ibm.com/software/developer/library/gnome2/index.html?dwzone=linux
 */

#include "../common.h"
#include "configuration.h"
#include "../path.h"
#include "../root-portal.h"
#include "../root-portal-tiny.xpm"
#include "preferences.h"

#include <gdk/gdkx.h>
#include <gdk/gdk.h>
#include <X11/Xlib.h>

#define SAVE_TIP N_("Save configuration")
#define CLOSE_TIP N_("Close configurator window")
#define EXIT_TIP N_("Shutdown Root-Portal")
#define PREFERENCES_TIP N_("Select preferences")

static void save_cb(GtkWidget *widget, gpointer data)
{
        rootPortal->saveConfig();
}

static void close_cb(GtkWidget *widget, Configuration *data)
{
        data->closeConfigurator();
}

static void about_cb(GtkWidget *widget, Configuration *data)
{
        data->display_about();
}

static void preferences_cb(GtkWidget *widget, Configuration *data)
{
        data->display_preferences();
}

/* when inserting items or re-ordering, remember to update the
 * object pointer setting (user data) at the start of main */
static GnomeUIInfo file_menu[] = {
        {
                GNOME_APP_UI_ITEM,
                N_("_Save config"), SAVE_TIP,
                (gpointer) save_cb, NULL, NULL,
                GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_SAVE,
                0, (GdkModifierType) 0, NULL
        },
        GNOMEUIINFO_SEPARATOR,
        {
                GNOME_APP_UI_ITEM,
                N_("_Close"), CLOSE_TIP,
                (gpointer) close_cb, NULL, NULL,
                GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_CLOSE,
                GDK_W, GDK_CONTROL_MASK, NULL
        },
        {
                GNOME_APP_UI_ITEM,
                N_("E_xit"), EXIT_TIP,
                (gpointer) &gtk_main_quit, NULL, NULL,
                GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_EXIT,
                GDK_Q, GDK_CONTROL_MASK, NULL
        },
        GNOMEUIINFO_END
};

GnomeUIInfo settings_menu[] = {
        {
                GNOME_APP_UI_ITEM,
                N_("_Preferences"), PREFERENCES_TIP,
                (gpointer) &preferences_cb, NULL, NULL,
                GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_PREF,
                0, (GdkModifierType) 0, NULL
        },
        GNOMEUIINFO_END
};

GnomeUIInfo help_menu[] = {
        GNOMEUIINFO_HELP((void *) "root-portal"),
        GNOMEUIINFO_SEPARATOR,
        {
                GNOME_APP_UI_ITEM, 
                N_("About"), N_("Info about this program"),
                (gpointer) &about_cb, NULL, NULL, 
                GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_ABOUT,
                0, (GdkModifierType) 0, NULL
        },
        GNOMEUIINFO_END
};

static GnomeUIInfo main_menu[] = {
        GNOMEUIINFO_MENU_FILE_TREE(file_menu),
        GNOMEUIINFO_MENU_SETTINGS_TREE(settings_menu),
        GNOMEUIINFO_MENU_HELP_TREE(help_menu),
        GNOMEUIINFO_END
};

static GnomeUIInfo toolbar [] = {
        GNOMEUIINFO_ITEM_STOCK("Save", SAVE_TIP,
                              (gpointer) save_cb, GNOME_STOCK_PIXMAP_SAVE),
        GNOMEUIINFO_ITEM_STOCK("Close", CLOSE_TIP,
                              (gpointer) close_cb, GNOME_STOCK_PIXMAP_CLOSE),
        GNOMEUIINFO_ITEM_STOCK("Exit", EXIT_TIP,
                              (gpointer) gtk_main_quit, GNOME_STOCK_PIXMAP_EXIT),
        GNOMEUIINFO_END
};

static void destroyed_(void* a, void* b, Configuration* t)
{
        t->closeConfigurator();
}

Configuration::Configuration()
        : configuratorActive(false),
          modtree_display(*this),
          focusedModule(0)
{
        help_menu[2].user_data = this;
        file_menu[2].user_data = this;
        toolbar[1].user_data = this;
        settings_menu[0].user_data = this;

        window = gnome_app_new(PACKAGE " " VERSION, PACKAGE " " VERSION " configurator");
        gtk_signal_connect(GTK_OBJECT(window), "delete_event", GTK_SIGNAL_FUNC(destroyed_), this);
        vbox = gtk_vbox_new(false, 0);
        
        gnome_app_create_menus(GNOME_APP(window), main_menu);
        gnome_app_create_toolbar(GNOME_APP(window), toolbar);

        appbar = gnome_appbar_new(FALSE, TRUE, GNOME_PREFERENCES_USER);
        gnome_app_set_statusbar(GNOME_APP(window), appbar);
        gnome_app_install_menu_hints(GNOME_APP(window), main_menu);
        
        moduleBook = gtk_notebook_new();
        
        paned = gtk_hpaned_new();
        gtk_paned_pack1(GTK_PANED(paned), modtree_display.getWidget(), true, true);
        gtk_paned_pack2(GTK_PANED(paned), moduleBook, true, true);
        gtk_paned_set_position(GTK_PANED(paned), 140);
        gtk_box_pack_start(GTK_BOX(vbox), paned, true, true, 0);
        gnome_app_set_contents(GNOME_APP(window), vbox);
        gtk_widget_set_usize(window, 600, 500);

        moduleFocus(Path());
        modtree_display.load();

        ModuleTree::instance()->register_listener(this);
}

void Configuration::set_window_icon(GtkWidget* win)
{
        /* setup icon -- this bit taken from xmms */
        static GdkPixmap *icon;
        static GdkBitmap *mask;
        Atom icon_atom;
        glong data[2];
        
        if (!icon)
                icon = gdk_pixmap_create_from_xpm_d (win->window, &mask, &win->style->bg[GTK_STATE_NORMAL], root_portal_tiny_xpm);
        data[0] = GDK_WINDOW_XWINDOW(icon);
        data[1] = GDK_WINDOW_XWINDOW(mask);
        
        icon_atom = gdk_atom_intern("KWM_WIN_ICON", false);
        gdk_property_change(win->window, icon_atom, icon_atom, 32,
                            GDK_PROP_MODE_REPLACE, (guchar *)data, 2);        
}


Configuration::~Configuration()
{
        ModuleTree::instance()->unregister_listener(this);
        for (unsigned int i = 0; i < modmodif.size(); i++)
                delete modmodif[i];
        modmodif.erase(modmodif.begin(), modmodif.end());
}

void Configuration::displayConfigurator()
{
        if (configuratorActive)
                return;
        configuratorActive = true;
        gtk_widget_realize(window);
        set_window_icon(window);
        gtk_widget_show_all(window);
}

void Configuration::closeConfigurator()
{
        if (configuratorActive)
                gtk_widget_hide_all(window);
        configuratorActive = false;
}

void Configuration::moduleFocus(const Path& mod)
{
        if (focusedModule != 0) {
                if (!focusedModule->getChanged())
                        closeModule(focusedModule->getModulePath());
        }

        // if it is alread open, use it
        for (unsigned int i = 0; i < modmodif.size(); i++) {
                if (modmodif[i]->getModulePath() == mod) {
                        int pn = gtk_notebook_page_num(GTK_NOTEBOOK(moduleBook), modmodif[i]->getWidget());
                        gtk_notebook_set_page(GTK_NOTEBOOK(moduleBook), pn);
                        focusedModule = modmodif[i];
                        return;
                }
        }

        // otherwise create a new one
        try {
                ModuleModifier* newModif = new ModuleModifier(*this, mod);
                GtkWidget* label = gtk_label_new(ModuleTree::instance()->get_type(mod).c_str());
                gtk_notebook_append_page(GTK_NOTEBOOK(moduleBook), newModif->getWidget(), label);
                gtk_widget_show_all(moduleBook);
                int pn = gtk_notebook_page_num(GTK_NOTEBOOK(moduleBook), newModif->getWidget());
                gtk_notebook_set_page(GTK_NOTEBOOK(moduleBook), pn);
                modmodif.push_back(newModif);
                focusedModule = newModif;
        } catch (ModuleError ex) {
                cerr << "Module exception: " << ex.description << endl;
        }
}

void Configuration::closeModule(const Path& mod)
{
        for (unsigned int i = 0; i < modmodif.size(); i++) {
                if (modmodif[i]->getModulePath() == mod) {
                        int pn = gtk_notebook_page_num(GTK_NOTEBOOK(moduleBook),
                                                       modmodif[i]->getWidget());                        
                        delete modmodif[i];
                        gtk_notebook_remove_page(GTK_NOTEBOOK(moduleBook), pn);
                        vector<ModuleModifier*>::iterator entry = modmodif.begin();
                        entry += i;
                        modmodif.erase(entry);
                        return;
                }
        }
}

void Configuration::display_preferences()
{
        show_preferences_dialog(window);
}

void Configuration::display_about()
{
        static GtkWidget* dialog = NULL;
        /* a list of authors */
        const char *authors[] = {
                "David Price <mantys@goldweb.com.au>",
                "Michael Lucas-Smith <ich@driftwood.darktech.org>",
                NULL
        };
        /* if the dialog is already created, just show and raise it
           (just in case it is obscured or minimized) */
        if(dialog) {
                gdk_window_show(dialog->window);
                gdk_window_raise(dialog->window);
                return;
        }

        /* create the dialog */
        dialog = gnome_about_new("Root-Portal", VERSION,
                                 "(c) 1999-2000",
                                 authors,
                                 "Display useful and interesting information on the user's desktop.",
                                 NULL);
        /* set the parent of the dialog to be the main application
           window, this will help windowmanagers */
        gnome_dialog_set_parent(GNOME_DIALOG(dialog), GTK_WINDOW(window));

        /* when the dialog gets destroyed, the function gtk_widget_destroyed
           is a utility function from gtk that will NULL the pointer that is
           passed as the data of the signal */
        gtk_signal_connect(GTK_OBJECT(dialog), "destroy",
                           GTK_SIGNAL_FUNC(gtk_widget_destroyed),
                           &dialog);

        gtk_widget_show(dialog);
}

void Configuration::return_message(const DataSet& msg, const Path& path)
{
        if (msg.count() > 0) {
                if (msg.getString(0) == "Error") {
                        if (msg.count() >= 1) {
                                string message = msg.getString(1);
                                gnome_app_error(GNOME_APP(window), message.c_str());
                        }
                }
        }
}

void Configuration::set_appbar_string(const string& msg)
{
        gnome_appbar_set_status(GNOME_APPBAR(appbar), msg.c_str());
}

