/***************************************************************************
 *   Copyright (C) 2006-2009 by Tomasz Moń                                 *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; under version 2 of the License.         *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.         *
 ***************************************************************************/

#include <QtGui>
#include "settings.h"
#include "qtscrob.h"

Settings::Settings(QTScrob *parent) : QDialog( parent ) {
	setupUi(this);
	qtscrob = parent;

	//General Tab
	editUserName->setText(qtscrob->conf.username);
	editPassword->setEchoMode(QLineEdit::Password);
	if ( !qtscrob->conf.password.isEmpty() )
		editPassword->setText("********"); // Indicate that the password has been saved
	chkAutoOpen->setChecked(qtscrob->conf.checkAutoOpen);
	chkDelPlayList->setChecked(qtscrob->conf.checkDelPlayList);

	//Proxy Tab
	edtProxyHost->setText(qtscrob->conf.proxy_host);
	QString port;
	port.setNum(qtscrob->conf.proxy_port);
	edtProxyPort->setText(port);
	edtProxyUser->setText(qtscrob->conf.proxy_user);
	edtProxyPass->setText(qtscrob->conf.proxy_pass);
#ifndef WIN32
	widgetntlm->setEnabled(false);
#else
	connect( chkwinauth, SIGNAL ( stateChanged(int) ), this, SLOT(winauthchanged(int)));
#endif
	chkwinauth->setChecked(qtscrob->conf.proxy_winauth);

	//Timezone tab
	QTextCodec *codec = QTextCodec::codecForLocale();
	QString tzinfo = codec->toUnicode(qtscrob->scrob->get_zonename().c_str());
	tzinfo += "<br>";
	tzinfo += tr("Offset: ");
	tzinfo += QString::fromStdString(qtscrob->scrob->offset_str());
	tzinfo += "<br>";
	tzinfo += tr("Daylight saving: ");
	if (qtscrob->scrob->get_dst())
		tzinfo += tr("Yes");
	else
		tzinfo += tr("No");
	lblTZInfo->setText(tzinfo);
	grpManual->setChecked(qtscrob->conf.tz_override);
	cmb_tz_hour->setCurrentIndex((qtscrob->conf.utc_offset / 3600) + 12);
	cmb_tz_min->setCurrentIndex(abs(qtscrob->conf.utc_offset % 3600) / 900);

	connect( btnApply, SIGNAL ( clicked() ), qtscrob, SLOT(saveSettings()));
	connect( editPassword, SIGNAL ( textEdited(const QString&) ), qtscrob, SLOT(passEdited(const QString&)));

	tabWidget->setCurrentIndex(0);

	setFixedSize(sizeHint());
}

QString Settings::username() {
	return editUserName->text();
}

QString Settings::password() {
	if (editPassword->text()=="")
		return NULL;
	else {
		QString pass = QString::fromStdString(MD5Digest(QString(editPassword->text()).toUtf8()));
		return pass;
	}
}

int Settings::utc_offset() {
	int offset = (cmb_tz_hour->currentIndex()-12) * 3600;
	if (offset < 0)
		offset -= cmb_tz_min->currentIndex() * 900;
	else
		offset += cmb_tz_min->currentIndex() * 900;
	return offset;
}

bool Settings::tz_override() {
	return grpManual->isChecked();
}

bool Settings::checkAutoOpen() {
	return chkAutoOpen->isChecked();
}

bool Settings::checkDelPlayList() {
	return chkDelPlayList->isChecked();
}

QString Settings::proxy_host() {
	return edtProxyHost->text();
}

int Settings::proxy_port() {
	return atoi(edtProxyPort->text().toAscii().data());
}

QString Settings::proxy_user() {
	return edtProxyUser->text();
}

QString Settings::proxy_pass() {
	return edtProxyPass->text();
}

bool Settings::proxy_winauth()
{
	return (chkwinauth->checkState() == Qt::Checked)?true:false;
}

void Settings::winauthchanged(int state)
{
	if (state == Qt::Checked)
		widgetauth->setEnabled(false);
	else
		widgetauth->setEnabled(true);
}
