/*
	Copyright (C) 2003 Frdric Giudicelli (contact_nos@yahoo.com). 
	All rights reserved.

	This product includes cryptographic software written by Eric Young
	(eay@cryptsoft.com)

	This program is released under the GPL with the additional exemption that
	compiling, linking, and/or using OpenSSL is allowed.

	This program is free software; you can redistribute it and/or modify it
	under the terms of the GNU General Public License as published by the Free
	Software Foundation; either version 2 of the License.

	This program is distributed in the hope that it will be useful, but WITHOUT
	ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
	FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
	more details.

	You should have received a copy of the GNU General Public License along with
	this program; if not, write to the Free Software Foundation, Inc., 59 Temple
	Place, Suite 330, Boston, MA 02111-1307 USA
*/

// SOCK_SERVER.h: interface for the SOCK_SERVER class.
//
//////////////////////////////////////////////////////////////////////

#ifndef SOCK_SERVER_H
#define SOCK_SERVER_H

#include <mString.h>
#include <stdio.h>
#include <UnixWinSock.h>
#include <newpki_threads.h>
#include <openssl/bio.h>
#include <time.h>
#include "FileLog.h"
#include "Conf.h"

#include <list>
#include <vector>
using namespace std;



/*!
	This class defines the startup information for the socket server
*/
class ServerConf
{
public:
	/*! \brief This is the constructor.
	 */
	ServerConf();

	/*! \brief This is the constructor.
	 *  \param other [IN] Load datas from another instance.
	 */
	ServerConf(const ServerConf & other);
	
	/*! \brief This is the destructor.
	 */
	~ServerConf();

	/*! \brief This function sets the IP for the server to bind on.
	 *  \param BindAddress [IN] The IP.
	 */
	void set_BindAddress(const mString & BindAddress);

	/*! \brief This function returns the IP for the server to bind on.
	 *  \return The IP.
	 */
	const mString & get_BindAddress() const;

	/*! \brief This function sets the port for the server to listen on.
	 *  \param LocalPort [IN] The port.
	 */
	void set_LocalPort(unsigned int LocalPort);

	/*! \brief This function returns the port for the server to listen on.
	 *  \return The port.
	 */
	unsigned int get_LocalPort() const;

	/*! \brief This function sets the name of the server, for logging purposes.
	 *  \param ServerName [IN] The name.
	 */
	void set_ServerName(const mString & ServerName);

	/*! \brief This function returns the name of the server.
	 *  \return The name.
	 */
	const mString & get_ServerName() const;

	/*! \brief This function loads the info from a global conf.
	 *  \param Conf [IN] The global conf.
	 */
	void Load(const Config & Conf);

	/*! \brief This operator copies one ServerConf into another.
	 *  \param other [IN] The other conf to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const ServerConf &other);

	/*! \brief This operator compares the conf to another.
	 *  \param other [IN] The other conf to compare to.
	 *  \return true when equal, false when different.
	 */
	bool operator==(const ServerConf &other) const;
private:
	mString m_BindAddress;
	unsigned int m_LocalPort;	
	mString m_ServerName;
};



/*!
	This class represents a socket server
*/
class SOCK_SERVER  
{
public:
	/*! \brief This is the constructor.
	 *  \param NumThreadsPool [IN] The number of connection threds to create.
	 */
	SOCK_SERVER(int NumThreadsPool);

	/*! \brief This is the destructor.
	 */
	virtual ~SOCK_SERVER();

	/*! \brief This function starts the Server.
	 *  \param Conf [IN] The server configuration.
	 *  \return true on success, false on failure.
	 */
	bool Start(const ServerConf & Conf);
	
	/*! \brief This function stops the Server.
	 */
	void Stop();

	/*! \brief This function prints the stats of the server.
	 *  \param CurrConnectionsStats [IN] Should print the connections stack info.
	 */
	void PrintStats(bool CurrConnectionsStats);

	/*! \brief This function returns the server's configuration.
	 *  \return The server's configuration.
	 */
	const ServerConf & GetConf();
protected:
	bool ShouldStop();
	ServerConf m_Conf;
	void AddSentBytes(unsigned long bytes);
	void AddRecvBytes(unsigned long bytes);

private:
	bool DoListen();
	typedef struct st_CONNECTION_INFOS
	{
		SOCK_SERVER * me_this;
		SOCKET hSocket;
	}CONNECTION_INFOS;
	

	virtual void OnServerStopped() =0;
	virtual void OnConnection(const char * Ip, SOCKET connection) =0;
	virtual void OnServerStarted() =0;

	static void ThreadProcConnection(const NewpkiThread * Thread, void * Param);
	static void ThreadProcAccept(const NewpkiThread * Thread, void * Param);
	char * GetTimeUnit(unsigned long len_datas, unsigned long len_secs, unsigned long & result_len);

    CriticalSection StatsLock;
    CriticalSection AcceptLock;

	unsigned long TotalConnectionsCtr;
	unsigned long TotalRecvBytes;
	unsigned long TotalSentBytes;

    void KillActiveConnections();
	void AddActiveConnection(SOCKET hSocket);
	void DelActiveConnection(SOCKET hSocket);
	
	CriticalSection ActiveConnectionsLock;
	vector<SOCKET> ActiveConnectionsList;
	CriticalSection WaitingConnectionsLock;
	list<SOCKET> WaitingConnectionsList;

	vector<NewpkiThread*> AllThreads;
	NewpkiThread hThreadAccept;
	int m_NumThreadsPool;

	SOCKET SocketHandle;
	bool Stopped;
	time_t StartTime;
};

#endif
