/*
	Copyright (C) 2003 Frdric Giudicelli (contact_nos@yahoo.com). 
	All rights reserved.

	This product includes cryptographic software written by Eric Young
	(eay@cryptsoft.com)

	This program is released under the GPL with the additional exemption that
	compiling, linking, and/or using OpenSSL is allowed.

	This program is free software; you can redistribute it and/or modify it
	under the terms of the GNU General Public License as published by the Free
	Software Foundation; either version 2 of the License.

	This program is distributed in the hope that it will be useful, but WITHOUT
	ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
	FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
	more details.

	You should have received a copy of the GNU General Public License along with
	this program; if not, write to the Free Software Foundation, Inc., 59 Temple
	Place, Suite 330, Boston, MA 02111-1307 USA
*/


// AsynchMsgs.cpp: implementation of the AsynchMsgs class.
//
//////////////////////////////////////////////////////////////////////

#include "AsynchMsgs.h"
#include <PKI_ERR.h>
#include "svintl.h"

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////


AsynchMsgs::AsynchMsgs(const mString & EntityName, ENGINE * e, unsigned long Type):
			NewPKIStore(EntityName, e)
{
	m_Type = Type;

	if(m_Type)
	{
		hThreadRepositoriesSynchro.Create(ThreadRepositoriesSynchro, this);
		hThreadResponderWorker.Create(ThreadResponderWorker, this);
	}
}

AsynchMsgs::~AsynchMsgs()
{
	if(m_Type)
	{
		hThreadRepositoriesSynchro.Stop();
		hThreadResponderWorker.Stop();
	}
}

bool AsynchMsgs::CreateTables(const SQL_Connection * DbConn)
{
	if( (m_Type & ASYNCHMSGS_TYPE_REQUESTER) )
	{
		if(!CreateRequester(DbConn))
		{
			NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
			return false;
		}
	}

	if( (m_Type & ASYNCHMSGS_TYPE_RESPONDER) )
	{
		if(!CreateResponder(DbConn))
		{
			NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
			return false;
		}
	}

	return true;
}

bool AsynchMsgs::DoUpgrade(const char * Version)
{
	SQL sql(m_DbConn, SQL_ACCESS_WRITE);

	while(strcmp(Version, NEWPKI_VERSION) != 0)
	{
		if(strcmp(Version, "2.0.0-beta4") == 0)
		{
			if((m_Type & ASYNCHMSGS_TYPE_REQUESTER))
			{
				if(!sql.Execute("ALTER TABLE `requester` ADD `type` INT NOT NULL AFTER `status`;"))
				{
					NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
					return false;
				}
				if(!sql.Execute("update `requester` set `type`='-1';"))
				{
					NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
					return false;
				}
			}
			Version = "2.0.0-rc1";
		}
	}
	return true;
}

bool AsynchMsgs::SetRepositories(const mVector<RepEntryInfo> &Repositories)
{
	m_RepositoriesLock.LockWrite();
	m_Repositories = Repositories;
	m_RepositoriesLock.UnlockWrite();
	return true;
}

bool AsynchMsgs::Run()
{
	if(m_Type)
	{
		if(!hThreadRepositoriesSynchro.Start())
		{
			NEWPKIerr(PKI_ERROR_TXT, ERROR_UNKNOWN);
			return false;
		}
		if( (m_Type & ASYNCHMSGS_TYPE_RESPONDER) )
		{
			if(!hThreadResponderWorker.Start())
			{
				NEWPKIerr(PKI_ERROR_TXT, ERROR_UNKNOWN);
				return false;
			}
		}
	}	
	return true;
}

bool AsynchMsgs::CreateRequester(const SQL_Connection * DbConn)
{
	SQL sql(DbConn, SQL_ACCESS_WRITE);
	long i;
	char * CommonCreates[] = {ASYNCHMSGS_CREATE_2, NULL};


	//We execute each request
	for(i=0; CommonCreates[i]; i++)
	{
		if(!sql.Execute(CommonCreates[i]))
		{
			NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
			return false;
		}
	}

	return true;
}

bool AsynchMsgs::CreateResponder(const SQL_Connection * DbConn)
{
	SQL sql(DbConn, SQL_ACCESS_WRITE);
	long i;
	char * CommonCreates[] = {ASYNCHMSGS_CREATE_1, NULL};


	//We execute each request
	for(i=0; CommonCreates[i]; i++)
	{
		if(!sql.Execute(CommonCreates[i]))
		{
			NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
			return false;
		}
	}

	return true;
}


bool AsynchMsgs::InsertRequest(unsigned long priv_attr, const NewpkiRequest & Request, const X509_PUBKEY * Recipient) const
{
	EVP_PKEY * CryptKey;
	CryptedNewpkiRequest newReq;
	
	if(!(m_Type & ASYNCHMSGS_TYPE_REQUESTER))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}

	if(!newReq.set_recipient(Recipient))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(!newReq.set_sender(m_EntityCert.GetX509_PUBKEY()))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	if(!(CryptKey = X509_PUBKEY_get((X509_PUBKEY*)Recipient)))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(!Request.to_SignEncrypt(newReq.get_request(), m_EntityCert.GetPrivateKey().GetRsaKey(), CryptKey, EVP_sha1(), EVP_des_ede3_cbc()))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		EVP_PKEY_free(CryptKey);
		return false;
	}
	EVP_PKEY_free(CryptKey);

	if(!Requester_InsertRequest(newReq, priv_attr, Request.get_type()))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	return true;
}

bool AsynchMsgs::GetPrivAttr(const Asn1OctetString & transactionID, unsigned long & priv_attr)
{
	if( !(m_Type & ASYNCHMSGS_TYPE_REQUESTER) )
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}

	SQL sql(m_DbConn, SQL_ACCESS_READ);
	mString req;
	mString strTransactionId;
	long NumRows;

	if(!transactionIDtoString(transactionID, strTransactionId))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	if(req.sprintf(REQUESTER_ASYNCHMSGS_GET_TID_PRIV_ATTR, strTransactionId.c_str()) <= 0)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}
	if(!sql.Execute(req))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(!sql.NumRows(&NumRows))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	if(!NumRows)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_UNKNOWN);
		return false;
	}

	if(!sql.Value(0, "priv_attr", req))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	priv_attr = req.c_ulng();

	return true;
}

bool AsynchMsgs::GetRequestType(const Asn1OctetString & transactionID, int & type)
{
	if( !(m_Type & ASYNCHMSGS_TYPE_REQUESTER) )
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}

	SQL sql(m_DbConn, SQL_ACCESS_READ);
	mString req;
	mString strTransactionId;
	long NumRows;

	if(!transactionIDtoString(transactionID, strTransactionId))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	if(req.sprintf(REQUESTER_ASYNCHMSGS_GET_TID_TYPE, strTransactionId.c_str()) <= 0)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}
	if(!sql.Execute(req))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(!sql.NumRows(&NumRows))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	if(!NumRows)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_UNKNOWN);
		return false;
	}

	if(!sql.Value(0, "type", req))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	type = req.c_int();

	return true;
}

bool AsynchMsgs::InsertResponse(const Asn1OctetString & transactionID, const NewpkiResponse & Response, const X509_PUBKEY * Recipient) const
{
	CryptedNewpkiResponse s_resp;
	EVP_PKEY * cryptkey;


	if(!(m_Type & ASYNCHMSGS_TYPE_RESPONDER))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}

	if(!s_resp.set_transactionid(transactionID))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(!s_resp.set_sender(m_EntityCert.GetX509_PUBKEY()))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(!s_resp.set_recipient(Recipient))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	cryptkey = X509_PUBKEY_get((X509_PUBKEY*)Recipient);
	if(!cryptkey)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(!Response.to_SignEncrypt(s_resp.get_response(), m_EntityCert.GetPrivateKey().GetRsaKey(), cryptkey, EVP_sha1(), EVP_des_ede3_cbc()))
	{
		EVP_PKEY_free(cryptkey);
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	EVP_PKEY_free(cryptkey);

	// The response will be removed once it has been sent
	if(!Responder_SetResponse(s_resp))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	
	return true;
}

bool AsynchMsgs::OnNewResponse(CryptedNewpkiResponse &response)
{
	TRANSACTION_STATUS TransactionStatus;
	NewpkiResponse newResponse;
	EVP_PKEY * signerkey;

	// We first check that I am the recipient !
	if( ! (m_EntityCert == response.get_recipient()) )
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}

	// Check if we don't already know it
	if(!Requester_GetTransactionStatus(response.get_transactionid(), TransactionStatus))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	// TRANSACTION_STATUS_SENT is the only right status 
	// at this point
	switch(TransactionStatus)
	{
		case TRANSACTION_STATUS_UNKNOWN:
		case TRANSACTION_STATUS_WAITING:
		case TRANSACTION_STATUS_PROCESSED:
			// We request the deletion of this response
			// on the respository
			if(!DeleteResponse(response.get_transactionid()))
				ERR_clear_error();
			return true;
			break;
		default:
			break;
	}


	// Ok we now decrypt the request
	signerkey = X509_PUBKEY_get(response.get_sender());
	if(!signerkey)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	// Decrypt the request
	if(!newResponse.from_SignEncrypt(response.get_response(), signerkey, m_EntityCert.GetPrivateKey().GetRsaKey()))
	{
		EVP_PKEY_free(signerkey);
		ERR_clear_error();
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}
	EVP_PKEY_free(signerkey);
	

	if(!Requester_OnNewResponse(response.get_transactionid(), response.get_sender(), newResponse))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	// We mark this response as having been processed
	if(!Requester_RequestWasResponded(response.get_transactionid()))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	// The response has been successfully been processed
	// we can delete it from the repository
	DeleteResponse(response.get_transactionid());
	ERR_clear_error();

	return true;
}


bool AsynchMsgs::DeleteResponse(const Asn1OctetString & transationID) const
{
	m_waitingDelRespsLock.EnterCS();
	((AsynchMsgs*)this)->m_waitingDelResps.get_transactionids().push_back(transationID);
	m_waitingDelRespsLock.LeaveCS();
	return true;
}

bool AsynchMsgs::OnNewRequest(const CryptedNewpkiRequest & request, CryptedNewpkiResponses & m_waitingResps, mString & SenderName)
{
	TRANSACTION_STATUS TransactionStatus;
	NewpkiRequest newRequest;
	EVP_PKEY * signerkey;
	WorkingRequest w_req;
	CryptedNewpkiResponse Response;

	// We first check that I am the recipient !
	if( ! (m_EntityCert == request.get_recipient()) )
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}

	// Check if we don't already know it
	if(!Responder_GetTransactionStatus(request.get_transactionid(), TransactionStatus))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	switch(TransactionStatus)
	{
		case TRANSACTION_STATUS_PROCESSED:
		case TRANSACTION_STATUS_SENT:
			// Repository should have the response
			// we need to resend it
			if(!Responder_GetResponse(request.get_transactionid(), Response))
			{
				ERR_clear_error();
				return true;
			}
			m_waitingResps.get_responses().push_back(Response);
			return true;
			break;

		case TRANSACTION_STATUS_UNKNOWN:
			break;

		default:
			return true;
			break;
	}

	// Ok we now decrypt the request
	signerkey = X509_PUBKEY_get((X509_PUBKEY*)request.get_sender());
	if(!signerkey)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	// Decrypt the request
	if(!newRequest.from_SignEncrypt(request.get_request(), signerkey, m_EntityCert.GetPrivateKey().GetRsaKey()))
	{
		EVP_PKEY_free(signerkey);
		ERR_clear_error();
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
		ERR_to_STORED_NEWPKI_RESPONSE(request.get_transactionid(), request.get_sender());
		return false;
	}
	EVP_PKEY_free(signerkey);

	// We first ask the responder to validate the request and the sender
	if(!Responder_ValidateRequest(newRequest, request.get_sender(), SenderName))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		ERR_to_STORED_NEWPKI_RESPONSE(request.get_transactionid(), request.get_sender());
		return false;
	}

	// We now create the working request
	// Copy the sender's name
	w_req.set_sendername(SenderName);
	//Copy transaction ID
	if(!w_req.set_transactionid(request.get_transactionid()))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		ERR_to_STORED_NEWPKI_RESPONSE(request.get_transactionid(), request.get_sender());
		return false;
	}

	//Copy the sender public key
	if(!w_req.set_sender(request.get_sender()))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		ERR_to_STORED_NEWPKI_RESPONSE(request.get_transactionid(), request.get_sender());
		return false;
	}
	if(!w_req.set_request(newRequest))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		ERR_to_STORED_NEWPKI_RESPONSE(request.get_transactionid(), request.get_sender());
		return false;
	}
	ERR_clear_error();
	if(!Responder_InsertRequest(w_req))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		ERR_to_STORED_NEWPKI_RESPONSE(request.get_transactionid(), request.get_sender());
		return false;
	}
	return true;
}

void AsynchMsgs::ThreadResponderWorker(const NewpkiThread * Thread, void *param)
{
	AsynchMsgs * me_this = (AsynchMsgs *)param;
	NewpkiResponse Response;
	mVector<WorkingRequest> WorkingReqs;
	size_t i;

	while(!Thread->ShouldStop())
	{
		do
		{
			NewpkiThread::Sleep(1000);
		}
		while(!Thread->ShouldStop() && 
			!me_this->Responder_GetRequests(WorkingReqs));

		//I have a new request
		for(i=0; !Thread->ShouldStop() && i<WorkingReqs.size(); i++)
		{
			ERR_clear_error();
			// In asynch responder we don't get a response, it means the entity will
			// call InsertResponse by itself
			if( (me_this->m_Type & ASYNCHMSGS_RESPONDER_ASYNCH) )
			{
				if(!me_this->Responder_TreatRequestAsynch(WorkingReqs[i].get_request(), WorkingReqs[i].get_transactionid(), WorkingReqs[i].get_sendername()))
				{
					me_this->ERR_to_STORED_NEWPKI_RESPONSE(WorkingReqs[i].get_transactionid(), WorkingReqs[i].get_sender());
				}
				else
				{
					me_this->Responder_RequestOnHold(WorkingReqs[i].get_transactionid());
				}
			}
			else
			{
				if(!me_this->Responder_TreatRequest(WorkingReqs[i].get_request(), WorkingReqs[i].get_sendername(), Response))
				{
					me_this->ERR_to_STORED_NEWPKI_RESPONSE(WorkingReqs[i].get_transactionid(), WorkingReqs[i].get_sender());
				}
				else
				{
					me_this->InsertResponse(WorkingReqs[i].get_transactionid(), Response, WorkingReqs[i].get_sender());
				}
			}
		}
		WorkingReqs.clear();
	}
}

void AsynchMsgs::ERR_to_STORED_NEWPKI_RESPONSE(const Asn1OctetString & transactionID, const X509_PUBKEY * recipient)
{
	NewpkiResponse response;
	if(!response.set_type(NEWPKI_RESPONSE_TYPE_ERR))
	{
		return;
	}
	ERR_to_ERROR_ENTRIES(response.get_errors());
	InsertResponse(transactionID, response, recipient);
}



bool AsynchMsgs::Responder_InsertRequest(const WorkingRequest & Request)
{
	if( !(m_Type & ASYNCHMSGS_TYPE_RESPONDER) )
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}

	SQL sql(m_DbConn, SQL_ACCESS_WRITE);
	mString req;
	mString req_pem;
	mString strTransactionId;
	time_t startTime;

	if(!transactionIDtoString(Request.get_transactionid(), strTransactionId))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	
	if(!Request.to_PEM(req_pem))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(req.sprintf(RESPONDER_ASYNCHMSGS_INSERT_REQ, strTransactionId.c_str(), TRANSACTION_STATUS_WAITING, req_pem.c_str()) <= 0)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}
	if(!sql.Execute(req))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}	

	// Every 5000 requests we request the table to be optimized
	if( (sql.GetLastID() % 5000) == 0 )
	{
		time(&startTime);
		NewpkiDebug(LOG_LEVEL_INFO, m_EntityName.c_str(), _sv("Optimizing responder's requests table..."));
		if(!sql.OptimizeTable(RESPONDER_TABLE))
		{
			req = "";
			ERR_to_mstring(req);
			NewpkiDebug(LOG_LEVEL_WARNING, m_EntityName.c_str(), _sv("Failed to optimize responder's requests table - Reason: %s"), req.c_str());
			ERR_clear_error();
		}
		else
		{
			NewpkiDebug(LOG_LEVEL_INFO, m_EntityName.c_str(), _sv("Optimized responder's requests table in %ld secondes"), time(NULL) - startTime);
		}
	}
	return true;
}

bool AsynchMsgs::Responder_GetRequests(mVector<WorkingRequest> & Requests)
{
	if( !(m_Type & ASYNCHMSGS_TYPE_RESPONDER) )
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}

	long NumRows;
	SQL sql(m_DbConn, SQL_ACCESS_READ);
	mString req;
	long i;
	mString strReq;
	size_t index;

	if(req.sprintf(RESPONDER_ASYNCHMSGS_GET_REQS_BY_STATE, TRANSACTION_STATUS_WAITING) <= 0)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}
	if(!sql.Execute(req))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	
	if(!sql.NumRows(&NumRows))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	index = 0;
	for(i=0; i<NumRows; i++)
	{
		if(!sql.Value(i,"request", strReq))
		{
			NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
			return false;
		}
		Requests.insert(Requests.begin() + index);
		if(!Requests[index].from_PEM(strReq))
			continue;
		index++;
	}

	return true;
}

bool AsynchMsgs::Responder_GetUnsentResponses(mVector<CryptedNewpkiResponse> & Responses, int lindex, int max)
{
	if( !(m_Type & ASYNCHMSGS_TYPE_RESPONDER) )
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}

	long NumRows;
	SQL sql(m_DbConn, SQL_ACCESS_READ);
	mString req;
	long i;
	mString strReq;
	size_t index;
	mString strLimit;

	if(max)
	{
		if(strLimit.sprintf("LIMIT %d,%d", lindex, max) <= 0)
		{
			NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
			return false;
		}
	}

	if(req.sprintf(RESPONDER_ASYNCHMSGS_GET_RESP_BY_STATE, TRANSACTION_STATUS_PROCESSED, strLimit.c_str()) <= 0)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}
	if(!sql.Execute(req))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	
	if(!sql.NumRows(&NumRows))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	index = 0;
	for(i=0; i<NumRows; i++)
	{
		if(!sql.Value(i,"response", strReq))
		{
			NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
			return false;
		}
		Responses.insert(Responses.begin() + index);
		if(!Responses[index].from_PEM(strReq))
			continue;
		index++;
	}

	return true;
}

void AsynchMsgs::ThreadRepositoriesSynchro(const NewpkiThread * Thread, void *param)
{
	AsynchMsgs * me_this = (AsynchMsgs *)param;
	time_t currTime;
	time_t lastAllTime;
	time_t lastFullSend;
	int indexReqs;
	int indexResps;
	int what;
	CryptedNewpkiResponses m_waitingResps;
	mVector<StackRequest> m_waitingReqs;


	do
	{
		NewpkiThread::Sleep(500);
	}
	while(!Thread->ShouldStop() &&
		!me_this->m_Repositories.size());

	if(Thread->ShouldStop())
	{
		return;
	}

	indexReqs = 0;
	indexResps = 0;
	lastFullSend = 0;
	lastAllTime = 0;

	while(!Thread->ShouldStop())
	{
		time(&currTime);

		if(currTime - lastFullSend > 3600)
		{
			// Every hour we retry to send objects
			// that might have failed earlier
			lastFullSend = currTime;
			indexReqs = 0;
			indexResps = 0;
		}
		if(currTime - lastAllTime > ASYNCH_JOBS_INTERVAL)
		{
			what = SYNCH_ALL;
			lastAllTime = currTime;
		}
		else
		{
			what = SYNCH_REQS | SYNCH_RESPS;
		}

		// Get the requests and responses that weren't sent
		if( (me_this->m_Type & ASYNCHMSGS_TYPE_REQUESTER) )
		{
			m_waitingReqs.clear();
			me_this->Requester_GetUnsentRequests(m_waitingReqs, indexReqs, 100);
		}

		if( (me_this->m_Type & ASYNCHMSGS_TYPE_RESPONDER) )
		{
			m_waitingResps.get_responses().clear();
			me_this->Responder_GetUnsentResponses(m_waitingResps.get_responses(), indexResps, 100);
		}

		if(what == SYNCH_ALL || m_waitingReqs.size() ||
			m_waitingResps.get_responses().size() ||
			me_this->m_waitingDelResps.get_transactionids().size() )
		{
			me_this->DoRepositoriesSynchro(Thread, m_waitingReqs, m_waitingResps, SYNCH_ALL);
			if(m_waitingReqs.size())
				indexReqs++;
			if(m_waitingResps.get_responses().size())
				indexResps++;
		}

		// We still wait a bit
		if(!Thread->SleepInterrupt(5))
			break;
	}
}


bool AsynchMsgs::DoRepositoriesSynchro(const NewpkiThread * Thread, 
					mVector<StackRequest> & m_waitingReqs, 
					CryptedNewpkiResponses & m_waitingResps, 
					int what)
{
	EntityConfCrypted newConf;
	CryptedNewpkiRequests reqs;
	CryptedNewpkiResponses resps;
	mString err;
	CryptedNewpkiRequests currReq;
	CryptedNewpkiResponse currResp;
	mString TransactionId;
	mString SenderName;
	size_t i;

	ERR_clear_error();

	if(SynchronizeWithRepositories(Thread, newConf, reqs, resps, m_waitingReqs, m_waitingResps, what))
	{
		if(!Thread->ShouldStop() && newConf)
		{
			NewpkiDebug(LOG_LEVEL_DEBUG, m_EntityName.c_str(), _sv("Importing new conf"));
			if(!OnNewConf(newConf))
			{
				ERR_to_mstring(err);
				m_Logging->LogMessage(LOG_STATUS_TYPE_FAILURE, LOG_MESSAGE_TYPE_ENTITY_GET_MY_CONF, 0, m_EntityCert.GetStringName(), LOG_NO_OBJECTID, "Conf", err.c_str());
				NewpkiDebug(LOG_LEVEL_DEBUG, m_EntityName.c_str(), _sv("Failed to import new conf: %s"), err.c_str());
			}
			else
			{
				NewpkiDebug(LOG_LEVEL_DEBUG, m_EntityName.c_str(), _sv("New conf imported"));
			}
		}

		// Process the new requests
		if(!Thread->ShouldStop() && 
			(m_Type & ASYNCHMSGS_TYPE_RESPONDER) && 
			reqs)
		{
			// For each new request verify that we don't already have it
			// and trigger the event
			for(i=0; !Thread->ShouldStop() && i<reqs.get_requests().size(); i++)
			{
				if(AsynchMsgs::transactionIDtoString(reqs.get_requests()[i].get_transactionid(), TransactionId))
				{
					NewpkiDebug(LOG_LEVEL_DEBUG, m_EntityName.c_str(), _sv("Importing new request %s"), TransactionId.c_str());
					if(!OnNewRequest(reqs.get_requests()[i], m_waitingResps, SenderName))
					{
						ERR_to_mstring(err);
						m_Logging->LogMessage(LOG_STATUS_TYPE_FAILURE, LOG_MESSAGE_TYPE_NEW_REQUEST, 0, SenderName.c_str(), LOG_NO_OBJECTID, TransactionId.c_str(), err.c_str());
						NewpkiDebug(LOG_LEVEL_DEBUG, m_EntityName.c_str(), _sv("Failed to import new request %s : %s"), TransactionId.c_str(), err.c_str());

					}
					else
					{
						m_Logging->LogMessage(LOG_STATUS_TYPE_SUCCESS, LOG_MESSAGE_TYPE_NEW_REQUEST, 0, SenderName.c_str(), LOG_NO_OBJECTID, TransactionId.c_str());
						NewpkiDebug(LOG_LEVEL_DEBUG, m_EntityName.c_str(), _sv("Successfully imported new request %s"), TransactionId.c_str());
					}
				}
			}
			reqs.Clear();
		}
		// Process the new responses
		if(!Thread->ShouldStop() && 
			(m_Type & ASYNCHMSGS_TYPE_REQUESTER) && 
			resps)
		{
			// For each new response verify that we don't already have it
			// and trigger the event
			for(i=0; !Thread->ShouldStop() && i<resps.get_responses().size(); i++)
			{
				if(AsynchMsgs::transactionIDtoString(resps.get_responses()[i].get_transactionid(), TransactionId))
				{
					NewpkiDebug(LOG_LEVEL_DEBUG, m_EntityName.c_str(), _sv("Importing new response %s"), TransactionId.c_str());
					if(!OnNewResponse(resps.get_responses()[i]))
					{
						ERR_to_mstring(err);
						m_Logging->LogMessage(LOG_STATUS_TYPE_FAILURE, LOG_MESSAGE_TYPE_ENTITY_GET_MY_CONF, 0, m_EntityCert.GetStringName(), LOG_NO_OBJECTID, _sv("Response"), err.c_str());
						NewpkiDebug(LOG_LEVEL_DEBUG, m_EntityName.c_str(), _sv("Failed to import new response %s : %s"), TransactionId.c_str(), err.c_str());
					}
					else
					{
						NewpkiDebug(LOG_LEVEL_DEBUG, m_EntityName.c_str(), _sv("Successfully imported new response %s"), TransactionId.c_str());
					}
				}
			}
			resps.Clear();
		}
		return true;
	}
	else
	{
		return false;
	}
}



bool AsynchMsgs::SynchronizeWithRepository(const NewpkiThread * Thread, 
						EntityConfCrypted & myConf, 
						CryptedNewpkiRequests &reqs, 
						CryptedNewpkiResponses &resps, 
						mVector<StackRequest> & m_waitingReqs, 
						CryptedNewpkiResponses & m_waitingResps, 
						const mString & RepName, 
						const mString & RepAddress, 
						unsigned int RepPort, 
						const PKI_CERT & RepCert, 
						PkiClient * ClientPki, 
						mString & Err, 
						int what)
{
	TransactionIds m_transactionIds;
	Asn1OctetString transactionId;
	StackRequest currReq;
	CryptedNewpkiResponse currResp;
	Asn1OctetString currDel;
	size_t i;
	mString strTID;

	NewpkiDebug(LOG_LEVEL_DEBUG, m_EntityName.c_str(), _sv("Connecting to repository %s (%s:%ld)"), RepName.c_str(), RepAddress.c_str(), RepPort);
	//We connect to the repository
	if(!AsynchJobs::ConnectToRepository(m_EntityCert, 
										RepName, 
										RepAddress, 
										RepPort, 
										RepCert, ClientPki, 30, Err))
	{
		NewpkiDebug(LOG_LEVEL_DEBUG, m_EntityName.c_str(), _sv("Failed to connect to repository %s (%s:%ld)"), RepName.c_str(), RepAddress.c_str(), RepPort);
		return false;
	}
	NewpkiDebug(LOG_LEVEL_DEBUG, m_EntityName.c_str(), _sv("Connected to repository %s (%s:%ld)"), RepName.c_str(), RepAddress.c_str(), RepPort);

	if(Thread->ShouldStop())
	{
		return true;
	}
	
	//Send each request
	if((m_Type & ASYNCHMSGS_TYPE_REQUESTER) && (what & SYNCH_REQS))
	{
		while(!Thread->ShouldStop() && m_waitingReqs.size())
		{
			if(!ClientPki->SendRequest(m_waitingReqs[0].get_req()))
			{
				NewpkiDebug(LOG_LEVEL_DEBUG, m_EntityName.c_str(), _sv("Failed to sent request %ld to repository %s (%s:%ld)"), m_waitingReqs[0].get_id(), RepName.c_str(), RepAddress.c_str(), RepPort);
				Err = ClientPki->GetError();
				return false;
			}
			Requester_RequestSent(m_waitingReqs[0].get_id());
			//Remove entry
			m_waitingReqs.erase(m_waitingReqs.begin());
			NewpkiDebug(LOG_LEVEL_DEBUG, m_EntityName.c_str(), _sv("Successfully sent request %ld to repository %s (%s:%ld), still %d to go"), m_waitingReqs[0].get_id(), RepName.c_str(), RepAddress.c_str(), RepPort, m_waitingReqs.size());
		}
		
		if(Thread->ShouldStop())
		{
			return true;
		}


		// Send each deletion request
		// A deletion request is sent when a requester has successfully
		// processed the response, and therefore declares is at not
		// needed anymore
		m_waitingDelRespsLock.EnterCS();
		for(i=0; !Thread->ShouldStop() && m_waitingDelResps.get_transactionids().size(); i++)
		{
			currDel = m_waitingDelResps.get_transactionids()[0];
			m_waitingDelRespsLock.LeaveCS();

			if(!ClientPki->DeleteResponse(currDel))
			{
				NewpkiDebug(LOG_LEVEL_DEBUG, m_EntityName.c_str(), _sv("Failed to sent deletion request %ld to repository %s (%s:%ld)"), i, RepName.c_str(), RepAddress.c_str(), RepPort);
				Err = ClientPki->GetError();
				return false;
			}
			NewpkiDebug(LOG_LEVEL_DEBUG, m_EntityName.c_str(), _sv("Successfully sent deletion request %ld to repository %s (%s:%ld), still %d to go"), i, RepName.c_str(), RepAddress.c_str(), RepPort, m_waitingDelResps.get_transactionids().size());
			m_waitingDelRespsLock.EnterCS();
			m_waitingDelResps.get_transactionids().erase(m_waitingDelResps.get_transactionids().begin());
		}
		m_waitingDelRespsLock.LeaveCS();

		if(Thread->ShouldStop())
		{
			return true;
		}


		// Get new responses that were sent to me
		// Get the list of transactions for which we are waiting for an answer
		if(Requester_GetWaitingTIDs(m_transactionIds) && 
			m_transactionIds.get_transactionids().size())
		{
			// Get the responses (if they're available)
			if(Thread->ShouldStop())
			{
				return true;
			}
			if(!ClientPki->GetMyResponses(m_transactionIds, resps))
			{
				NewpkiDebug(LOG_LEVEL_DEBUG, m_EntityName.c_str(), _sv("Failed to get my responses from repository %s (%s:%ld)"), RepName.c_str(), RepAddress.c_str(), RepPort);
				Err = ClientPki->GetError();
				return false;
			}
			NewpkiDebug(LOG_LEVEL_DEBUG, m_EntityName.c_str(), _sv("Successfully got my responses from repository %s (%s:%ld)"), RepName.c_str(), RepAddress.c_str(), RepPort);
		}
		if(Thread->ShouldStop())
		{
			return true;
		}		
	}

	

	if((m_Type & ASYNCHMSGS_TYPE_RESPONDER) && (what & SYNCH_RESPS))
	{
		//Send each new response
		while(!Thread->ShouldStop() && m_waitingResps.get_responses().size())
		{
			if(!AsynchMsgs::transactionIDtoString(m_waitingResps.get_responses()[0].get_transactionid(), strTID))
			{
				strTID = "Unknown";
			}
			if(!ClientPki->SendResponse(m_waitingResps.get_responses()[0]))
			{
				NewpkiDebug(LOG_LEVEL_DEBUG, m_EntityName.c_str(), _sv("Failed to sent response %s to repository %s (%s:%ld)"), strTID.c_str(), RepName.c_str(), RepAddress.c_str(), RepPort);
				Err = ClientPki->GetError();
				return false;
			}
			Responder_ResponseSent(m_waitingResps.get_responses()[0].get_transactionid());
			//Remove entry
			m_waitingResps.get_responses().erase(m_waitingResps.get_responses().begin());
			NewpkiDebug(LOG_LEVEL_DEBUG, m_EntityName.c_str(), _sv("Successfully sent response %s to repository %s (%s:%ld), still %d to go"), strTID.c_str(), RepName.c_str(), RepAddress.c_str(), RepPort, m_waitingResps.get_responses().size());
		}

		if(Thread->ShouldStop())
		{
			return true;
		}

		// Get the new requests that were sent to me
		if(Responder_GetKnownTIDs(m_transactionIds))
		{
			if(Thread->ShouldStop())
			{
				return true;
			}
			if(!ClientPki->GetMyRequests(m_transactionIds, reqs))
			{
				NewpkiDebug(LOG_LEVEL_DEBUG, m_EntityName.c_str(), _sv("Failed to get my requests from repository %s (%s:%ld)"), RepName.c_str(), RepAddress.c_str(), RepPort);
				Err = ClientPki->GetError();
				return false;
			}
			NewpkiDebug(LOG_LEVEL_DEBUG, m_EntityName.c_str(), _sv("Successfully got my requests from repository %s (%s:%ld)"), RepName.c_str(), RepAddress.c_str(), RepPort);
		}
		if(Thread->ShouldStop())
		{
			return false;
		}
	}
	

	if(what & SYNCH_CONF)
	{
		// Get entity conf
		if(!ClientPki->GetMyConf(myConf))
		{			
			Err = ClientPki->GetError();
			return false;
		}
	}
	return true;
}

bool AsynchMsgs::SynchronizeWithRepositories(const NewpkiThread * Thread, 
						EntityConfCrypted & myConf, 
						CryptedNewpkiRequests &reqs, 
						CryptedNewpkiResponses &resps, 
						mVector<StackRequest> & m_waitingReqs, 
						CryptedNewpkiResponses & m_waitingResps, 
						int what)
{
	size_t i;
	mString RepName;
	mString RepAddress;
	unsigned int RepPort;
	PKI_CERT RepCert;
	mString Err;
	PkiClient ClientPki(NULL);

	resps.Clear();
	myConf.Clear();
	reqs.Clear();

	m_RepositoriesLock.LockRead();
	for(i=0; i<m_Repositories.size(); i++)
	{
		if(Thread->ShouldStop())
		{
			m_RepositoriesLock.UnlockRead();
			return true;
		}
		//If the destination is me we ignore it
		if(m_EntityCert == m_Repositories[i].get_repositoryssl())
			continue;

		// We copy the datas, so we can unlock
		RepName = m_Repositories[i].get_name();
		RepAddress = m_Repositories[i].get_address();
		RepPort = m_Repositories[i].get_port();
		RepCert = m_Repositories[i].get_repositoryssl();

		m_RepositoriesLock.UnlockRead();

		ERR_clear_error();

		if(SynchronizeWithRepository(Thread, myConf, reqs, resps, m_waitingReqs, m_waitingResps, RepName, RepAddress, RepPort, RepCert, &ClientPki, Err, what))
		{
			NewpkiDebug(LOG_LEVEL_INFO, m_EntityName.c_str(), _sv("Successfully got my conf and my objects from repository %s (%s:%ld)"), RepName.c_str(), RepAddress.c_str(), RepPort);
			ClientPki.CloseConnection();
			return true;
		}
		NewpkiDebug(LOG_LEVEL_WARNING, m_EntityName.c_str(), _sv("Failed to get my conf and my objects from repository %s (%s:%ld)\nReason:%s"), RepName.c_str(), RepAddress.c_str(), RepPort, Err.c_str());
		m_Logging->LogMessage(LOG_STATUS_TYPE_FAILURE, LOG_MESSAGE_TYPE_ENTITY_GET_MY_CONF, 0, NULL, LOG_NO_OBJECTID, RepName.c_str(), Err.c_str());
		ClientPki.CloseConnection();

		m_RepositoriesLock.LockRead();
	}
	m_RepositoriesLock.UnlockRead();
	return false;
}











bool AsynchMsgs::Responder_SetResponse(const CryptedNewpkiResponse & Response) const
{
	if( !(m_Type & ASYNCHMSGS_TYPE_RESPONDER) )
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}

	SQL sql(m_DbConn, SQL_ACCESS_WRITE);
	mString req;
	mString strTransactionId;
	mString resp_pem;

	if(!transactionIDtoString(Response.get_transactionid(), strTransactionId))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	if(!Response.to_PEM(resp_pem))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	if(req.sprintf(RESPONDER_ASYNCHMSGS_SET_RESPONSE, TRANSACTION_STATUS_PROCESSED, resp_pem.c_str(), strTransactionId.c_str()) <= 0)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}
	if(!sql.Execute(req))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	return true;
}

bool AsynchMsgs::Responder_GetResponse(const Asn1OctetString & TransactionID, CryptedNewpkiResponse & Response)
{
	if( !(m_Type & ASYNCHMSGS_TYPE_RESPONDER) )
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}

	SQL sql(m_DbConn, SQL_ACCESS_READ);
	mString req;
	mString strTransactionId;
	long NumRows;

	if(!transactionIDtoString(TransactionID, strTransactionId))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	if(req.sprintf(RESPONDER_ASYNCHMSGS_GET_RESPONSE, strTransactionId.c_str()) <= 0)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}
	if(!sql.Execute(req))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(!sql.NumRows(&NumRows))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(!NumRows)
	{
		return false;
	}
	if(!sql.Value(0, "response", req))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(!Response.from_PEM(req))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	return true;
}

bool AsynchMsgs::Responder_ResponseSent(const Asn1OctetString & TransactionID)
{
	if(!Responder_SetStatus(TransactionID, TRANSACTION_STATUS_SENT))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	return true;
}

bool AsynchMsgs::Responder_RequestOnHold(const Asn1OctetString & TransactionID)
{
	SQL sql(m_DbConn, SQL_ACCESS_WRITE);
	mString req;
	mString strTransactionId;
	long NumRows;

	
	if( !(m_Type & ASYNCHMSGS_TYPE_RESPONDER) )
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}

	// I cannot put a request on hold if the response has already been sent


	if(!transactionIDtoString(TransactionID, strTransactionId))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	if(req.sprintf(RESPONDER_ASYNCHMSGS_GET_STATUS, strTransactionId.c_str()) <= 0)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}
	if(!sql.Execute(req))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(!sql.NumRows(&NumRows))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(!NumRows)
	{
		return false;
	}
	if(!sql.Value(0, "status", req))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(req.c_int() == TRANSACTION_STATUS_PROCESSED)
		return true;

	if(!Responder_SetStatus(TransactionID, TRANSACTION_STATUS_HOLD))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	return true;
}

bool AsynchMsgs::Responder_SetStatus(const Asn1OctetString & TransactionID, TRANSACTION_STATUS Status)
{
	if( !(m_Type & ASYNCHMSGS_TYPE_RESPONDER) )
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}

	SQL sql(m_DbConn, SQL_ACCESS_WRITE);
	mString req;
	mString strTransactionId;

	if(!transactionIDtoString(TransactionID, strTransactionId))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	if(req.sprintf(RESPONDER_ASYNCHMSGS_SET_STATUS, Status, strTransactionId.c_str()) <= 0)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}
	if(!sql.Execute(req))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	return true;
}

bool AsynchMsgs::Responder_GetTransactionStatus(const Asn1OctetString & TransactionID, TRANSACTION_STATUS & Status)
{
	if( !(m_Type & ASYNCHMSGS_TYPE_RESPONDER) )
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}

	SQL sql(m_DbConn, SQL_ACCESS_READ);
	mString req;
	mString strTransactionId;
	long NumRows;

	if(!transactionIDtoString(TransactionID, strTransactionId))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	if(req.sprintf(RESPONDER_ASYNCHMSGS_GET_TID_STATUS, strTransactionId.c_str()) <= 0)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}
	if(!sql.Execute(req))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(!sql.NumRows(&NumRows))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	if(!NumRows)
	{
		Status = TRANSACTION_STATUS_UNKNOWN;
		return true;
	}

	if(!sql.Value(0, "status", req))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	Status = (TRANSACTION_STATUS)req.c_int();

	return true;
}


bool AsynchMsgs::Responder_GetKnownTIDs(TransactionIds &transactionIDs)
{
	if( !(m_Type & ASYNCHMSGS_TYPE_RESPONDER) )
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}

	long NumRows;
	SQL sql(m_DbConn, SQL_ACCESS_READ);
	mString req;
	long i;
	mString strTransactionID;
	Asn1OctetString transactionID;

	// When a responder transaction is marked as TRANSACTION_STATUS_SENT
	// It means that the repository got the response, and that it
	// marked the associated request as deleted, so even if
	// we know it, there is no need to add it to list since for
	// the repository the request doesn't exist anymore.
	if(req.sprintf(RESPONDER_ASYNCHMSGS_GET_TIDS, TRANSACTION_STATUS_SENT) <= 0)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}
	if(!sql.Execute(req))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	
	if(!sql.NumRows(&NumRows))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	for(i=0; i<NumRows; i++)
	{
		if(!sql.Value(i,"transactionID", strTransactionID))
		{
			NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
			return false;
		}

		if(strTransactionID.size())
		{
			if(!StringtoTransactionID(strTransactionID, transactionID))
				continue;
			transactionIDs.get_transactionids().push_back(transactionID);
		}
	}

	return true;
}

bool AsynchMsgs::Requester_InsertRequest(CryptedNewpkiRequest & Request, unsigned long priv_attr, int type) const
{
	if( !(m_Type & ASYNCHMSGS_TYPE_REQUESTER) )
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}

	SQL sql(m_DbConn, SQL_ACCESS_WRITE);
	mString req;
	mString req_pem;
	mString TID;
	unsigned long req_id;
	time_t startTime;
	
	if(req.sprintf(REQUESTER_ASYNCHMSGS_INSERT_REQ, TRANSACTION_STATUS_WAITING, priv_attr, type) <= 0)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}
	if(!sql.Execute(req))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	
	req_id = sql.GetLastID();

	if(!GenerateTransactionID(Request.get_transactionid(), Request.get_sender(), Request.get_recipient(), req_id))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		Requester_DeleteTransaction(req_id);
		return false;
	}
	
	if(!transactionIDtoString(Request.get_transactionid(), TID))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		Requester_DeleteTransaction(req_id);
		return false;
	}
	
	if(!Request.to_PEM(req_pem))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		Requester_DeleteTransaction(req_id);
		return false;
	}

	if(req.sprintf(REQUESTER_ASYNCHMSGS_SET_REQ, TID.c_str(), req_pem.c_str(), req_id) <= 0)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}
	if(!sql.Execute(req))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		Requester_DeleteTransaction(req_id);
		return false;
	}
	
	// Every 5000 requests we request the table to be optimized
	if( (req_id % 5000) == 0 )
	{
		time(&startTime);
		NewpkiDebug(LOG_LEVEL_INFO, m_EntityName.c_str(), _sv("Optimizing requester's requests table..."));
		if(!sql.OptimizeTable(REQUESTER_TABLE))
		{
			req = "";
			ERR_to_mstring(req);
			NewpkiDebug(LOG_LEVEL_WARNING, m_EntityName.c_str(), _sv("Failed to optimize requester's requests table - Reason: %s"), req.c_str());
			ERR_clear_error();
		}
		else
		{
			NewpkiDebug(LOG_LEVEL_INFO, m_EntityName.c_str(), _sv("Optimized requester's requests table in %ld secondes"), time(NULL) - startTime);
		}
	}

	return true;
}

bool AsynchMsgs::Requester_RequestSent(unsigned long req_id)
{
	if( !(m_Type & ASYNCHMSGS_TYPE_REQUESTER) )
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}

	SQL sql(m_DbConn, SQL_ACCESS_WRITE);
	mString req;

	if(req.sprintf(REQUESTER_ASYNCHMSGS_SET_SENT, TRANSACTION_STATUS_SENT, req_id) <= 0)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}
	if(!sql.Execute(req))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	return true;
}

bool AsynchMsgs::Requester_GetWaitingTIDs(TransactionIds &transactionIDs)
{
	if( !(m_Type & ASYNCHMSGS_TYPE_REQUESTER) )
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}

	long NumRows;
	SQL sql(m_DbConn, SQL_ACCESS_READ);
	mString req;
	long i;
	mString strTransactionID;
	Asn1OctetString transactionID;

	if(req.sprintf(REQUESTER_ASYNCHMSGS_GET_TIDS_BY_STATE, TRANSACTION_STATUS_SENT) <= 0)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}
	if(!sql.Execute(req))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	
	if(!sql.NumRows(&NumRows))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	for(i=0; i<NumRows; i++)
	{
		if(!sql.Value(i,"transactionID", strTransactionID))
		{
			NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
			return false;
		}

		if(strTransactionID.size())
		{
			if(!StringtoTransactionID(strTransactionID, transactionID))
				continue;
			transactionIDs.get_transactionids().push_back(transactionID);
		}
	}

	return true;
}

bool AsynchMsgs::Requester_DeleteTransaction(unsigned long req_id) const
{
	if( !(m_Type & ASYNCHMSGS_TYPE_REQUESTER) )
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}

	SQL sql(m_DbConn, SQL_ACCESS_WRITE);
	mString req;
	
	if(req.sprintf(REQUESTER_ASYNCHMSGS_DELETE_REQ, req_id) <= 0)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}
	if(!sql.Execute(req))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	return true;
}

bool AsynchMsgs::Requester_DeleteTransactionPrivAttr(unsigned long priv_attr)
{
	if( !(m_Type & ASYNCHMSGS_TYPE_REQUESTER) )
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}

	SQL sql(m_DbConn, SQL_ACCESS_WRITE);
	mString req;
	
	if(req.sprintf(REQUESTER_ASYNCHMSGS_DELETE_REQ_PRIV_ATTR, priv_attr) <= 0)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}
	if(!sql.Execute(req))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	return true;
}

bool AsynchMsgs::Requester_GetUnsentRequests(mVector<StackRequest> & Requests, int lindex, int max)
{
	if( !(m_Type & ASYNCHMSGS_TYPE_REQUESTER) )
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}

	long NumRows;
	SQL sql(m_DbConn, SQL_ACCESS_READ);
	mString req;
	long i;
	mString strReq;
	size_t index;
	mString strLimit;

	if(max)
	{
		if(strLimit.sprintf("LIMIT %d,%d", lindex, max) <= 0)
		{
			NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
			return false;
		}
	}

	if(req.sprintf(REQUESTER_ASYNCHMSGS_GET_REQS_BY_STATE, TRANSACTION_STATUS_WAITING, strLimit.c_str()) <= 0)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}
	if(!sql.Execute(req))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	
	if(!sql.NumRows(&NumRows))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	index = 0;
	for(i=0; i<NumRows; i++)
	{
		Requests.insert(Requests.begin() + index);
		// Get the ID
		if(!sql.Value(i,"req_id", strReq))
		{
			continue;
		}
		Requests[index].set_id(strReq.c_ulng());

		// Get the request
		if(!sql.Value(i,"request", strReq))
		{
			continue;
		}
		if(!Requests[index].get_req().from_PEM(strReq))
		{
			continue;
		}
		index++;
	}

	return true;
}


bool AsynchMsgs::Requester_GetTransactionStatus(const Asn1OctetString & TransactionID, TRANSACTION_STATUS & Status)
{
	if( !(m_Type & ASYNCHMSGS_TYPE_REQUESTER) )
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}

	SQL sql(m_DbConn, SQL_ACCESS_READ);
	mString req;
	mString strTransactionId;
	long NumRows;

	if(!transactionIDtoString(TransactionID, strTransactionId))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	if(req.sprintf(REQUESTER_ASYNCHMSGS_GET_TID_STATUS, strTransactionId.c_str()) <= 0)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}
	if(!sql.Execute(req))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(!sql.NumRows(&NumRows))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	if(!NumRows)
	{
		Status = TRANSACTION_STATUS_UNKNOWN;
		return true;
	}

	if(!sql.Value(0, "status", req))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	Status = (TRANSACTION_STATUS)req.c_int();

	return true;
}


bool AsynchMsgs::Requester_RequestWasResponded(const Asn1OctetString & TransactionID)
{
	if( !(m_Type & ASYNCHMSGS_TYPE_REQUESTER) )
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}

	SQL sql(m_DbConn, SQL_ACCESS_WRITE);
	mString req;
	mString strTransactionId;

	if(!transactionIDtoString(TransactionID, strTransactionId))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(req.sprintf(REQUESTER_ASYNCHMSGS_SET_REQ_STATE, TRANSACTION_STATUS_PROCESSED, strTransactionId.c_str()) <= 0)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}
	if(!sql.Execute(req))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	return true;
}

