/********************************************************************
 *                                                                  *
 * File: preferences.c                                              *
 *                                                                  *
 ********************************************************************
 *                                                                  *
 * Authors:                                                         *
 *   Chris Osgod          <oznet@mac.com>                           *
 *   Eirik A. Herskedal   <ehersked@cs.purdue.edu>                  *
 *                                                                  *
 ********************************************************************
 *                                                                  *
 * Parses the configuration file.                                   *
 * This file is a modification of a parser originally written       *
 * by Chris Osgod (oznet@mac.com).                                  *
 *                                                                  *
 ********************************************************************/

#include <stdio.h>
#include <syslog.h>
#include <string.h>

#include "preferences.h"

struct preferences pr = {
  DEBUGOFF,
  0,
  1,
  "/home/",
  "user.cert",
  "/etc/musclepam/root.cert",
  "",
  "",
  ROOTCERT
};

/******************************************************************************
** Function: util_ParsePreference
**
** Parse a preference item from a line of text.  The text should be null
** terminated and buf_size keeps overruns from happening.
**
** If a preference item is successfully parsed then it is stored in the 
** st.prefs settings.
**
** This whole function is fairly verbose and could be broken into smaller
** pieces to handle things like "get true/false pref" but at least this is
** very straightforward.
**
** Fixme: put this in p11x_prefs.c
**
** Parameters:
**  buf      - Null terminated text buffer
**  buf_size - Size of buffer 
**
** Returns:
**  none
******************************************************************************/
void util_ParsePreference(char *buf, int buf_size)
{
  char sep[] = "=\t\r\n ";
  char *token;
  
  /* We will be using many unsafe string functions so force a NULL at the */
  /* end of the buffer to protect ourselves                               */
  buf[buf_size - 1] = 0;
  token = (char *) strchr(buf, '#');
  
  if (token) 
    *token = 0;
  
  token = (char *) strtok(buf, sep);
  if (token)
    {
      if (!strcasecmp("Debug", token))
        {
	  token = (char *) strtok(0, sep);
	  if (!token)
	    syslog(LOG_ERR, "Config option \"Debug\" failed");
	  else
            {
	      if (!strcasecmp("OFF", token))
		pr.debug = DEBUGOFF;
	      else
		pr.debug = DEBUGON;
            }
        }
      else if (!strcasecmp("CertNumber", token))
        {
	  token = (char *) strtok(0, sep);
	  if (!token)
	    syslog(LOG_ERR, "Config option \"CertNumber\" failed");
	  else
            {
	      pr.certnumber = (int) (*token - '0');
            }
        }
      else if (!strcasecmp("PinNumber", token))
        {
	  token = (char *) strtok(0, sep);
	  if (!token)
	    syslog(LOG_ERR, "Config option \"PinNumber\" failed");
	  else
            {
	      pr.pinnumber = (int) (*token - '0');
            }
        }
      else if (!strcasecmp("CertName", token))
        {
	  token = (char *) strtok(0, sep);
	  if (!token)
	    syslog(LOG_ERR, "Config option \"CertName\" failed");
	  else
            {
	      strncpy(pr.certname, token, 200);
            }
        }
      else if (!strcasecmp("UserPath", token))
        {
	  token = (char *) strtok(0, sep);
	  if (!token)
	    syslog(LOG_ERR, "Config option \"UserPath\" failed");
	  else
            {
	      strncpy(pr.userpath, token, 200);
            }
        }
      else if (!strcasecmp("RootCACert", token))
        {
	  token = (char *) strtok(0, sep);
	  if (!token)
	    syslog(LOG_ERR, "Config option \"RootCACert\" failed");
	  else
            {
	      strncpy(pr.rootcacert, token, 200);
            }
        }
      else if (!strcasecmp("LDAPHost", token))
        {
	  token = (char *) strtok(0, sep);
	  if (!token)
	    syslog(LOG_ERR, "Config option \"LDAPHost\" failed");
	  else
            {
	      strncpy(pr.ldaphost, token, 200);
            }
        }
      else if (!strcasecmp("LDAPPath", token))
        {
	  token = (char *) strtok(0, sep);
	  if (!token)
	    syslog(LOG_ERR, "Config option \"LDAPPath\" failed");
	  else
            {
	      strncpy(pr.ldappath, token, 200);
            }
        }
      else if (!strcasecmp("AuthMode", token))
        {
	  token = (char *) strtok(0, sep);
	  if (!token)
	    syslog(LOG_ERR, "Config option \"AuthMode\" failed");
	  else
            {
	      if (!strcasecmp("UserCert", token))
		pr.authmode = USERCERT;
	      else if (!strcasecmp("RootCert", token))
		pr.authmode = ROOTCERT;
            }
        }
    }
}

/******************************************************************************
** Function: util_ReadPreferences
**
** Gets preferences, if available.  On UNIX, looks for .pkcs11rc
** in the $HOME directory, or root directory if $HOME is not 
** defined.  Having a preferences file is optional and it is assumed
** that most of the time users will not have one unless debug/logging
** or other special settings are required.
**
** Parameters:
**  none
**
** Returns:
**  none
******************************************************************************/

int util_ReadPreferences()
{
  int rv = 0;
  FILE *fp;
  char rcfilename[] = "/etc/musclepam/pam-muscle.conf";
  char buf[1024];
  

  fp = fopen(rcfilename, "rb");
  if (fp)
    {
      while (fgets(buf, sizeof(buf), fp))
	util_ParsePreference(buf, sizeof(buf));
      
      fclose(fp);
    }
  return rv;
}

