/*
 * $Id: parse.c,v 1.10 2001/01/27 22:23:07 linus Exp $
 * Copyright (C) 1991  Lysator Academic Computer Association.
 *
 * This file is part of the LysKOM server.
 * 
 * LysKOM is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by 
 * the Free Software Foundation; either version 1, or (at your option) 
 * any later version.
 * 
 * LysKOM is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with LysKOM; see the file COPYING.  If not, write to
 * Lysator, c/o ISY, Linkoping University, S-581 83 Linkoping, SWEDEN,
 * or the Free Software Foundation, Inc., 675 Mass Ave, Cambridge, 
 * MA 02139, USA.
 *
 * Please mail bug reports to bug-lyskom@lysator.liu.se. 
 */
/*
 * parse.c  -  receive objects from an isc connection.
 *
 * Written by ceder 1990-07-13
 */

/*
 * KNOWN BUGS: Does not read in the array when returning KOM_SERVER_NO_MEM.
 *	       Thus the client will be out of sync after that error.
 * SOLUTION:   Use skip_token from parse-async.c (not yet written).
 */
/* Error checking is poor. */

#include "../../config.h"

#include <stdio.h>
#include <ctype.h>
#include <sys/types.h>
#include <sys/file.h>
#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#endif
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif
/* #include <time.h> included from kom-types.h */

#include <kom-types.h>
#include <kom-errno.h>
#include <services.h>

#include "client-mall.h"
#include "parse.h"
#include "async.h"


#define REALLOC(ptr, size)	(((ptr) == NULL)		\
				 ? isc_malloc (size)		\
				 : isc_realloc ((ptr), (size)))


#define READ_ONE_FLAG(flagname)			\
    switch (my_getc(fp))			\
    {						\
    case '1':					\
	result->flagname = 1;			\
	break;					\
    case '0':					\
	result->flagname = 0;			\
	break;					\
    default:					\
	/* The bitarray ends abruptly here. */	\
	kom_errno = KOM_SERVER_IS_CRAZY;	\
	return FAILURE;				\
    }



void
skipwhite(FILE *fp)
{
    int c;

    while ( isspace(c = my_getc(fp)) )
#ifdef DEBUGSKIPPING
	fprintf(stderr,"SKIPPING: %c (%d)\n",c,c);
#endif
	;

    my_ungetc(c, fp);
}


unsigned long
parse_long(FILE *fp)
{
    unsigned long res=0;
    short donesome = 0;
    int c;

    skipwhite(fp);
    while ( isdigit(c = my_getc(fp)) ) {
	res = res * 10 + (c - '0');
	donesome = 1;
    }
    my_ungetc(c,fp);
    if (donesome) {
#ifdef DEBUGSKIPPING
	fprintf(stderr,"READ: %u\n",res);
#endif
	return res;
    }
    else {
#ifdef DEBUG
	fprintf(stderr,"parse_long did not work. c = %c (%d)\n",c,c);
#endif
	return 0;
    }
}


Success
parse_person (FILE *fp,
	      Person *person)
{
    if (parse_string(fp, &person->username) != OK
	|| parse_priv_bits (fp, &person->privileges) != OK
	|| parse_personal_flags (fp, &person->flags) != OK 
	|| parse_time(fp, &person -> last_login) != OK)
    {
	return FAILURE;
    }

    person -> user_area		= parse_long(fp);
    person -> total_time_present = parse_long(fp);
    person -> sessions		= parse_long(fp);
    person -> created_lines	= parse_long(fp);
    person -> created_bytes	= parse_long(fp);
    person -> read_texts	= parse_long(fp);
    person -> no_of_text_fetches = parse_long(fp);
    person -> created_persons	= parse_long(fp);
    person -> created_confs	= parse_long(fp);
    person -> first_created_text = parse_long (fp);
    person -> no_of_created_texts = parse_long (fp);
    person -> no_of_marks	= parse_long (fp);
    person -> no_of_confs	= parse_long (fp);

    return OK;
}

Success
parse_membership(FILE *fp,
		 Membership *mship)
{
    unsigned int i;

    if (parse_time(fp, &mship->last_time_read) != OK)
    {
	return FAILURE;
    }
    
    mship->conf_no = parse_long(fp);
    mship->priority = parse_long(fp);
    mship->last_text_read = parse_long(fp);
    mship->no_of_read = parse_long(fp);

    if ( mship->no_of_read > 0)
    {
	skipwhite(fp);
	switch(my_getc(fp))
	{
	case '{':
	    mship->read_texts = REALLOC(mship->read_texts,
					(mship->no_of_read
					 * sizeof(Local_text_no)));
	    
	    if ( mship->read_texts == NULL )
	    {
		kom_errno = KOM_OUT_OF_MEMORY;
		return FAILURE;
	    }

	    for ( i = 0; i < mship->no_of_read; i++)
		mship->read_texts[ i ] = parse_long(fp);

	    skipwhite(fp);
	    if ( my_getc(fp) != '}' )
	    {
		kom_errno = KOM_SERVER_IS_CRAZY;
		return FAILURE;
	    }
	    break;
	    
	case '*':
	    isc_free(mship->read_texts);
	    mship->read_texts = NULL;
	    break;
	    
	default:
	    kom_errno = KOM_SERVER_IS_CRAZY;
	    return FAILURE;
	}
    }
    else
    {
	skipwhite(fp);
	if ( my_getc(fp) != '*' ) 
	{
	    kom_errno = KOM_SERVER_IS_CRAZY;
	    return FAILURE;
	}

	isc_free(mship->read_texts);
	mship->read_texts = NULL;
    }

    return OK;
}

	
extern Success
parse_membership_list(FILE *fp,
		      Membership_list *result)
{
    unsigned short i;

    /* First free all the read_texts. */

    if ( result->confs != NULL )
    {
	for ( i = 0; i < result->no_of_confs; i++)
	    isc_free(result->confs[ i ].read_texts);
    }
    
    result->no_of_confs = parse_long(fp);

    if ( result->no_of_confs > 0 )
    {
	skipwhite(fp);
	switch(my_getc(fp))
	{
	case '{':
	    result->confs = REALLOC(result->confs,
				    (result->no_of_confs
				     * sizeof(Membership)));
	    if ( result->confs == NULL )
	    {
		kom_errno = KOM_OUT_OF_MEMORY;
		return FAILURE;
	    }

	    for ( i = 0; i < result->no_of_confs; i++)
	    {
		result->confs[ i ] = EMPTY_MEMBERSHIP;
		if ( parse_membership(fp, &result->confs[i]) != OK )
		    return FAILURE;
	    }
	    
	    skipwhite(fp);
	    if ( my_getc(fp) != '}' )
	    {
		kom_errno = KOM_SERVER_IS_CRAZY;
		return FAILURE;
	    }
	    
	    break;
	case '*':
	    if ( result->confs != NULL )
	    {
		isc_free(result->confs);
		result->confs = NULL;
	    }
	    break;
	default:
	    kom_errno = KOM_SERVER_IS_CRAZY;
	    return FAILURE;
	}
    }
    else
    {
	skipwhite(fp);
	if ( my_getc(fp) != '*' ) 
	{
	    kom_errno = KOM_SERVER_IS_CRAZY;
	    return FAILURE;
	}
	if ( result->confs != NULL )
	{
	    isc_free(result->confs);
	    result->confs = NULL;
	}
    }
    return OK;
}


extern Success
parse_conference(FILE *fp,
		 Conference *result)
{
    if (parse_string(fp, &result->name) != OK
	|| parse_conf_type(fp, &result->type) != OK
	|| parse_time (fp, &result -> creation_time) != OK
	|| parse_time (fp, &result -> last_written) != OK)
    {
	return FAILURE;
    }
    

    result -> creator		= parse_long (fp);
    result -> presentation	= parse_long (fp);
    result -> supervisor	= parse_long (fp);
    result -> permitted_submitters = parse_long (fp);
    result -> super_conf	= parse_long (fp);
    result -> msg_of_day	= parse_long (fp);
    result -> nice		= parse_long (fp);
    result -> no_of_members	= parse_long (fp);
    result -> first_local_no	= parse_long (fp);
    result -> no_of_texts	= parse_long (fp);

    return OK;
}

extern Success
parse_mark_list(FILE *fp,
		Mark_list *result)
{
    unsigned short i;

    result->no_of_marks = parse_long(fp);

    if ( result->no_of_marks > 0 )
    {
	skipwhite(fp);
	switch(my_getc(fp))
	{
	case '{':
	    result->marks = REALLOC(result->marks,
				    (result->no_of_marks
				     * sizeof(Mark)));

	    if ( result->marks == NULL )
	    {
		kom_errno = KOM_OUT_OF_MEMORY;
		return FAILURE;
	    }
	    
	    for ( i = 0; i < result->no_of_marks; i++)
		if ( parse_mark(fp, &result->marks[ i ] ) != OK )
		{
		    return FAILURE;
		}
	    
	    skipwhite(fp);
	    if ( my_getc(fp) != '}' )
	    {
		kom_errno = KOM_SERVER_IS_CRAZY;
		return FAILURE;
	    }
	    
	    break;

	case '*':
	    if ( result->marks != NULL )
	    {
		isc_free(result->marks);
		result->marks = NULL;
	    }
	    break;

	default:
	    kom_errno = KOM_SERVER_IS_CRAZY;
	    return FAILURE;
	}
    }
    else
    {
	skipwhite(fp);
	if ( my_getc(fp) != '*' ) 
	{
	    kom_errno = KOM_SERVER_IS_CRAZY;
	    return FAILURE;
	}
	if ( result->marks != NULL )
	{
	    isc_free(result->marks);
	    result->marks = NULL;
	}
    }
    return OK;
}


extern Success
parse_Misc_infos(FILE * fp, 
		 unsigned short * no_of_miscs,
		 Misc_info ** misc_items)
{
    unsigned short i;

    *no_of_miscs = parse_long(fp);

    if (*no_of_miscs > 0 )
    {
	skipwhite(fp);
	switch(my_getc(fp))
	{
	case '{':
	    *misc_items = REALLOC(*misc_items, 
				  (*no_of_miscs * sizeof(Misc_info)));

	    if (*misc_items == NULL)
	    {
		kom_errno = KOM_OUT_OF_MEMORY;
		return FAILURE;
	    }
	    
	    for (i = 0; i < *no_of_miscs; i++)
		if (parse_misc_info(fp, &((*misc_items)[i])) != OK )
		{
		    return FAILURE;
		}
	    
	    skipwhite(fp);
	    if (my_getc(fp) != '}')
	    {
		kom_errno = KOM_SERVER_IS_CRAZY;
		return FAILURE;
	    }
	    
	    break;

	case '*':
	    if (*misc_items != NULL)
	    {
		isc_free(*misc_items);
		*misc_items = NULL;
	    }
	    break;

	default:
	    kom_errno = KOM_SERVER_IS_CRAZY;
	    return FAILURE;
	}
    }
    else
    {
	skipwhite(fp);
	if (my_getc(fp) != '*') 
	{
	    kom_errno = KOM_SERVER_IS_CRAZY;
	    return FAILURE;
	}
	if (*misc_items != NULL)
	{
	    isc_free(*misc_items);
	    *misc_items = NULL;
	}
    }
    return OK;
}

extern Success
parse_text_stat(FILE *fp,
		Text_stat *result)
{
    if (parse_time(fp, &result->creation_time) != OK)
    {
        return FAILURE;
    }

    result->author = parse_long(fp);
    result->no_of_lines = parse_long(fp);
    result->no_of_chars = parse_long(fp);
    result->no_of_marks = parse_long(fp);

    return parse_Misc_infos(fp, &result->no_of_misc, &result->misc_items);

    return OK;
}	


extern Success
parse_text_list(FILE *fp,
		Text_list *result)
{
    unsigned int i;

    result->first_local_no = parse_long(fp);
    result->no_of_texts = parse_long(fp);

    if ( result->no_of_texts > 0 )
    {
	skipwhite(fp);
	switch(my_getc(fp))
	{
	case '{':
	    result->texts = REALLOC(result->texts,
				    (result->no_of_texts
				     * sizeof(Text_no)));

	    if ( result->texts == NULL )
	    {
		kom_errno = KOM_OUT_OF_MEMORY;
		return FAILURE;
	    }
	    
	    for ( i = 0; i < result->no_of_texts; i++)
		result->texts[ i ] = parse_long(fp);
	    
	    skipwhite(fp);
	    if ( my_getc(fp) != '}' )
	    {
		kom_errno = KOM_SERVER_IS_CRAZY;
		return FAILURE;
	    }
	    
	    break;

	case '*':
	    if ( result->texts != NULL )
	    {
		isc_free(result->texts);
		result->texts = NULL;
	    }
	    break;

	default:
	    kom_errno = KOM_SERVER_IS_CRAZY;
	    return FAILURE;
	}
    }
    else
    {
	skipwhite(fp);
	if ( my_getc(fp) != '*' ) 
	{
	    kom_errno = KOM_SERVER_IS_CRAZY;
	    return FAILURE;
	}
	if ( result->texts != NULL )
	{
	    isc_free(result->texts);
	    result->texts = NULL;
	}
    }
    return OK;
}
    

extern Success
parse_info(FILE *fp,
	   Info *result)
{
    result->version = parse_long(fp);
    result->conf_pres_conf = parse_long(fp);
    result->pers_pres_conf = parse_long(fp);
    result->motd_conf = parse_long(fp);
    result->kom_news_conf = parse_long(fp);
    result->motd_of_lyskom = parse_long(fp);
    return OK;
}


extern Success
parse_string(FILE *fp,
	     String *result)
{
    String_size i;
    int c;
    
    result->len = parse_long(fp);

    if ( my_getc(fp) != 'H' )
    {
	kom_errno = KOM_SERVER_IS_CRAZY;
	return FAILURE;
    }

    result->string = REALLOC(result->string, result->len);

    if ( result->string == NULL )
    {
	kom_errno = KOM_OUT_OF_MEMORY;
	return FAILURE;
    }
    
    for ( i = 0; i < result->len; i++ )
    {
	if ( (c=my_getc(fp)) == EOF )
	{
	    kom_errno = KOM_SERVER_IS_CRAZY;
	    return FAILURE;
	}
	result->string[ i ] = c;
    }
    

    return OK;
}


extern Success
parse_member_list(FILE *fp,
		  Member_list *result)
{
    unsigned short i;

    result->no_of_members = parse_long(fp);
    if ( result->no_of_members > 0 )
    {
	skipwhite(fp);
	switch(my_getc(fp))
	{
	case '{':
	    result->members = REALLOC(result->members,
				      (result->no_of_members
				       * sizeof(Member)));

	    if ( result->members == NULL )
	    {
		kom_errno = KOM_OUT_OF_MEMORY;
		return FAILURE;
	    }
	    
	    for ( i = 0; i < result->no_of_members; i++)
		if ( parse_member(fp, &result->members[ i ]) != OK )
		{
		    return FAILURE;
		}
	    
	    
	    skipwhite(fp);
	    if ( my_getc(fp) != '}' )
	    {
		kom_errno = KOM_SERVER_IS_CRAZY;
		return FAILURE;
	    }
	    
	    break;

	case '*':
	    if ( result->members != NULL )
	    {
		isc_free(result->members);
		result->members = NULL;
	    }
	    break;

	default:
	    kom_errno = KOM_SERVER_IS_CRAZY;
	    return FAILURE;
	}
    }
    else
    {
	skipwhite(fp);
	if ( my_getc(fp) != '*' ) 
	{
	    kom_errno = KOM_SERVER_IS_CRAZY;
	    return FAILURE;
	}
	if ( result->members != NULL )
	{
	    isc_free(result->members);
	    result->members = NULL;
	}
    }
    return OK;
}


extern Success
parse_member(FILE *fp,
	     Member *result)
{
    result->member = parse_long(fp);
    return OK;
}

extern Success
parse_mark(FILE *fp,
	   Mark *result)
{
    result->text_no = parse_long(fp);
    result->mark_type = parse_long(fp);

    if (result->text_no == 0
	&& result->mark_type == 0)
    {
        kom_errno = KOM_SERVER_IS_CRAZY;
	return FAILURE;
    }
    return OK;
}

extern Success
parse_misc_info(FILE *fp, 
		Misc_info *result)
{
    result->type = (Info_type) parse_long(fp);
    
    switch(result->type)
    {
    case recpt:
	result->datum.recipient = parse_long(fp);
	break;
	
    case cc_recpt:
	result->datum.cc_recipient = parse_long(fp);
	break;
	
    case loc_no:
	result->datum.local_no = parse_long(fp);
	if (result->datum.local_no == 0)
	{
	    kom_errno = KOM_SERVER_IS_CRAZY;
	    return FAILURE;
	}
	break;
	
    case rec_time:
        if (parse_time(fp, &result->datum.received_at) != OK)
	{
	    return FAILURE;
	}
	break;
	
    case comm_to:
	result->datum.comment_to = parse_long(fp);
	if (result->datum.comment_to == 0)
	{
	    kom_errno = KOM_SERVER_IS_CRAZY;
	    return FAILURE;
	}
	break;
	
    case comm_in:
	result->datum.commented_in = parse_long(fp);
	if (result->datum.commented_in == 0)
	{
	    kom_errno = KOM_SERVER_IS_CRAZY;
	    return FAILURE;
	}
	break;
	
    case footn_to:
	result->datum.footnote_to = parse_long(fp);
	if (result->datum.footnote_to == 0)
	{
	    kom_errno = KOM_SERVER_IS_CRAZY;
	    return FAILURE;
	}
	break;
	
    case footn_in:
	result->datum.footnoted_in = parse_long(fp);
	if (result->datum.footnoted_in == 0)
	{
	    kom_errno = KOM_SERVER_IS_CRAZY;
	    return FAILURE;
	}
	break;
	
    case sent_by:
	result->datum.sender = parse_long(fp);
	break;
	
    case sent_at:
        if (parse_time(fp, &result->datum.sent_at) != OK)
	{
	    return FAILURE;
	}
	break;

    default:
        kom_errno = KOM_SERVER_IS_CRAZY;
	return FAILURE;
    }
    return OK;
}

extern Success
parse_priv_bits(FILE *fp,
		Priv_bits *result)
{
    skipwhite(fp);

    READ_ONE_FLAG(wheel);
    READ_ONE_FLAG(admin);
    READ_ONE_FLAG(statistic);
    READ_ONE_FLAG(create_pers);
    READ_ONE_FLAG(create_conf);
    READ_ONE_FLAG(change_name);
    READ_ONE_FLAG(extern_gw);
    READ_ONE_FLAG(flg8);
    READ_ONE_FLAG(flg9);
    READ_ONE_FLAG(flg10);
    READ_ONE_FLAG(flg11);
    READ_ONE_FLAG(flg12);
    READ_ONE_FLAG(flg13);
    READ_ONE_FLAG(flg14);
    READ_ONE_FLAG(flg15);
    READ_ONE_FLAG(flg16);

    return OK;
}


extern Success
parse_personal_flags(FILE *fp,
		     Personal_flags *result)
{
    skipwhite(fp);
    
    READ_ONE_FLAG(unread_is_secret);
    READ_ONE_FLAG(flg2);
    READ_ONE_FLAG(flg3);
    READ_ONE_FLAG(flg4);
    READ_ONE_FLAG(flg5);
    READ_ONE_FLAG(flg6);
    READ_ONE_FLAG(flg7);
    READ_ONE_FLAG(flg8);

    return OK;
}	

extern Success
parse_conf_type(FILE *fp,
		Conf_type *result)
{
    skipwhite(fp);
    
    READ_ONE_FLAG(rd_prot);
    READ_ONE_FLAG(original);
    READ_ONE_FLAG(secret);
    READ_ONE_FLAG(letter_box);

    return OK;
}


extern Success
parse_who_info(FILE *fp,
	       Who_info *result)
{
    result->person = parse_long(fp);
    result->working_conference = parse_long(fp);
    result->session_no = parse_long(fp);
    
    if ( parse_string(fp, &result->what_am_i_doing) != OK
	|| parse_string(fp, &result->username) != OK )
    {
	kom_errno = KOM_SERVER_IS_CRAZY;
	return FAILURE;
    }

    return OK;
}

    
extern Success
parse_conf_no_list(FILE *fp,
		   Conf_no_list *result)
{
    unsigned short i;
    
    skipwhite(fp);

    result->no_of_confs = parse_long(fp);
    
    if ( result->no_of_confs > 0 )
    {
	skipwhite(fp);
	switch(my_getc(fp))
	{
	case '{':
	    result->conf_nos = REALLOC(result->conf_nos,
				       (result->no_of_confs
					* sizeof(Conf_no)));

	    if ( result->conf_nos == NULL )
	    {
		kom_errno = KOM_OUT_OF_MEMORY;
		return FAILURE;
	    }
	    
	    for ( i = 0; i < result->no_of_confs; i++)
	    {
		result->conf_nos[ i ] = parse_long(fp);
	    }
	    
	    skipwhite(fp);
	    if ( my_getc(fp) != '}' )
	    {
		kom_errno = KOM_SERVER_IS_CRAZY;
		return FAILURE;
	    }
	    
	    break;

	case '*':
	    if ( result->conf_nos != NULL )
	    {
		isc_free(result->conf_nos);
		result->conf_nos = NULL;
	    }
	    break;

	default:
	    kom_errno = KOM_SERVER_IS_CRAZY;
	    return FAILURE;
	}
    }
    else
    {
	skipwhite(fp);
	if ( my_getc(fp) != '*' ) 
	{
	    kom_errno = KOM_SERVER_IS_CRAZY;
	    return FAILURE;
	}
	if ( result->conf_nos != NULL )
	{
	    isc_free(result->conf_nos);
	    result->conf_nos = NULL;
	}
    }
    return OK;
}


extern Success
parse_conf_z_info_list(FILE *fp,
		       Conf_z_info_list *result)
{
    unsigned short i;
    
    skipwhite(fp);

    result->no_of_confs = parse_long(fp);
    
    if ( result->no_of_confs > 0 )
    {
	skipwhite(fp);
	switch(my_getc(fp))
	{
	case '{':
	    result->confs = REALLOC(result->confs,
				    (result->no_of_confs
				     * sizeof(Conf_z_info)));

	    if ( result->confs == NULL )
	    {
		kom_errno = KOM_OUT_OF_MEMORY;
		return FAILURE;
	    }
	    
	    for ( i = 0; i < result->no_of_confs; i++)
	    {
		result->confs[i] = EMPTY_CONF_Z_INFO;
		if (parse_conf_z_info(fp, &(result->confs[i])) != OK)
		    return FAILURE;
		/* BUG: possible memory corruption. We might have partly filled
		   the array and then we are returning it. */
	    }
	    
	    skipwhite(fp);
	    if ( my_getc(fp) != '}' )
	    {
		kom_errno = KOM_SERVER_IS_CRAZY;
		return FAILURE;
	    }
	    
	    break;

	case '*':
	    if ( result->confs != NULL )
	    {
		isc_free(result->confs);
		result->confs = NULL;
	    }
	    break;

	default:
	    kom_errno = KOM_SERVER_IS_CRAZY;
	    return FAILURE;
	}
    }
    else
    {
	skipwhite(fp);
	if ( my_getc(fp) != '*' ) 
	{
	    kom_errno = KOM_SERVER_IS_CRAZY;
	    return FAILURE;
	}
	if ( result->confs != NULL )
	{
	    isc_free(result->confs);
	    result->confs = NULL;
	}
    }
    return OK;
}


extern Success
parse_conf_z_info(FILE *fp, Conf_z_info * result)
{
    if (parse_string(fp, &result->name) != OK
	|| parse_conf_type(fp, &result->type) != OK)
	return FAILURE;
    result->conf_no = parse_long(fp);
    return OK;
}



extern Success
parse_dynamic_session_info_list(FILE *fp,
				Dynamic_session_info_list *result)
{
    unsigned short i;
    
    skipwhite(fp);

    result->no_of_sessions = parse_long(fp);
    
    if ( result->no_of_sessions > 0 )
    {
	skipwhite(fp);
	switch(my_getc(fp))
	{
	case '{':
	    result->sessions = REALLOC(result->sessions,
				       (result->no_of_sessions
					* sizeof(Dynamic_session_info)));

	    if ( result->sessions == NULL )
	    {
		kom_errno = KOM_OUT_OF_MEMORY;
		return FAILURE;
	    }
	    
	    for ( i = 0; i < result->no_of_sessions; i++)
	    {
		result->sessions[i] = EMPTY_DYNAMIC_SESSION_INFO;
		if (parse_dynamic_session_info(fp, &(result->sessions[i])) != OK)
		    return FAILURE;
		/* BUG: possible memory corruption. We might have partly filled
		   the array and then we are returning it. */
	    }
	    
	    skipwhite(fp);
	    if ( my_getc(fp) != '}' )
	    {
		kom_errno = KOM_SERVER_IS_CRAZY;
		return FAILURE;
	    }
	    
	    break;

	case '*':
	    if ( result->sessions != NULL )
	    {
		isc_free(result->sessions);
		result->sessions = NULL;
	    }
	    break;

	default:
	    kom_errno = KOM_SERVER_IS_CRAZY;
	    return FAILURE;
	}
    }
    else
    {
	skipwhite(fp);
	if ( my_getc(fp) != '*' ) 
	{
	    kom_errno = KOM_SERVER_IS_CRAZY;
	    return FAILURE;
	}
	if ( result->sessions != NULL )
	{
	    isc_free(result->sessions);
	    result->sessions = NULL;
	}
    }
    return OK;
}


extern Success
parse_dynamic_session_info(FILE *fp, Dynamic_session_info * result)
{
    result->session = parse_long(fp);
    result->person = parse_long(fp);
    result->working_conference = parse_long(fp);
    result->idle_time = parse_long(fp);

    if (parse_session_flags(fp, &result->flags) != OK
	|| parse_string(fp, &result->what_am_i_doing) != OK)
	return FAILURE;
    return OK;
}

extern Success
parse_session_flags(FILE *fp, Session_flags * result)
{
    /* This function can handle 4 or 8 flags in the bitfield. */
    /* The for flags possibly not provided are always zeroed. */
    int c;

    result->reserved4 = 0;
    result->reserved5 = 0;
    result->reserved6 = 0;
    result->reserved7 = 0;

    skipwhite(fp);

    READ_ONE_FLAG(invisible);
    READ_ONE_FLAG(user_active_used);
    READ_ONE_FLAG(user_absent);
    READ_ONE_FLAG(reserved3);

    switch ((c = my_getc(fp)))
    {
    case '1':
    case '0':
        my_ungetc(c, fp);
	break;
    default:
        return OK;
    }

    READ_ONE_FLAG(reserved4);
    READ_ONE_FLAG(reserved5);
    READ_ONE_FLAG(reserved6);
    READ_ONE_FLAG(reserved7);

    switch ((c = my_getc(fp)))
    {
    case '1':
    case '0':
        /* Bitfield is too long. */
        kom_errno = KOM_SERVER_IS_CRAZY;
	return FAILURE;
    default:
        return OK;
    }
}

extern Success
parse_time(FILE *fp, struct tm *timeptr)
{
    timeptr->tm_sec = parse_long(fp);
    timeptr->tm_min = parse_long(fp);
    timeptr->tm_hour = parse_long(fp);
    timeptr->tm_mday = parse_long(fp);
    timeptr->tm_mon = parse_long(fp);
    timeptr->tm_year = parse_long(fp);
    timeptr->tm_wday = parse_long(fp);
    timeptr->tm_yday = parse_long(fp);
    timeptr->tm_isdst = parse_long(fp);

    /* A time cannot be all zeroes. */
    if (timeptr->tm_sec == 0
	&& timeptr->tm_sec == 0
	&& timeptr->tm_min == 0
	&& timeptr->tm_hour == 0
	&& timeptr->tm_mday == 0
	&& timeptr->tm_mon == 0
	&& timeptr->tm_year == 0
	&& timeptr->tm_wday == 0
	&& timeptr->tm_yday == 0
	&& timeptr->tm_isdst == 0)
    {
        kom_errno = KOM_SERVER_IS_CRAZY;
        return FAILURE;
    }

    return OK;
}


/* My own implementation of getc and putc. This is because there are
 * problems with these and sockets on some systems: solaris, ultrix.
 */

#ifdef DEBUG
#define BUFSIZE 30
#else
#define BUFSIZE 16384
#endif
static unsigned char buffer[BUFSIZE];
static int  readstart = 1;
static int  readend = 0;
static FILE * thefilestar = 0;

int
my_getc(FILE *fp)
{
    int c;
    int laps = 0;

    /* Assert... */
    if (thefilestar) {
        if (thefilestar != fp)
	    return getc(fp);
    } else
        thefilestar = fp;

    while (readstart > readend) {
        /* We have to read some from the socket. */
	readstart = 0;
        if ((readend = read(fileno(fp), buffer, BUFSIZE)) == -1) {
	    return -1;
	}
#ifdef DEBUG
	fprintf(stderr,"DEBUG: Read %d bytes: ",readend);
	write(2,buffer,readend);
#endif
	if (!readend)
	{
	    if (laps++ > 3)	/*+++ Skitfult!! */
	        return EOF;
	    sleep(1);
	}
	readend--;
    }
    
    c = (int)buffer[readstart];
    readstart++;
    return c;
}

void
my_ungetc(int c, FILE *fp)
{
    /* Assert... */

    if (c == EOF)
        return;

    if (thefilestar) {
        if (thefilestar != fp)
	{
	    ungetc(c, fp);
	    return;
	}
    } else
        thefilestar = fp;

    readstart--;
    buffer[readstart] = (char) c;
    return;
}
