# vm.s
#
# The kForth Virtual Machine
#
# Copyright (c) 1998--2004 Krishna Myneni, Creative Consulting for
#   Research and Education
#
# This software is provided under the terms of the General Public License.
#
# Usage from C++
#
#       extern "C" int vm (byte* ip);
#       ecode = vm(ip);
#
# Originally written for the A386 assembler
# Revisions:
#	3-21-98
#	8-25-1998 ported to GNU assembler under Linux
#	9-8-1998  additional functions added: c@, c!, f=, f<>
#	9-10-1998 added: and, or, not, xor
#	9-11-1998 added: = , <>, <, >, <=, >=
#	9-15-1998 added: +!, fsqrt, fsin, fcos, -rot, sp@, rp@
#	9-17-1998 added: a@
#	9-27-1998 added: mod, 2*, 2/
#	10-1-1998 added: pick
#	10-4-1998 fixed signed integer division and mod; added /MOD
#	10-6-1998 added ?dup
#	10-14-1998 added count
#	10-16-1998 fixed L_div error
#	10-19-1998 added 0<, 0=, 0>, true, false
#	10-20-1998 added 2+, 2-
#	02-09-1999 added execute
#	03-01-1999 added open, lseek, close, read, write
#	03-02-1999 added ioctl
#	03-03-1999 added usleep	
#	03-07-1999 added fill, cmove
#	03-09-1999 interchanged meaning of execute and call
#		   to be consistent with ANS Forth
#	03-27-1999 added +loop, unloop
#	03-29-1999 added roll
#	03-31-1999 added cmove>, key
#	05-05-1999 fixed +loop
#	05-06-1999 added fround
#	05-15-1999 added floor
#	05-26-1999 added fatan2, lshift, rshift
#	05-27-1999 added u<, quit, base
#	06-02-1999 added */, */mod
#	06-09-1999 call CPP functions from vm
#	07-18-1999 added find
#	09-06-1999 added pTIB, word, tick
#	09-12-1999 added system
#	10-04-1999 added create, variable, fvariable as intrinsic words
#	10-06-1999 added constant, fconstant as intrinsic words
#	10-07-1999 added chdir
#       10-08-1999 added erase, brackettick
#	10-09-1999 added time&date, ms, question, bl
#       10-10-1999 added char, forget, cold
#       10-20-1999 added >file, console
#	10-28-1999 added key?
#	12-24-1999 added hooks for f0=, f0<, f0>, u>, s>d, d>f, f>d, 
#	              um*, um/mod, m*, m+, m/, m*/
#	12-25-1999 added cells, cell+, dfloats, dfloat+
#	12-27-1999 added bye
#	01-08-2000 fixed f0<, increased vm loop efficiency
#	01-13-1999 added ?allot, fixed f0=
#	01-22-2000 modified + to remove ordering sensitivity for address arithmetic
#	01-23-2000 added 0<>, [char], .r, u.r, changed opcodes for 
#	             relational operators
#	01-24-2000 added CPP_literal, CPP_cquote, CPP_squote, CPP_dotquote
#	02-04-2000 implemented m*, um*
#	02-26-2000 fixed fm/mod, added sm/rem, um/mod
#	03-02-2000 modified QUIT to clear the return stacks, added CPP_do
#	03-05-2000 added CPP_begin, CPP_while, CPP_repeat, CPP_until, CPP_again,
#	             CPP_leave, CPP_if, CPP_else, CPP_then, CPP_lparen
#	05-17-2000 added CPP_does
#	05-18-2000 fix L_plusloop for negative increments
#	06-04-2000 fix L_roll to roll the typestack as well
#	06-11-2000 added CPP_case, CPP_endcase, CPP_of, and CPP_endof
#	06-15-2000 added CPP_querydo, CPP_abortquote
#	09-05-2000 handled overflow condition on um/mod; added code for m/;
#		     added m+	;  hooks for CPP_lbracket, CPP_rbracket, CPP_ddot
#	11-30-2000 added L_dabs, L_dnegate, L_udmstar, L_mstarslash
#	01-26-2001 changed jumptable address for usleep to C_usec
#	04-01-2001 changed jmpl to jmp for L_fzerolt
#	04-22-2001 added L_dplus, L_dminus
#	04-24-2001 added L_twopush, L_twopop, L_tworfetch
#       05-07-2001 added L_definition
#	05-08-2001 improved vm execution efficiency by about 10%
#	05-13-2001 added CPP_dotparen, L_dlt, L_dzeroeq, L_deq
#	05-20-2001 added CPP_sharp, CPP_bracketsharp, CPP_sharpbracket, 
#	             CPP_sharps, CPP_sign, CPP_hold; fixed problem with L_dnegate
#	07-12-2001 fixed type-stack alignment problem with L_utmslash
#	08-26-2001 updated L_fval to avoid routing fp through NDP, changed all
#	             typestack settings of OP_FVAL to OP_IVAL
#	09-02-2001 added L_tobody
#       12-08-2001 added CPP_evaluate
#       07-29-2002 fixed L_ret to set GlobalRp and GlobalRtp to bottom upon
#                    return from vm.
#	07-31-2002 fixed L_cmovefrom for zero count argument; added
#		     CPP_backslash; fixed bug in L_fmslashmod for divisor < 0
#	08-01-2002 added C_msfetch, C_search, C_compare
#       09-11-2002 modified vm and L_ret to make vm re-entrant
#	09-29-2002 restored L_usleep since it provides a different function 
#	             than L_ms, added CPP_immediate
#	04-11-2003 renamed L_ftos to L_froundtos
#	04-15-2003 added L_ftrunc and L_ftrunctos
#	04-18-2003 fixed L_ftrunctos typestack pointer update
#	06-15-2003 recoded L_rot, L_swap, and L_over for increased efficiency
#	01-31-2004 extended JumpTable past 256 entries, 
#                    added CPP_include, CPP_source, CPP_refill, and
#                    CPP_nondeferred, CPP_state, CPP_colon, CPP_semicolon
#	02-09-2004 added CPP_allocate and CPP_free
#	03-18-2004 added L_fsincos, C_fsinh, C_fcosh, C_ftanh, C_fasinh,
#                    C_facosh, C_fatanh
#       03-21-2004 implemented L_fsincos
#       03-30-2004 modified L_ret to remove unneeded instructions
#       04-08-2004 added CPP_postpone and CPP_compilecomma
#       04-09-2004 added CPP_semicolon
#       04-18-2004 completed recoding with macros for hybrid threading.
#	04-26-2004 exported L_udmstar and L_utmslash for use by ForthVM.cpp
#       06-20-2004 modified L_call to prevent loss of return error code
.equ WSIZE,	4

.equ OP_ADDR,	65
.equ OP_FVAL,	70
.equ OP_IVAL,	73
.equ OP_RET,	238
.equ SIGN_MASK,	0x80000000
	
# Error Codes

.equ E_NOT_ADDR,	1
.equ E_DIV_ZERO,	4
.equ E_RET_STK_CORRUPT,	5
.equ E_UNKNOWN_OP,	6

.data
FCONST_180: .double 180.
JumpTable: .int L_false, L_true, L_cells, L_cellplus # 0 -- 3
           .int L_dfloats, L_dfloatplus, CPP_case, CPP_endcase  # 4 -- 7
           .int CPP_of, CPP_endof, C_open, C_lseek     # 8 -- 11
           .int C_close, C_read, C_write, C_ioctl # 12 -- 15
           .int L_usleep, L_ms, C_msfetch, L_nop  # 16 -- 19
           .int L_fill, L_cmove, L_cmovefrom, CPP_dotparen # 20 -- 23
           .int CPP_bracketsharp, CPP_tofile, CPP_console, CPP_sharpbracket  # 24 -- 27
           .int CPP_sharps, CPP_squote, CPP_cr, L_bl    # 28 -- 31
           .int CPP_spaces, L_store, CPP_cquote, CPP_sharp # 32 -- 35
           .int CPP_sign, L_mod, L_and, L_tick       # 36 -- 39
           .int CPP_lparen, CPP_hold, L_mul, L_add  # 40 -- 43
           .int L_nop, L_sub, CPP_dot, L_div  # 44 -- 47
           .int L_dabs, L_dnegate, L_umstar, L_umslashmod   # 48 -- 51
           .int L_mstar, L_mplus, L_mslash, L_mstarslash # 52 -- 55
           .int L_fmslashmod, L_smslashrem, CPP_colon, CPP_semicolon # 56 -- 59
           .int L_lt, L_eq, L_gt, L_question      # 60 -- 63
           .int L_fetch, L_addr, L_base, L_call   # 64 -- 67
           .int L_definition, L_erase, L_fval, CPP_forget # 68 -- 71
           .int L_tobody, L_ival, CPP_evaluate, C_key     # 72 -- 75
           .int L_lshift, L_slashmod, C_numberquery, CPP_dotr # 76 -- 79
           .int CPP_ddot, C_keyquery, L_rshift, CPP_dots  # 80 -- 83
           .int C_accept, CPP_char, CPP_bracketchar, CPP_word  # 84 -- 87
           .int L_starslash, L_starslashmod, CPP_udotr, CPP_lbracket  # 88 -- 91
           .int CPP_backslash, CPP_rbracket, L_xor, CPP_literal  # 92 -- 95
           .int CPP_queryallot, CPP_allot, L_binary, L_count # 96 -- 99
           .int L_decimal, CPP_emit, CPP_fdot, CPP_cold # 100 -- 103
           .int L_hex, L_i, L_j, CPP_brackettick         # 104 -- 107
           .int CPP_fvariable, C_timeanddate, CPP_find, CPP_constant # 108 -- 111
           .int CPP_immediate, CPP_fconstant, CPP_create, CPP_dotquote  # 112 -- 115
           .int CPP_type, CPP_udot, CPP_variable, CPP_words # 116 -- 119
           .int CPP_does, C_system, C_chdir, C_search   # 120 -- 123
           .int L_or, C_compare, L_not, L_nop     # 124 -- 127
           .int L_fsin, L_fcos, C_ftan, C_fasin   # 128 -- 131
           .int C_facos, C_fatan, C_fexp, C_fln   # 132 -- 135
           .int C_flog, L_fatan2, L_ftrunc, L_ftrunctos    # 136 -- 139
           .int C_fmin, C_fmax, L_floor, L_fround # 140 -- 143
           .int L_dlt, L_dzeroeq, L_deq, L_twopush  # 144 -- 147
           .int L_twopop, L_tworfetch, L_stod, L_stof # 148 -- 151
           .int L_dtof, L_froundtos, L_ftod, L_degtorad  # 152 -- 155
           .int L_radtodeg, L_dplus, L_dminus, L_nop   # 156 -- 159
           .int L_inc, L_dec, L_abs, L_neg        # 160 -- 163
           .int L_min, L_max, L_twostar, L_twodiv # 164 -- 167
           .int L_twoplus, L_twominus, L_cfetch, L_cstore  # 168 -- 171
           .int L_wfetch, L_wstore, L_dffetch, L_dfstore  # 172 -- 175
           .int L_sffetch, L_sfstore, L_spfetch, L_plusstore # 176 -- 179
           .int L_fadd, L_fsub, L_fmul, L_fdiv    # 180 -- 183
           .int L_fabs, L_fneg, C_fpow, L_fsqrt   # 184 -- 187
           .int L_nop, L_nop, L_feq, L_fne        # 188 -- 191
           .int L_flt, L_fgt, L_fle, L_fge        # 192 -- 195
           .int L_fzeroeq, L_fzerolt, L_fzerogt, L_nop # 196 -- 199
           .int L_drop, L_dup, L_swap, L_over     # 200 -- 203
           .int L_rot, L_minusrot, L_nip, L_tuck  # 204 -- 207
           .int L_pick, L_roll, L_2drop, L_2dup   # 208 -- 211
           .int L_2swap, L_2over, L_2rot, L_depth # 212 -- 215
           .int L_querydup, CPP_if, CPP_else, CPP_then # 216 -- 219
           .int L_push, L_pop, L_puship, L_rfetch # 220 -- 223
           .int L_rpfetch, L_afetch, CPP_do, CPP_leave # 224 -- 227
           .int CPP_querydo, CPP_abortquote, L_jz, L_jnz  # 228 -- 231
           .int L_jmp, L_loop, L_plusloop, L_unloop  # 232 -- 235
           .int L_execute, CPP_recurse, L_ret, L_abort  # 236 -- 239
           .int L_quit, L_ge, L_le, L_ne          # 240 -- 243
           .int L_zeroeq, L_zerone, L_zerolt, L_zerogt # 244 -- 247
           .int L_ult, L_ugt, CPP_begin, CPP_while    # 248 -- 251
           .int CPP_repeat, CPP_until, CPP_again, CPP_bye  # 252 -- 255
	   .int L_nop, L_nop, L_nop, L_nop        # 256 -- 259
	   .int CPP_include, CPP_source, CPP_refill, CPP_nondeferred # 260--263
	   .int CPP_state, CPP_allocate, CPP_free, CPP_spstore  # 264--267
	   .int CPP_rpstore, L_nop, CPP_compilecomma, L_nop  # 268--271
	   .int CPP_postpone, L_nop, L_nop, L_nop # 272--275
	   .int L_nop, L_nop, L_nop, L_nop        # 276--279
	   .int L_nop, L_fsincos, C_facosh, C_fasinh  # 280--283
	   .int C_fatanh, C_fcosh, C_fsinh, C_ftanh   # 284--287
.text
	.align 4
.global JumpTable
.global L_depth, L_tick, L_quit, L_abort, L_ret, L_dabs
.global L_dminus, L_mstarslash, L_udmstar, L_utmslash

.macro NEXT
	movl GlobalIp, %ebx
	incl %ebx		 # increment the Forth instruction ptr
	movl %ebx, GlobalIp
	movb (%ebx), %al         # get the opcode
	movl JumpTable(,%eax,4), %ebx	# machine code address of word
	xor %eax, %eax
	jmpl *%ebx		# call the word
.endm

.macro SWAP
        movl GlobalSp, %ebx
        addl $WSIZE, %ebx
	movl %ebx, %ebp
	movl (%ebp), %ecx
	addl $WSIZE, %ebx
	movl (%ebx), %edx
	movl %ecx, (%ebx)
	movl %edx, (%ebp)	
#        movl WSIZE(%ebx), %eax
#        xchgl (%ebx), %eax
#        movl %eax, WSIZE(%ebx)
        movl GlobalTp, %ebx
        incl %ebx
	movl %ebx, %ebp
	movb (%ebp), %al
	incl %ebx
	movb (%ebx), %cl
	movb %al, (%ebx)
	movb %cl, (%ebp)
#        movb 1(%ebx), %al
#        xchgb (%ebx), %al
#        movb %al, 1(%ebx)
        xor %eax, %eax
.endm

.macro DUP
	movl GlobalSp, %ebx
        movl WSIZE(%ebx), %eax
        movl %eax, (%ebx)
	subl $WSIZE, %ebx
	movl %ebx, GlobalSp
        movl GlobalTp, %ebx
        movb 1(%ebx), %al
        movb %al, (%ebx)
        decl GlobalTp
        xor %eax, %eax
.endm 

.macro DROP
        addl $WSIZE, GlobalSp
        incl GlobalTp
.endm

.macro OVER
        movl GlobalSp, %ebx
        movl 2*WSIZE(%ebx), %eax
        movl %eax, (%ebx)
	subl $WSIZE, %ebx
	movl %ebx, GlobalSp
        movl GlobalTp, %ebx
        movb 2(%ebx), %al
        movb %al, (%ebx)
        decl GlobalTp
        xor %eax, %eax
.endm

.macro FOVER
	movl GlobalSp, %ebx
	movl %ebx, %ecx
	addl $3*WSIZE, %ebx
	movl (%ebx), %edx
	addl $WSIZE, %ebx
	movl (%ebx), %eax
	movl %ecx, %ebx
	movl %eax, (%ebx)
	subl $WSIZE, %ebx
	movl %edx, (%ebx)
	subl $WSIZE, %ebx
	movl %ebx, GlobalSp
	movl GlobalTp, %ebx
	movl %ebx, %ecx
	addl $3, %ebx
	movw (%ebx), %ax
	movl %ecx, %ebx
	decl %ebx
	movw %ax, (%ebx)
	decl %ebx
	movl %ebx, GlobalTp
	xor %eax, %eax	
.endm
	
.macro _PUSH
	movl $WSIZE, %eax
	movl GlobalSp, %ebx
	addl %eax, %ebx	
        movl %ebx, GlobalSp
	movl (%ebx), %ecx
	movl GlobalRp, %ebx
	movl %ecx, (%ebx)
	subl %eax, %ebx
	movl %ebx, GlobalRp
	movl GlobalTp, %ebx
	incl %ebx
	movl %ebx, GlobalTp
	movb (%ebx), %al
	movl GlobalRtp, %ebx
	movb %al, (%ebx)
	decl %ebx
	movl %ebx, GlobalRtp
        xor %eax, %eax
.endm
	
.macro _POP
	movl $WSIZE, %eax
	movl GlobalRp, %ebx
	addl %eax, %ebx
	movl %ebx, GlobalRp
	movl (%ebx), %ecx
	movl GlobalSp, %ebx
	movl %ecx, (%ebx)
	subl %eax, %ebx
	movl %ebx, GlobalSp
	movl GlobalRtp, %ebx
	incl %ebx
	movl %ebx, GlobalRtp
	movb (%ebx), %al
	movl GlobalTp, %ebx
	movb %al, (%ebx)
	decl %ebx
	movl %ebx, GlobalTp
	xor %eax, %eax
.endm
	
	
.macro _AND
	movl GlobalSp, %ebx
	addl $WSIZE, %ebx
	movl (%ebx), %eax
	addl $WSIZE, %ebx
	pushl %ebx
	movl (%ebx), %ebx
	andl %ebx, %eax
	popl %ebx
	movl %eax, (%ebx)
	subl $WSIZE, %ebx
	movl %ebx, GlobalSp
	incl GlobalTp
	movl GlobalTp, %ebx
	movb $OP_IVAL, 1(%ebx)
	xor %eax, %eax
.endm
	
.macro _OR
	movl $WSIZE, %eax
	addl %eax, GlobalSp
	incl GlobalTp
	movl GlobalSp, %ebx
	movl (%ebx), %eax
	movl WSIZE(%ebx), %ebx
	orl %ebx, %eax
	movl GlobalSp, %ebx
	movl %eax, WSIZE(%ebx)
	movl GlobalTp, %ebx
	movb $OP_IVAL, 1(%ebx)
	xor %eax, %eax
.endm
	
.macro _XOR
	movl $WSIZE, %eax
	addl %eax, GlobalSp
	incl GlobalTp
	movl GlobalSp, %ebx
	movl (%ebx), %eax
	movl WSIZE(%ebx), %ebx
	xorl %ebx, %eax
	movl GlobalSp, %ebx
	movl %eax, WSIZE(%ebx)
	movl GlobalTp, %ebx
	movb $OP_IVAL, 1(%ebx)
	xor %eax, %eax
.endm
	
.macro _NOT
	movl GlobalSp, %ebx
	movl WSIZE(%ebx), %eax
	notl %eax
	movl %eax, WSIZE(%ebx)
	xor %eax, %eax
.endm
	
.macro DNEGATE
	movl GlobalSp, %ebx
	addl $WSIZE, %ebx
	movl %ebx, %ecx
	addl $WSIZE, %ebx
	movl (%ebx), %eax
	notl %eax
	clc
	addl $1, %eax
	movl %eax, (%ebx)
	movl %ecx, %ebx
	movl (%ebx), %eax
	notl %eax
	adcl $0, %eax
	movl %eax, (%ebx)
	xor %eax, %eax	
.endm
	
.macro DPLUS
	movl GlobalSp, %ebx
	addl $2*WSIZE, %ebx
	movl (%ebx), %eax
	clc
	addl 2*WSIZE(%ebx), %eax
	movl %eax, 2*WSIZE(%ebx)
	movl WSIZE(%ebx), %eax
	adcl -WSIZE(%ebx), %eax
	movl %eax, WSIZE(%ebx)
	movl %ebx, GlobalSp
	incl GlobalTp
	incl GlobalTp
	xor %eax, %eax
.endm
	
.macro DMINUS
	movl GlobalSp, %ebx
	addl $2*WSIZE, %ebx
	movl 2*WSIZE(%ebx), %eax
	clc
	subl (%ebx), %eax
	movl %eax, 2*WSIZE(%ebx)
	movl WSIZE(%ebx), %eax
	sbbl -WSIZE(%ebx), %eax
	movl %eax, WSIZE(%ebx)
	movl %ebx, GlobalSp
	incl GlobalTp
	incl GlobalTp
	xor %eax, %eax
.endm
	
.macro STARSLASH
	movl $WSIZE, %eax
	sall $1, %eax
        addl %eax, GlobalSp
        movl GlobalSp, %ebx
        movl WSIZE(%ebx), %eax
        imull (%ebx)
	idivl -WSIZE(%ebx)
	movl %eax, WSIZE(%ebx)
	incl GlobalTp
	incl GlobalTp
	xor %eax, %eax
.endm
	
				
.global vm
	.type	vm,@function
vm:	
        pushl %ebp
        pushl %ebx
        pushl %ecx
        pushl %edx
	pushl GlobalIp
	pushl vmEntryRp
        movl %esp, %ebp
        movl 28(%ebp), %ebx     # load the Forth instruction pointer
        movl %ebx, GlobalIp
	movl GlobalRp, %eax
	movl %eax, vmEntryRp
	xor %eax, %eax
next:
        movb (%ebx), %al         # get the opcode
	movl JumpTable(,%eax,4), %ebx	# machine code address of word
	xor %eax, %eax          # clear error code
	call *%ebx		# call the word
	movl GlobalIp, %ebx
	incl %ebx		 # increment the Forth instruction ptr
	movl %ebx, GlobalIp
	cmpb $0, %al		 # check for error
	jz next        
exitloop:
        cmpl $OP_RET, %eax         # return from vm?
        jnz vmexit
        xor %eax, %eax            # clear the error
vmexit:
	pop vmEntryRp
	pop GlobalIp
	pop %edx
        pop %ecx
        pop %ebx
        pop %ebp
        ret
L_nop:
        movl $E_UNKNOWN_OP, %eax   # unknown operation
        ret
L_quit:
	movl BottomOfReturnStack, %eax	# clear the return stacks
	movl %eax, GlobalRp
	movl %eax, vmEntryRp
	movl BottomOfReturnTypeStack, %eax
	movl %eax, GlobalRtp
	movl $8, %eax		# exit the virtual machine
	ret
L_abort:
	movl BottomOfStack, %eax
	movl %eax, GlobalSp
	movl BottomOfTypeStack, %eax
	movl %eax, GlobalTp
	jmp L_quit
L_base:	
	movl GlobalSp, %ebx
	movl $Base, %eax
	movl %eax, (%ebx)
	subl $WSIZE, %ebx
	movl %ebx, GlobalSp
	movl GlobalTp, %ebx
	movb $OP_ADDR, (%ebx)
	decl GlobalTp
	xor %eax, %eax
	NEXT
L_binary:
	movl $Base, %ebx
	movl $2, (%ebx)
	NEXT
L_decimal:	
	movl $Base, %ebx
	movl $10, (%ebx)
	NEXT
L_hex:	
	movl $Base, %ebx
	movl $16, (%ebx)
	NEXT		
L_false:
	movl GlobalSp, %ebx
	movl $0, (%ebx)
	subl $WSIZE, %ebx
	movl %ebx, GlobalSp
	movl GlobalTp, %ebx
	movb $OP_IVAL, (%ebx)
	decl GlobalTp
	NEXT
L_true:
	movl GlobalSp, %ebx
	movl $-1, (%ebx)
	subl $WSIZE, %ebx
	movl %ebx, GlobalSp
	movl GlobalTp, %ebx
	movb $OP_IVAL, (%ebx)
	decl GlobalTp
	NEXT
L_cells:
	movl GlobalSp, %ebx
	addl $WSIZE, %ebx
	movl (%ebx), %eax
	sall $2, %eax
	movl %eax, (%ebx)
	xor %eax, %eax
	NEXT
L_cellplus:
	movl GlobalSp, %ebx
	addl $WSIZE, %ebx
	movl (%ebx), %eax
	addl $WSIZE, %eax
	movl %eax, (%ebx)
	xor %eax, %eax
	NEXT
L_dfloats:	
	movl GlobalSp, %ebx
	addl $WSIZE, %ebx
	movl (%ebx), %eax
	sall $3, %eax
	movl %eax, (%ebx)
	xor %eax, %eax
	NEXT
L_dfloatplus:	
	movl GlobalSp, %ebx
	addl $WSIZE, %ebx
	movl (%ebx), %eax
	addl $WSIZE, %eax
	addl $WSIZE, %eax
	movl %eax, (%ebx)
	xor %eax, %eax
	NEXT				
L_bl:
	movl GlobalSp, %ebx
	movl $32, (%ebx)
	subl $WSIZE, %ebx
	movl %ebx, GlobalSp
	movl GlobalTp, %ebx
	movb $OP_IVAL, (%ebx)
	decl GlobalTp	
	NEXT
L_ret:
	movl vmEntryRp, %eax		# Return Stack Ptr on entry to VM
	movl GlobalRp, %ecx
	cmpl %eax, %ecx
	jl ret1
        movl $OP_RET, %eax             # exhausted the return stack so exit vm
        ret
ret1:
	addl $WSIZE, %ecx
        movl %ecx, GlobalRp
        incl GlobalRtp	
	movl GlobalRtp, %ebx
	movb (%ebx), %al
        cmpb $OP_ADDR, %al
        jz ret2
        movl $E_RET_STK_CORRUPT, %eax   # indicate return stack corrupted
        jmp retexit
ret2:   movl (%ecx), %eax
	movl %eax, GlobalIp		# reset the instruction ptr
        xor %eax, %eax
retexit:
        ret
L_tick:
	movl GlobalSp, %ebx
	movl $32, (%ebx)
	movl $WSIZE, %eax
	subl %eax, GlobalSp
	decl GlobalTp
	call CPP_word
	call CPP_find
	DROP
	ret
L_tobody:
	movl GlobalSp, %ebx
	addl $WSIZE, %ebx
	movl (%ebx), %ecx	# code address
	incl %ecx		# the data address is offset by one
	movl (%ecx), %ecx
	movl %ecx, (%ebx)
	ret
#
# For precision delays, use MS instead of USLEEP
# Use USLEEP when task can be put to sleep and reawakened by OS
#
L_usleep:
	movl $WSIZE, %eax
	addl %eax, GlobalSp
	incl GlobalTp
	movl GlobalSp, %ebx
	movl (%ebx), %eax
	pushl %eax
	call usleep
	popl %eax
	xor %eax, %eax
	ret
L_ms:
	movl GlobalSp, %ebx
	movl WSIZE(%ebx), %eax
	imull $1000, %eax
	movl %eax, WSIZE(%ebx)
	call C_usec
	ret
L_fill:
	SWAP
	movl $WSIZE, %eax
	addl %eax, GlobalSp
	incl GlobalTp
	movl GlobalSp, %ebx
	movl (%ebx), %ebx
	pushl %ebx
	addl %eax, GlobalSp
	incl GlobalTp
	movl GlobalSp, %ebx
	movl (%ebx), %ebx
	pushl %ebx
	addl %eax, GlobalSp
	incl GlobalTp
	movl GlobalTp, %ebx
	movb (%ebx), %al
	cmpb $OP_ADDR, %al
	jz fill2
	popl %ebx
	popl %ebx
	movl $E_NOT_ADDR, %eax
	jmp fillexit
fill2:	movl GlobalSp, %ebx
	movl (%ebx), %ebx
	pushl %ebx
	call memset
	addl $12, %esp
	xor %eax, %eax
fillexit:	
	ret
L_erase:
	movl GlobalSp, %ebx
	movl $0, (%ebx)
	subl $WSIZE, %ebx
	movl %ebx, GlobalSp
	decl GlobalTp
	call L_fill
	ret	
L_cmove:
	movl $WSIZE, %eax
	addl %eax, GlobalSp
	incl GlobalTp
	movl GlobalSp, %ebx
	movl (%ebx), %ebx
	pushl %ebx
	SWAP
	movl $WSIZE, %eax
	addl %eax, GlobalSp
	incl GlobalTp
	movl GlobalTp, %ebx
	movb (%ebx), %al
	cmpb $OP_ADDR, %al
	jz cmove2
	popl %ebx
	movl $E_NOT_ADDR, %eax
	ret
cmove2:	movl GlobalSp, %ebx
	movl (%ebx), %ebx
	pushl %ebx
	movl $WSIZE, %eax
	addl %eax, GlobalSp
	incl GlobalTp
	movl GlobalTp, %ebx
	movb (%ebx), %al
	cmpb $OP_ADDR, %al
	jz cmove3
	popl %ebx
	popl %ebx
	movl $E_NOT_ADDR, %eax
	ret
cmove3:	movl GlobalSp, %ebx
	movl (%ebx), %ebx
	pushl %ebx
	call memcpy
	addl $12, %esp
	xor %eax, %eax				
	ret		
L_cmovefrom:
	movl $WSIZE, %eax
	addl %eax, GlobalSp
	incl GlobalTp
	movl GlobalSp, %ebx
	movl (%ebx), %ecx	# load count register
	addl %eax, GlobalSp
	incl GlobalTp
	movl GlobalTp, %ebx
	movb (%ebx), %al
	cmpb $OP_ADDR, %al
	jz cmovefrom2
	movl $E_NOT_ADDR, %eax						
	ret
cmovefrom2:
	movl GlobalSp, %ebx
	movl (%ebx), %ebx
	movl %ecx, %eax
	decl %eax
	addl %eax, %ebx
	movl %ebx, %edx		# dest addr in %edx
	movl $WSIZE, %eax
	addl %eax, GlobalSp
	incl GlobalTp
	movl GlobalTp, %ebx
	movb (%ebx), %al
	cmpb $OP_ADDR, %al
	jz cmovefrom3
	movl $E_NOT_ADDR, %eax
	ret
cmovefrom3:
	movl GlobalSp, %ebx
	movl (%ebx), %ebx
	movl %ecx, %eax
	cmpl $0, %eax
	jnz cmovefrom4
	ret
cmovefrom4:	
	decl %eax
	addl %eax, %ebx		# src addr in %ebx
cmovefromloop:	
	movb (%ebx), %al
	decl %ebx
	xchgl %ebx, %edx
	movb %al, (%ebx)
	decl %ebx
	xchgl %ebx, %edx
	loop cmovefromloop	
	xor %eax, %eax
	ret
L_call:	
	movl $WSIZE, %eax
	addl %eax, GlobalSp
	xor %eax, %eax
	incl GlobalTp
	movl GlobalSp, %ebx
	call *(%ebx)
	ret		
L_push:
	_PUSH
        NEXT
L_pop:
	_POP
	NEXT
L_twopush:
	movl GlobalSp, %ebx
	addl $WSIZE, %ebx
	movl (%ebx), %edx
	addl $WSIZE, %ebx
	movl (%ebx), %eax
	movl %ebx, GlobalSp
	movl GlobalRp, %ebx
	movl %eax, (%ebx)
	subl $WSIZE, %ebx
	movl %edx, (%ebx)
	subl $WSIZE, %ebx
	movl %ebx, GlobalRp
	movl GlobalTp, %ebx
	incl %ebx
	movw (%ebx), %ax
	incl %ebx
	movl %ebx, GlobalTp
	movl GlobalRtp, %ebx
	decl %ebx
	movw %ax, (%ebx)
	decl %ebx
	movl %ebx, GlobalRtp
	xor %eax, %eax
	NEXT
L_twopop:
	movl GlobalRp, %ebx
	addl $WSIZE, %ebx
	movl (%ebx), %edx
	addl $WSIZE, %ebx
	movl (%ebx), %eax
	movl %ebx, GlobalRp
	movl GlobalSp, %ebx
	movl %eax, (%ebx)
	subl $WSIZE, %ebx
	movl %edx, (%ebx)
	subl $WSIZE, %ebx
	movl %ebx, GlobalSp
	movl GlobalRtp, %ebx
	incl %ebx
	movw (%ebx), %ax
	incl %ebx
	movl %ebx, GlobalRtp
	movl GlobalTp, %ebx
	decl %ebx
	movw %ax, (%ebx)
	decl %ebx
	movl %ebx, GlobalTp
	xor %eax, %eax				
	NEXT
L_puship:
        movl GlobalIp, %eax
        movl GlobalRp, %ebx
        mov %eax, (%ebx)
	movl $WSIZE, %eax
        subl %eax, GlobalRp
        movl GlobalRtp, %ebx
	movb $OP_ADDR, %al
        movb %al, (%ebx)
	decl GlobalRtp
        xor %eax, %eax
        NEXT
L_execute:	
        movl GlobalIp, %ecx
        movl GlobalRp, %ebx
        movl %ecx, (%ebx)
	movl $WSIZE, %eax 
        subl %eax, %ebx 
	movl %ebx, GlobalRp
        movl GlobalRtp, %ebx
	movb $OP_ADDR, (%ebx)
	decl %ebx
        movl %ebx, GlobalRtp
	movl GlobalSp, %ebx
        addl %eax, %ebx
	movl %ebx, GlobalSp
        movl (%ebx), %eax
	decl %eax
	movl %eax, GlobalIp
	incl GlobalTp
        xor %eax, %eax
        NEXT
L_definition:
        movl GlobalIp, %ebx
	movl $WSIZE, %eax
	incl %ebx
	movl (%ebx), %ecx # address to execute
	addl $3, %ebx
	movl %ebx, %edx
	movl GlobalRp, %ebx
	movl %edx, (%ebx)
	subl %eax, %ebx
	movl %ebx, GlobalRp
	movl GlobalRtp, %ebx
	movb $OP_ADDR, (%ebx)
	decl %ebx
	movl %ebx, GlobalRtp
	decl %ecx
	movl %ecx, GlobalIp
        xor %eax, %eax	
	NEXT
L_rfetch:
        movl GlobalRp, %ebx
        addl $WSIZE, %ebx
        movl (%ebx), %eax
        movl GlobalSp, %ebx
        movl %eax, (%ebx)
	movl $WSIZE, %eax
        subl %eax, GlobalSp
        movl GlobalRtp, %ebx
        incl %ebx
        movb (%ebx), %al
        movl GlobalTp, %ebx
        movb %al, (%ebx)
        decl GlobalTp
        xor %eax, %eax
	NEXT
L_tworfetch:
	movl GlobalRp, %ebx
	addl $WSIZE, %ebx
	movl (%ebx), %edx
	addl $WSIZE, %ebx
	movl (%ebx), %eax
	movl GlobalSp, %ebx
	movl %eax, (%ebx)
	subl $WSIZE, %ebx
	movl %edx, (%ebx)
	subl $WSIZE, %ebx
	movl %ebx, GlobalSp
	movl GlobalRtp, %ebx
	incl %ebx
	movw (%ebx), %ax
	incl %ebx
	movl GlobalTp, %ebx
	decl %ebx
	movw %ax, (%ebx)
	decl %ebx
	movl %ebx, GlobalTp
	xor %eax, %eax				
	NEXT
L_rpfetch:
	movl GlobalRp, %eax
	addl $WSIZE, %eax
	movl GlobalSp, %ebx
	movl %eax, (%ebx)
	subl $WSIZE, %ebx
	movl %ebx, GlobalSp
	movl GlobalTp, %ebx
	movb $OP_ADDR, (%ebx)
	decl %ebx
	movl %ebx, GlobalTp
	xor %eax, %eax
	NEXT
L_spfetch:
	movl GlobalSp, %eax
	movl %eax, %ebx
	addl $WSIZE, %eax
	movl %eax, (%ebx)
	subl $WSIZE, %ebx
	movl %ebx, GlobalSp
	movl GlobalTp, %ebx
	movb $OP_ADDR, (%ebx)
	decl %ebx
	movl %ebx, GlobalTp
	xor %eax, %eax 
	NEXT
L_i:
        movl GlobalRtp, %ebx
        movb 3(%ebx), %al
        movl GlobalTp, %ebx
	movb %al, (%ebx)
	decl %ebx
	movl %ebx, GlobalTp
        movl GlobalRp, %ebx
        movl 3*WSIZE(%ebx), %eax
        movl GlobalSp, %ebx
        movl %eax, (%ebx)
	movl $WSIZE, %eax
        subl %eax, %ebx 
	movl %ebx, GlobalSp
        xor %eax, %eax
        NEXT
L_j:
        movl GlobalRtp, %ebx
        movb 6(%ebx), %al
	movl GlobalTp, %ebx
	movb %al, (%ebx)
	decl %ebx
	movl %ebx, GlobalTp
        movl GlobalRp, %ebx
        movl 6*WSIZE(%ebx), %eax
        movl GlobalSp, %ebx
        movl %eax, (%ebx)
	movl $WSIZE, %eax
        subl %eax, %ebx
	movl %ebx, GlobalSp
        xor %eax, %eax
        NEXT
L_loop:
        movl GlobalRtp, %ebx
        incl %ebx
        movb (%ebx), %al
        cmpb $OP_ADDR, %al
        jnz loopbad
        movl GlobalRp, %ebx	
	movl $WSIZE, %eax
	addl %eax, %ebx
	movl (%ebx), %edx
	addl %eax, %ebx
	movl (%ebx), %ecx
	addl %eax, %ebx
        movl (%ebx), %eax
        incl %eax
	cmpl %ecx, %eax	
        jz L_unloop
loop1:	
        movl %eax, (%ebx)	# set loop counter to next value
	movl %edx, GlobalIp	# set instruction ptr to start of loop
        xor %eax, %eax
        NEXT
L_unloop:  
	movl $WSIZE, %eax
	shll $1, %eax
	addl $WSIZE, %eax
#	imul $3, %eax
	addl %eax, GlobalRp  # terminal count reached, discard top 3 items
	movl $3, %eax
        addl %eax, GlobalRtp
        xor %eax, %eax
        NEXT
loopbad:
        movl $E_RET_STK_CORRUPT, %eax
        ret	
L_plusloop:
	movl GlobalRtp, %ebx
        incl %ebx
        movb (%ebx), %al
        cmpb $OP_ADDR, %al
        jnz loopbad
	movl $WSIZE, %eax
	movl GlobalSp, %ebx
	addl %eax, %ebx
	movl (%ebx), %ebp	# get loop increment 
	movl %ebx, GlobalSp
	incl GlobalTp		
        movl GlobalRp, %ebx
	addl %eax, %ebx		# get ip and save in edx
	movl (%ebx), %edx
	addl %eax, %ebx
	movl (%ebx), %ecx	# get terminal count in ecx
	addl %eax, %ebx
	movl (%ebx), %eax	# get current loop count 
	addl %ebp, %eax		# increment loop count
	cmpl $0, %ebp
	jl plusloop1
	cmpl %ecx, %eax
	jge L_unloop				
	movl %eax, (%ebx)	# set loop counter to incremented value
        movl %edx, GlobalIp     # set instruction ptr to start of loop
        xor %eax, %eax
        NEXT
plusloop1:
	cmpl %ecx, %eax
	jl L_unloop
	movl %eax, (%ebx)
	movl %edx, GlobalIp
	xor %eax, %eax
	NEXT
L_jz:
	movl $WSIZE, %eax
        movl GlobalSp, %ebx
	addl %eax, %ebx
	movl %ebx, GlobalSp
        incl GlobalTp
        movl (%ebx), %eax
        cmpl $0, %eax
        jz jz1
	movl $4, %eax
        addl %eax, GlobalIp       # do not jump
	xor %eax, %eax
        NEXT
jz1:    movl GlobalIp, %ebx
        incl %ebx
        movl (%ebx), %eax          # get the relative jump count
        decl %eax
        addl %eax, GlobalIp
	xor %eax, %eax
        NEXT
L_jnz:				# not implemented
	ret
L_jmp:
        movl GlobalIp, %ebx
        incl %ebx
        movl (%ebx), %eax          # get the relative jump count
        addl %eax, %ebx
        subl $2, %ebx
        movl %ebx, GlobalIp      # set instruction ptr
	xor %eax, %eax
        NEXT
L_count:
	movl GlobalTp, %ebx
	movb 1(%ebx), %al
	cmpb $OP_ADDR, %al
	jnz counterror
	movb $OP_IVAL, (%ebx)
	decl GlobalTp
	movl GlobalSp, %ebx
	movl WSIZE(%ebx), %ebx
	xor %eax, %eax
	movb (%ebx), %al
	movl GlobalSp, %ebx
	incl WSIZE(%ebx)
	movl %eax, (%ebx)
	movl $WSIZE, %eax
	subl %eax, GlobalSp
	xor %eax, %eax
	ret
counterror:
	movl $E_NOT_ADDR, %eax
	ret
L_ival:
        movl GlobalIp, %ebx
        incl %ebx
        movl (%ebx), %eax
	addl $WSIZE-1, %ebx
	movl %ebx, GlobalIp
	movl GlobalSp, %ebx
	movl %eax, (%ebx)
	subl $WSIZE, %ebx
	movl %ebx, GlobalSp
	movl GlobalTp, %ebx
	movb $OP_IVAL, (%ebx)
	decl %ebx
	movl %ebx, GlobalTp
	xor %eax, %eax
	NEXT
L_addr:
	movl GlobalIp, %ebx
	incl %ebx
	movl (%ebx), %eax
	addl $WSIZE-1, %ebx
	movl %ebx, GlobalIp
	movl GlobalSp, %ebx
	movl %eax, (%ebx)
	subl $WSIZE, %ebx
	movl %ebx, GlobalSp
	movl GlobalTp, %ebx
	movb $OP_ADDR, (%ebx)
	decl %ebx
	movl %ebx, GlobalTp
	xor %eax, %eax
	NEXT
L_fval:
        movl GlobalIp, %ebx
        incl %ebx
        movl GlobalSp, %ebp
        subl $WSIZE, %ebp
        movl (%ebx), %eax
	movl %eax, (%ebp)
	movl WSIZE(%ebx), %eax
	movl %eax, WSIZE(%ebp)
	subl $WSIZE, %ebp
	movl %ebp, GlobalSp
	addl $2*WSIZE-1, %ebx
	movl %ebx, GlobalIp
	movl GlobalTp, %ebx
	movb $OP_IVAL, (%ebx)
	decl %ebx
	movb $OP_IVAL, (%ebx)
	decl %ebx
	movl %ebx, GlobalTp
	xor %eax, %eax
	NEXT
L_and:
	_AND
	NEXT
L_or:
	_OR
	NEXT
L_not:
	_NOT
	NEXT
L_xor:
	_XOR
	NEXT
L_lshift:
	movl $WSIZE, %eax
	addl %eax, GlobalSp
	incl GlobalTp
	movl GlobalSp, %ebx
	movl (%ebx), %ecx
	movl WSIZE(%ebx), %eax
	shll %cl, %eax
	movl %eax, WSIZE(%ebx)
	xor %eax, %eax
	NEXT
L_rshift:
	movl $WSIZE, %eax
	addl %eax, GlobalSp
	incl GlobalTp
	movl GlobalSp, %ebx
	movl (%ebx), %ecx
	movl WSIZE(%ebx), %eax
	shrl %cl, %eax
	movl %eax, WSIZE(%ebx)
	xor %eax, %eax
	NEXT
L_eq:
	movl $WSIZE, %eax
	addl %eax, GlobalSp
	incl GlobalTp
	movl GlobalSp, %ebx
	movl (%ebx), %eax
	movl WSIZE(%ebx), %ebx
	cmpl %ebx, %eax
	jne eq2
eq1:	movl GlobalSp, %ebx
	movl $-1, WSIZE(%ebx)
	jmp eq3
eq2:    movl GlobalSp, %ebx
	movl $0, WSIZE(%ebx)
eq3:    movl GlobalTp, %ebx
	movb $OP_IVAL, 1(%ebx)
	xor %eax, %eax
	ret
L_ne:
	call L_eq
	_NOT
	ret
L_ult:
	movl $WSIZE, %eax
	addl %eax, GlobalSp
	incl GlobalTp
	movl GlobalSp, %ebx
	movl (%ebx), %eax
	movl WSIZE(%ebx), %ebx
	cmpl %eax, %ebx
	jae eq2
	jmp eq1
	ret
L_ugt:
	movl $WSIZE, %eax
	addl %eax, GlobalSp
	incl GlobalTp
	movl GlobalSp, %ebx
	movl (%ebx), %eax
	movl WSIZE(%ebx), %ebx
	cmpl %eax, %ebx
	jbe eq2
	jmp eq1	
	ret	
L_lt:
	movl GlobalTp, %ebx
	incl %ebx
	movb $OP_IVAL, (%ebx)
	movl %ebx, GlobalTp
	movl GlobalSp, %ebx
	movl $WSIZE, %eax
	addl %eax, %ebx
	movl %ebx, GlobalSp
	movl (%ebx), %ecx
	addl %eax, %ebx
	movl (%ebx), %eax
	cmpl %ecx, %eax
	jge lt1
	movl $-1, (%ebx)
	xor %eax, %eax
	ret
lt1:
	movl $0, (%ebx)
	xor %eax, %eax
	ret	
L_gt:
	movl GlobalTp, %ebx
	incl %ebx
	movb $OP_IVAL, (%ebx)
	movl %ebx, GlobalTp
	movl GlobalSp, %ebx
	movl $WSIZE, %eax
	addl %eax, %ebx
	movl %ebx, GlobalSp
	movl (%ebx), %ecx
	addl %eax, %ebx
	movl (%ebx), %eax
	cmpl %ecx, %eax
	jle gt1
	movl $-1, (%ebx)
	xor %eax, %eax
	ret
gt1:
	movl $0, (%ebx)
	xor %eax, %eax
	ret	
L_le:
	movl $WSIZE, %eax
	addl %eax, GlobalSp
	incl GlobalTp
	movl GlobalSp, %ebx
	movl (%ebx), %eax
	movl WSIZE(%ebx), %ebx
	cmpl %eax, %ebx
	jg eq2
	jmp eq1
L_ge:
	movl $WSIZE, %eax
	addl %eax, GlobalSp
	incl GlobalTp
	movl GlobalSp, %ebx
	movl (%ebx), %eax
	movl WSIZE(%ebx), %ebx
	cmpl %eax, %ebx
	jl eq2
	jmp eq1
L_zerolt:
	movl GlobalSp, %ebx
	movl WSIZE(%ebx), %eax
	cmpl $0, %eax
	jl zerolt2
zerolt1:
	movl $0, WSIZE(%ebx)
	jmp zeroltexit
zerolt2:
	movl $-1, WSIZE(%ebx)
zeroltexit:
	movl GlobalTp, %ebx
	movb $OP_IVAL, 1(%ebx)
	xor %eax, %eax
	ret
L_zeroeq:
	movl GlobalSp, %ebx
	movl WSIZE(%ebx), %eax
	cmpl $0, %eax
	je zerolt2
	jmp zerolt1
L_zerone:
	movl GlobalSp, %ebx
	movl WSIZE(%ebx), %eax
	cmpl $0, %eax
	je zerolt1
	jmp zerolt2
L_zerogt:
	movl GlobalSp, %ebx
	movl WSIZE(%ebx), %eax
	cmpl $0, %eax
	jg zerolt2
	jmp zerolt1
L_deq:
	movl GlobalTp, %ebx
	addl $4, %ebx
	movb $OP_IVAL, (%ebx)
	decl %ebx
	movl %ebx, GlobalTp
	movl GlobalSp, %ebx
	addl $WSIZE, %ebx
	movl (%ebx), %edx
	addl $WSIZE, %ebx
	movl (%ebx), %ecx
	addl $WSIZE, %ebx
	movl %ebx, GlobalSp
	movl (%ebx), %eax
	subl %edx, %eax
	addl $WSIZE, %ebx
	movl (%ebx), %edx
	subl %ecx, %edx
	orl %edx, %eax
	cmpl $0, %eax
	jz deq1
	movl $0, (%ebx)
	xor %eax, %eax
	ret
deq1:	movl $-1, (%ebx)
	xor %eax, %eax	
	ret
L_dzeroeq:
	movl GlobalTp, %ebx
	addl $2, %ebx
	movb $OP_IVAL, (%ebx)
	decl %ebx
	movl %ebx, GlobalTp
	movl GlobalSp, %ebx
	addl $WSIZE, %ebx
	movl %ebx, GlobalSp
	movl (%ebx), %eax
	addl $WSIZE, %ebx
	orl (%ebx), %eax
	cmpl $0, %eax
	jz deq1
	movl $0, (%ebx)
	xor %eax, %eax
	ret
L_dlt:
	DMINUS
	movl GlobalTp, %ebx
	incl %ebx
	movl %ebx, GlobalTp
	incl %ebx
	movb $OP_IVAL, (%ebx)
	movl GlobalSp, %ebx
	addl $WSIZE, %ebx
	movl %ebx, GlobalSp
	movl (%ebx), %eax
	addl $WSIZE, %ebx
	cmpl $0, %eax
	jl deq1
	movl $0, (%ebx)
	xor %eax, %eax	
	ret
L_dult:	
	ret 	
L_querydup:
	movl GlobalSp, %ebx
	movl WSIZE(%ebx), %eax
	cmpl $0, %eax
	je L_querydupexit
	movl %eax, (%ebx)
	movl GlobalTp, %ebx
	movb 1(%ebx), %al
	movb %al, (%ebx)
	decl GlobalTp
	movl $WSIZE, %eax
	subl %eax, GlobalSp
	xor %eax, %eax
L_querydupexit:
	ret	
L_drop:
	DROP
        NEXT
L_dup:
	DUP
        NEXT
L_swap:
	SWAP
        NEXT
L_over:
	OVER
        NEXT
L_rot:
	movl GlobalSp, %ebx
	movl $WSIZE, %eax
	addl %eax, %ebx
	movl %ebx, %ebp
	addl %eax, %ebx
	addl %eax, %ebx
	movl (%ebx), %ecx
	movl (%ebp), %edx
	movl %ecx, (%ebp)
	addl %eax, %ebp
	movl (%ebp), %ecx
	movl %edx, (%ebp)
	movl %ecx, (%ebx)
        # call L_swap
        # movl GlobalSp, %ebx
        # addl $WSIZE, %ebx
        # movl 2*WSIZE(%ebx), %eax
        # xchgl (%ebx), %eax
        # xchgl 2*WSIZE(%ebx), %eax
        movl GlobalTp, %ebx
        incl %ebx
	movl %ebx, %ebp
	movw (%ebx), %cx
	addl $2, %ebx
	movb (%ebx), %al
	movb %al, (%ebp)
	incl %ebp
	movw %cx, (%ebp)
        # movb  2(%ebx), %al
        # xchgb (%ebx), %al
        # xchgb 2(%ebx), %al
	xor %eax, %eax
	NEXT
L_minusrot:
	movl GlobalSp, %ebx
	movl WSIZE(%ebx), %eax
	movl %eax, (%ebx)
	addl $WSIZE, %ebx
	movl WSIZE(%ebx), %eax
	movl %eax, (%ebx)
	addl $WSIZE, %ebx
	movl WSIZE(%ebx), %eax
	movl %eax, (%ebx)
	movl -2*WSIZE(%ebx), %eax
	movl %eax, WSIZE(%ebx)
	movl GlobalTp, %ebx
	movb 1(%ebx), %al
	movb %al, (%ebx)
	incl %ebx
	movw 1(%ebx), %ax
	movw %ax, (%ebx)
	movb -1(%ebx), %al
	movb %al, 2(%ebx)
	xor %eax, %eax
	NEXT
L_nip:
        SWAP
	movl $WSIZE, %eax
        addl %eax, GlobalSp
        incl GlobalTp
	xor %eax, %eax
        NEXT
L_tuck:
        SWAP
        OVER
        NEXT
L_pick:
	movl GlobalSp, %ebx
	movl WSIZE(%ebx), %eax
	addl $2, %eax
	imul $WSIZE, %eax
	addl %eax, %ebx
	movl (%ebx), %eax
	movl GlobalSp, %ebx
	movl %eax, (%ebx)
	movl WSIZE(%ebx), %eax
	addl $2, %eax
	movl GlobalTp, %ebx
	addl %eax, %ebx
	movb (%ebx), %al
	movl GlobalTp, %ebx
	movb %al, 1(%ebx)
	movl GlobalSp, %ebx
	movl (%ebx), %eax
	movl %eax, WSIZE(%ebx)
	xor %eax, %eax
	NEXT
L_roll:
	movl $WSIZE, %eax
	addl %eax, GlobalSp
	incl GlobalTp
	movl GlobalSp, %ebx 
	movl (%ebx), %eax
	incl %eax
	pushl %eax
	pushl %eax
	pushl %eax
	pushl %ebx
	imul $WSIZE, %eax
	addl %eax, %ebx		# addr of item to roll
	movl (%ebx), %eax
	popl %ebx
	movl %eax, (%ebx)
	popl %eax		# number of cells to copy
	movl %eax, %ecx
	imul $WSIZE, %eax
	addl %eax, %ebx
	movl %ebx, %edx		# dest addr
	subl $WSIZE, %ebx	# src addr
rollloop:
	movl (%ebx), %eax
	sub $WSIZE, %ebx
	xchgl %ebx, %edx
	movl %eax, (%ebx)
	sub $WSIZE, %ebx
	xchgl %ebx, %edx
	loop rollloop

	popl %eax		# now we have to roll the typestack
	mov GlobalTp, %ebx	
	addl %eax, %ebx
	movb (%ebx), %al
	movl GlobalTp, %ebx
	movb %al, (%ebx)
	popl %eax
	movl %eax, %ecx
	addl %eax, %ebx
	movl %ebx, %edx
	decl %ebx
rolltloop:
	movb (%ebx), %al
	decl %ebx
	xchgl %ebx, %edx
	movb %al, (%ebx)
	decl %ebx
	xchgl %ebx, %edx
	loop rolltloop
	xor %eax, %eax
	ret
L_depth:
	movl GlobalSp, %ebx
	movl BottomOfStack, %eax
	subl %ebx, %eax
	movl $WSIZE, (%ebx)
	movl $0, %edx
	idivl (%ebx)
	movl %eax, (%ebx)
	movl $WSIZE, %eax
	subl %eax, GlobalSp
	movl GlobalTp, %ebx
	movb $OP_IVAL, (%ebx)
	decl GlobalTp
	xor %eax, %eax
        ret
L_2drop:
	movl $WSIZE, %eax
	addl %eax, %eax
	addl %eax, GlobalSp
        addl $2, GlobalTp
	xor %eax, %eax
        NEXT
L_2dup:
	movl GlobalSp, %ebx
	movl %ebx, %ecx
	addl $WSIZE, %ebx
	movl (%ebx), %edx
	addl $WSIZE, %ebx
	movl (%ebx), %eax
	movl %ecx, %ebx
	movl %eax, (%ebx)
	subl $WSIZE, %ebx
	movl %edx, (%ebx)
	subl $WSIZE, %ebx
	movl %ebx, GlobalSp
	movl GlobalTp, %ebx
	incl %ebx
	movw (%ebx), %ax
	subl $2, %ebx
	movw %ax, (%ebx)
	decl %ebx
	movl %ebx, GlobalTp
	xor %eax, %eax
        NEXT
L_2swap:
	movl GlobalSp, %ebx
	addl $WSIZE, %ebx
	movl (%ebx), %edx
	addl $WSIZE, %ebx
	movl (%ebx), %eax
	addl $WSIZE, %ebx
	xchgl %edx, (%ebx)
	addl $WSIZE, %ebx
	xchgl %eax, (%ebx)
	subl $2*WSIZE, %ebx
	movl %eax, (%ebx)
	subl $WSIZE, %ebx
	movl %edx, (%ebx)
	movl GlobalTp, %ebx
	incl %ebx
	movw (%ebx), %ax
	addl $2, %ebx
	xchgw %ax, (%ebx)
	subl $2, %ebx
	movw %ax, (%ebx)
	xor %eax, %eax	
        NEXT
L_2over:
	FOVER
        NEXT
L_2rot:
	movl GlobalSp, %ebx
	addl $WSIZE, %ebx
	movl %ebx, %ecx
	movl (%ebx), %edx
	addl $WSIZE, %ebx
	movl (%ebx), %eax
	addl $WSIZE, %ebx
	xchgl %edx, (%ebx)
	addl $WSIZE, %ebx
	xchgl %eax, (%ebx)
	addl $WSIZE, %ebx
	xchgl %edx, (%ebx)
	addl $WSIZE, %ebx
	xchgl %eax, (%ebx)
	movl %ecx, %ebx
	movl %edx, (%ebx)
	addl $WSIZE, %ebx
	movl %eax, (%ebx)
	movl GlobalTp, %ebx
	incl %ebx
	movl %ebx, %ecx
	movw (%ebx), %ax
	addl $2, %ebx
	xchgw %ax, (%ebx)
	addl $2, %ebx
	xchgw %ax, (%ebx)
	movl %ecx, %ebx
	movw %ax, (%ebx)
	xor %eax, %eax
        NEXT
L_question:
	call L_fetch
	cmpl $0, %eax
	jnz questionexit
	call CPP_dot
questionexit:	
	ret	
L_fetch:
	movl GlobalSp, %edx
	movl GlobalTp, %ebx
	incl %ebx
        movb (%ebx), %al
        cmpb $OP_ADDR, %al
        jnz fetcherror
        movb $OP_IVAL, (%ebx)	
	addl $WSIZE, %edx
        movl (%edx), %ebx	
        movl (%ebx), %eax
	movl %eax, (%edx)
	xor %eax, %eax
	ret
fetcherror:
        movl $E_NOT_ADDR, %eax
        ret
L_store:
        movl GlobalTp, %ebx
	incl %ebx
	movl %ebx, %ebp
        movb (%ebx), %al
        cmpb $OP_ADDR, %al
        jnz fetcherror
	movl $WSIZE, %eax
	movl GlobalSp, %ebx
        addl %eax, %ebx
        movl (%ebx), %ecx	# address to store to in ecx
	addl %eax, %ebx
	movl (%ebx), %edx	# value to store in edx
	movl %ebx, GlobalSp
	movl %edx, (%ecx)
	incl %ebp
	movl %ebp, GlobalTp
	xor %eax, %eax
	NEXT
L_afetch:
	movl GlobalSp, %edx
	movl GlobalTp, %ebx
	incl %ebx
        movb (%ebx), %al
        cmpb $OP_ADDR, %al
        jnz fetcherror
        movb $OP_ADDR, (%ebx)
	addl $WSIZE, %edx
	movl (%edx), %ebx
	movl (%ebx), %eax
	movl %eax, (%edx)
	xor %eax, %eax
	NEXT
L_cfetch:
	movl GlobalTp, %ebx
	incl %ebx
	movb (%ebx), %al
	cmpb $OP_ADDR, %al
	jnz fetcherror
	movb $OP_IVAL, (%ebx)
	xor %eax, %eax
	movl GlobalSp, %ebx
	addl $WSIZE, %ebx
	movl (%ebx), %ecx
	movb (%ecx), %al
	movl %eax, (%ebx)
	xor %eax, %eax
        NEXT
L_cstore:
	movl GlobalTp, %edx
	incl %edx
	movb (%edx), %al
	cmpb $OP_ADDR, %al
	jnz fetcherror
	movl GlobalSp, %ebx
	addl $WSIZE, %ebx
	movl (%ebx), %ecx	# address to store
	addl $WSIZE, %ebx
	movl (%ebx), %eax	# value to store
	movb %al, (%ecx)
	movl %ebx, GlobalSp
	incl %edx
	movl %edx, GlobalTp
	xor %eax, %eax
	NEXT	
L_wfetch:
	movl GlobalTp, %ebx
	movb 1(%ebx), %al
	cmpb $OP_ADDR, %al
	jnz fetcherror
	movb $OP_IVAL, 1(%ebx)
	movl GlobalSp, %ebx
	movl WSIZE(%ebx), %ebx
	movw (%ebx), %ax
	cwde
	movl GlobalSp, %ebx
	movl %eax, WSIZE(%ebx)
	xor %eax, %eax
        NEXT
L_wstore:
	movl $WSIZE, %eax
	addl %eax, GlobalSp
	incl GlobalTp
	movl GlobalTp, %ebx
	movb (%ebx), %al
	cmpb $OP_ADDR, %al
	jnz fetcherror
	movl GlobalSp, %ebx
	movl (%ebx), %eax
	pushl %eax
	addl $WSIZE, %ebx
	movl (%ebx), %eax
	pop %ebx
	movw %ax, (%ebx)
	movl $WSIZE, %eax
	addl %eax, GlobalSp
	incl GlobalTp
	xor %eax, %eax
        NEXT
L_sffetch:
	movl $WSIZE, %eax
        addl %eax, GlobalSp
        incl GlobalTp
        movl GlobalTp, %ebx
        movb (%ebx), %al
        cmpb $OP_ADDR, %al
        jnz fetcherror
        movb $OP_IVAL, (%ebx)
        decl %ebx
        movb $OP_IVAL, (%ebx)
        decl GlobalTp
	decl GlobalTp
        movl GlobalSp, %ebx
        movl (%ebx), %ebx
        flds (%ebx)
	movl $WSIZE, %eax
        subl %eax, GlobalSp
        movl GlobalSp, %ebx
        fstpl (%ebx)
        subl %eax, GlobalSp
	xor %eax, %eax
        NEXT
L_sfstore:
	movl $WSIZE, %eax
        addl %eax, GlobalSp
        incl GlobalTp
        movl GlobalTp, %ebx
        movb (%ebx), %al
        cmpb $OP_ADDR, %al
        jnz fetcherror
        movl GlobalSp, %ebx
        addl $WSIZE, %ebx
        fldl (%ebx)              # load the f number into NDP
        subl $WSIZE, %ebx
        movl (%ebx), %ebx          # load the dest address
        fstps (%ebx)             # store as single precision float
	movl $WSIZE, %eax
	sall $1, %eax
        addl %eax, GlobalSp
	incl GlobalTp
	incl GlobalTp
	xor %eax, %eax
        NEXT
L_dffetch:	
        movl GlobalTp, %ebx
	incl %ebx
        movb (%ebx), %al
        cmpb $OP_ADDR, %al
        jnz fetcherror
	movb $OP_IVAL, (%ebx)
	decl %ebx
	movb $OP_IVAL, (%ebx)
	decl %ebx
	movl %ebx, GlobalTp 
	movl GlobalSp, %ebx
	movl %ebx, %edx
	addl $WSIZE, %ebx
	movl (%ebx), %ecx
	movl (%ecx), %eax
	movl %eax, (%edx)
	addl $WSIZE, %ecx
	movl (%ecx), %eax
	movl %eax, (%ebx)
	subl $WSIZE, %edx
	movl %edx, GlobalSp
	xor %eax, %eax
	NEXT
L_dfstore:
        movl GlobalTp, %ebx
	incl %ebx
        movb (%ebx), %al
        cmpb $OP_ADDR, %al
        jnz fetcherror
	addl $2, %ebx
	movl %ebx, GlobalTp
	movl GlobalSp, %ebx
	movl $WSIZE, %edx
	addl %edx, %ebx
	movl %ebx, %eax
	movl (%ebx), %ebx  # address to store
	addl %edx, %eax
	movl (%eax), %ecx
	movl %ecx, (%ebx)
	addl %edx, %eax
	addl %edx, %ebx
	movl (%eax), %ecx
	movl %ecx, (%ebx)
	movl %eax, GlobalSp
	xor %eax, %eax
	NEXT
L_inc:
        movl GlobalSp, %ebx
        incl WSIZE(%ebx)
        NEXT
L_dec:
        movl GlobalSp, %ebx
        decl WSIZE(%ebx)
        NEXT
L_twoplus:
	movl GlobalSp, %ebx
	incl WSIZE(%ebx)
	incl WSIZE(%ebx)
	NEXT
L_twominus:
	movl GlobalSp, %ebx
	decl WSIZE(%ebx)
	decl WSIZE(%ebx)
	NEXT
L_abs:
	movl GlobalSp, %ebx
	addl $WSIZE, %ebx
	movl (%ebx), %eax
	cmpl $0, %eax
	jl abs1
	xor %eax, %eax
	ret
abs1:	negl %eax
	movl %eax, (%ebx)
	xor %eax, %eax
        ret
L_neg:
	movl GlobalSp, %ebx
	addl $WSIZE, %ebx
	movl (%ebx), %eax
	negl %eax
	movl %eax, (%ebx)
	xor %eax, %eax
        NEXT
L_max:
	movl $WSIZE, %eax
	addl %eax, GlobalSp
	movl GlobalSp, %ebx
	movl (%ebx), %eax
	movl WSIZE(%ebx), %ebx
	cmpl %eax, %ebx
	jl max1
	movl %ebx, %eax
	movl GlobalSp, %ebx
	movl %eax, WSIZE(%ebx)
	jmp maxexit
max1:
	movl GlobalSp, %ebx
	movl %eax, WSIZE(%ebx)
maxexit:
	incl GlobalTp
	xor %eax, %eax
        ret
L_min:
	movl $WSIZE, %eax
	addl %eax, GlobalSp
	movl GlobalSp, %ebx
	movl (%ebx), %eax
	movl WSIZE(%ebx), %ebx
	cmpl %eax, %ebx
	jg min1
	movl %ebx, %eax
	movl GlobalSp, %ebx
	movl %eax, WSIZE(%ebx)
	jmp minexit
min1:
	movl GlobalSp, %ebx
	movl %eax, WSIZE(%ebx)
minexit:
	incl GlobalTp
	xor %eax, %eax
        ret
L_dmax:
	ret
L_dmin:
	ret
L_twostar:
	movl GlobalSp, %ebx
	sall $1, WSIZE(%ebx)
	NEXT
L_twodiv:
	movl GlobalSp, %ebx
	sarl $1, WSIZE(%ebx)
	NEXT
L_add:
	movl $WSIZE, %eax
	movl GlobalSp, %ebx
	addl %eax, %ebx
	movl (%ebx), %eax
	addl %eax, WSIZE(%ebx)
	movl %ebx, GlobalSp
	movl GlobalTp, %ebx
	incl %ebx
	movl %ebx, GlobalTp
	movw (%ebx), %ax
	andb %ah, %al		# and the two types to preserve addr type
	incl %ebx
	movb %al, (%ebx)
        xor %eax, %eax
        NEXT
L_sub:
	movl $WSIZE, %eax
	movl GlobalSp, %ebx
	addl %eax, %ebx
	movl (%ebx), %eax
	subl %eax, WSIZE(%ebx)
	movl %ebx, GlobalSp
        incl GlobalTp		# result will have type of first operand
        xor %eax, %eax
        NEXT
L_mul:
	movl $WSIZE, %ecx
	movl GlobalSp, %ebx
	addl %ecx, %ebx
	movl %ebx, GlobalSp
	movl (%ebx), %eax
	addl %ecx, %ebx
	imull (%ebx)
	movl %eax, (%ebx)
	incl GlobalTp
	xor %eax, %eax
        NEXT
L_div:
	movl $WSIZE, %eax
        addl %eax, GlobalSp
        incl GlobalTp
        movl GlobalSp, %ebx
        movl (%ebx), %eax
        cmpl $0, %eax
        jnz div1
        movl $E_DIV_ZERO, %eax
        jmp divexit
div1:	
	addl $WSIZE, %ebx
        movl (%ebx), %eax
	cdq
        idivl -WSIZE(%ebx)
        movl %eax, (%ebx)
	xor %eax, %eax
divexit:
        ret
L_mod:
	call L_div
	movl %edx, (%ebx)
	NEXT
L_slashmod:
	call L_div
	subl $WSIZE, %ebx
	movl %edx, (%ebx)
	subl $WSIZE, %ebx
	movl %ebx, GlobalSp
	decl GlobalTp
	SWAP
	NEXT
L_starslash:
	STARSLASH	
	NEXT
L_starslashmod:
	STARSLASH
	movl %edx, (%ebx)
	subl $WSIZE, %ebx
	movl %ebx, GlobalSp
	decl GlobalTp
	SWAP
	ret
L_plusstore:
	movl GlobalTp, %ebx
	movb 1(%ebx), %al
	cmpb $OP_ADDR, %al
	jnz fetcherror
	movl GlobalSp, %ebx
	push %ebx
	push %ebx
	push %ebx
	movl WSIZE(%ebx), %ebx
	movl (%ebx), %eax
	pop %ebx
	movl 2*WSIZE(%ebx), %ebx
	addl %ebx, %eax
	pop %ebx
	movl WSIZE(%ebx), %ebx
	movl %eax, (%ebx)
	pop %ebx
	movl $WSIZE, %eax
	sall $1, %eax
	addl %eax, %ebx
	movl %ebx, GlobalSp
	incl GlobalTp
	incl GlobalTp
	xor %eax, %eax
	NEXT
L_dabs:
	movl GlobalSp, %ebx
	addl $WSIZE, %ebx
	movl (%ebx), %ecx
	movl %ecx, %eax
	cmpl $0, %eax
	jl dabs_go
	xor %eax, %eax
	ret
dabs_go:
	addl $WSIZE, %ebx
	movl (%ebx), %eax
	clc
	subl $1, %eax
	notl %eax
	movl %eax, (%ebx)
	movl %ecx, %eax
	sbbl $0, %eax
	notl %eax
	movl %eax, -WSIZE(%ebx)
	xor %eax, %eax
	ret
L_dnegate:
	DNEGATE
	NEXT	
L_dplus:
	DPLUS
	NEXT
L_dminus:
	DMINUS
	ret
L_umstar:
	movl $WSIZE, %eax
	movl GlobalSp, %ebx
	addl %eax, %ebx
	movl (%ebx), %ecx
	addl %eax, %ebx
	movl %ecx, %eax
	mull (%ebx)
	movl %eax, (%ebx)
	subl $WSIZE, %ebx
	movl %edx, (%ebx)
	xor %eax, %eax				
	ret
L_umslashmod:
	movl GlobalSp, %ebx
	movl $WSIZE, %eax
	addl %eax, %ebx
	movl %ebx, GlobalSp
	movl (%ebx), %ecx
	addl %eax, %ebx
	movl $0, %edx
	movl (%ebx), %eax
	divl %ecx
	cmpl $0, %eax
	jne umslashmod_ovflow
	movl (%ebx), %edx
	addl $WSIZE, %ebx
	movl (%ebx), %eax
	divl %ecx
	jmp umslashmod_exit
umslashmod_ovflow:
	addl $WSIZE, %ebx
	movl $-1, %edx
	movl $-1, %eax
umslashmod_exit:	
	movl %edx, (%ebx)
	subl $WSIZE, %ebx
	movl %eax, (%ebx)
	incl GlobalTp
	xor %eax, %eax		
	ret
L_mstar:
	movl $WSIZE, %eax
	movl GlobalSp, %ebx
	addl %eax, %ebx
	movl (%ebx), %ecx
	addl %eax, %ebx
	movl %ecx, %eax
	imull (%ebx)
	movl %eax, (%ebx)
	subl $WSIZE, %ebx
	movl %edx, (%ebx)
	xor %eax, %eax		
	ret
L_mplus:
	movl $WSIZE, %eax
	movl GlobalSp, %ebx
	addl %eax, %ebx
	movl (%ebx), %ecx
	movl %ebx, GlobalSp
	incl GlobalTp
	addl %eax, %ebx
	movl (%ebx), %edx
	addl %eax, %ebx
	movl (%ebx), %eax
	clc
	addl %ecx, %eax
	js mplus1
	adcl $0, %edx
mplus1:	movl %eax, (%ebx)
	movl %edx, -WSIZE(%ebx)
	xor %eax, %eax
	ret
L_mslash:
	movl $WSIZE, %eax
        incl GlobalTp
        movl GlobalSp, %ebx
	addl %eax, %ebx
        movl (%ebx), %ecx
	incl GlobalTp
	addl %eax, %ebx
	movl %ebx, GlobalSp
        cmpl $0, %ecx
        je mslash1
	addl %eax, %ebx
        movl (%ebx), %edx
	addl %eax, %ebx
	movl (%ebx), %eax
        idivl %ecx
        movl %eax, (%ebx)
	xor %eax, %eax		
	ret
mslash1:			
        movl $E_DIV_ZERO, %eax
        ret
L_udmstar:
	# multiply unsigned double and unsigned to give triple length product
	movl GlobalSp, %ebx
	addl $WSIZE, %ebx
	movl (%ebx), %ecx
	addl $WSIZE, %ebx
	movl (%ebx), %eax
	mull %ecx
	movl %edx, -WSIZE(%ebx)
	movl %eax, (%ebx)
	addl $WSIZE, %ebx
	movl %ecx, %eax
	mull (%ebx)
	movl %eax, (%ebx)
	subl $WSIZE, %ebx
	movl (%ebx), %eax
	subl $WSIZE, %ebx
	clc
	addl %edx, %eax
	movl %eax, WSIZE(%ebx)
	# subl $WSIZE, %ebx
	movl (%ebx), %eax
	adcl $0, %eax
	movl %eax, (%ebx)
	xor %eax, %eax 		
	ret
L_utmslash:
	# divide unsigned triple length by unsigned to give ud quotient
	movl GlobalSp, %ebx
	addl $WSIZE, %ebx
	movl (%ebx), %ecx		# divisor in ecx
	addl $WSIZE, %ebx
	movl (%ebx), %eax		# ut3
	movl $0, %edx
	divl %ecx			# ut3/u
	pushl %ebx			# keep local stack ptr
	movl GlobalSp, %ebx
	movl %eax, -4*WSIZE(%ebx)	# q3
	movl %edx, -5*WSIZE(%ebx)	# r3
	popl %ebx
	addl $WSIZE, %ebx
	movl (%ebx), %eax		# ut2
	movl $0, %edx
	divl %ecx			# ut2/u
	pushl %ebx
	movl GlobalSp, %ebx
	movl %eax, -2*WSIZE(%ebx)	# q2
	movl %edx, -3*WSIZE(%ebx)	# r2
	popl %ebx
	addl $WSIZE, %ebx
	movl (%ebx), %eax		# ut1
	movl $0, %edx
	divl %ecx			# ut1/u
	pushl %ebx
	movl GlobalSp, %ebx
	movl %eax, (%ebx)		# q1
	movl %edx, -WSIZE(%ebx)		# r1
	movl -5*WSIZE(%ebx), %edx	# r3 << 32
	movl $0, %eax
	divl %ecx			# (r3 << 32)/u
	movl %eax, -6*WSIZE(%ebx)	# q4
	movl %edx, -7*WSIZE(%ebx)	# r4
	movl -3*WSIZE(%ebx), %edx	# r2 << 32
	movl $0, %eax
	divl %ecx			# (r2 << 32)/u
	movl %eax, -8*WSIZE(%ebx)	# q5
	movl %edx, -9*WSIZE(%ebx)	# r5
	movl -7*WSIZE(%ebx), %edx	# r4 << 32
	movl $0, %eax
	divl %ecx			# (r4 << 32)/u
	movl %eax, -10*WSIZE(%ebx)	# q6
	movl %edx, -11*WSIZE(%ebx)	# r6
	movl -WSIZE(%ebx), %eax		# r1
	addl -9*WSIZE(%ebx), %eax	# r1 + r5
	addl -11*WSIZE(%ebx), %eax	# r1 + r5 + r6
	movl $0, %edx
	divl %ecx
	addl -10*WSIZE(%ebx), %eax	# q7 + q6
	addl -8*WSIZE(%ebx), %eax	# q7 + q6 + q5
	addl (%ebx), %eax		# q7 + q6 + q5 + q1
	popl %ebx
	movl %eax, (%ebx)
	subl $WSIZE, %ebx
	pushl %ebx
	movl GlobalSp, %ebx
	movl -2*WSIZE(%ebx), %eax	# q2
	addl -6*WSIZE(%ebx), %eax	# q2 + q4
	popl %ebx
	movl %eax, (%ebx)
	subl $WSIZE, %ebx
	movl %ebx, GlobalSp
	addl $2, GlobalTp
	xor %eax, %eax
	ret
L_mstarslash:
	movl GlobalSp, %ebx
	addl $WSIZE, %ebx
	addl $WSIZE, %ebx
	movl (%ebx), %eax
	addl $WSIZE, %ebx
	xorl (%ebx), %eax
	shrl $31, %eax
	pushl %eax	# keep sign of result -- negative is nonzero
	movl GlobalSp, %ebx
	addl $WSIZE, %ebx
	movl %ebx, GlobalSp
	incl GlobalTp
	call L_abs
	movl GlobalSp, %ebx
	addl $WSIZE, %ebx
	movl %ebx, GlobalSp
	incl GlobalTp
	call L_dabs
	movl GlobalSp, %ebx
	subl $WSIZE, %ebx
	movl %ebx, GlobalSp
	decl GlobalTp
	call L_udmstar
	movl GlobalSp, %ebx
	subl $WSIZE, %ebx
	movl %ebx, GlobalSp
	decl GlobalTp
	call L_utmslash	
	popl %eax
	cmpl $0, %eax
	jnz mstarslash_neg
	xor %eax, %eax
	ret
mstarslash_neg:
	DNEGATE
	xor %eax, %eax
	ret		
L_fmslashmod:
	movl GlobalSp, %ebx
	movl $WSIZE, %eax
	addl %eax, %ebx
	movl %ebx, GlobalSp
	movl (%ebx), %ecx
	addl %eax, %ebx
	movl (%ebx), %edx
	addl %eax, %ebx
	movl (%ebx), %eax
	idivl %ecx
	movl %edx, (%ebx)
	subl $WSIZE, %ebx
	movl %eax, (%ebx)
	incl GlobalTp
	cmpl $0, %ecx
	jg fmslashmod2
	cmpl $0, %edx
	jg fmslashmod3
	xor %eax, %eax
	ret
fmslashmod2:		
	cmpl $0, %edx
	jge fmslashmodexit
fmslashmod3:	
	decl %eax		# floor the result
	movl %eax, (%ebx)
	addl $WSIZE, %ebx
	addl %ecx, (%ebx)
fmslashmodexit:
	xor %eax, %eax
	ret
L_smslashrem:
	movl GlobalSp, %ebx
	movl $WSIZE, %eax
	addl %eax, %ebx
	movl %ebx, GlobalSp
	movl (%ebx), %ecx
	addl %eax, %ebx
	movl (%ebx), %edx
	addl %eax, %ebx
	movl (%ebx), %eax
	idivl %ecx
	movl %edx, (%ebx)
	subl $WSIZE, %ebx
	movl %eax, (%ebx)
	incl GlobalTp
	xor %eax, %eax		
	ret
L_stod:
	movl GlobalSp, %ebx
	movl WSIZE(%ebx), %eax
	pushl %edx
	cdq
	movl %edx, (%ebx)
	popl %edx
	movl $WSIZE, %eax
	subl %eax, %ebx
	movl %ebx, GlobalSp
	mov GlobalTp, %ebx
	movb $OP_IVAL, (%ebx)
	decl GlobalTp
	xor %eax, %eax	
	NEXT
L_stof:
	movl $WSIZE, %eax
        addl %eax, GlobalSp
        incl GlobalTp
        movl GlobalSp, %ebx
        fildl (%ebx)
        movl GlobalTp, %ebx
        movb $OP_IVAL, (%ebx)
        decl %ebx
        movb $OP_IVAL, (%ebx)
	decl GlobalTp
	decl GlobalTp
        movl GlobalSp, %ebx
	movl $WSIZE, %eax
        subl %eax, %ebx
        fstpl (%ebx)
	sall $1, %eax
        subl %eax, GlobalSp
	xor %eax, %eax
        NEXT
L_dtof:
	movl $WSIZE, %eax
        movl GlobalSp, %ebx
	addl %eax, %ebx
	movl (%ebx), %eax
	xchgl WSIZE(%ebx), %eax
	movl %eax, (%ebx)
        fildq (%ebx)
        fstpl (%ebx)
	xor %eax, %eax	
	NEXT	
L_froundtos:
	movl $WSIZE, %eax
        addl %eax, GlobalSp
        movl GlobalSp, %ebx
        fldl (%ebx)
        addl %eax, %ebx
        fistpl (%ebx)
        incl GlobalTp
        movl GlobalTp, %ebx
        incl %ebx
        movb $OP_IVAL, (%ebx)
	xor %eax, %eax
        NEXT
L_ftrunctos:
	movl $WSIZE, %eax
	addl %eax, GlobalSp
	movl GlobalSp, %ebx
	fldl (%ebx)
	fnstcw (%ebx)
	movl (%ebx), %ecx	# save NDP control word		
	movl %ecx, %edx	
	movb $12, %dh
	movl %edx, (%ebx)
	fldcw (%ebx)
	addl %eax, %ebx	
	fistpl (%ebx)
	subl %eax, %ebx
	movl %ecx, (%ebx)
	fldcw (%ebx)		# restore NDP control word
	incl GlobalTp
	movl GlobalTp, %ebx
	incl %ebx
	movb $OP_IVAL, (%ebx)
	xor %eax, %eax	
	NEXT	
L_ftod:
	movl $WSIZE, %eax
	movl GlobalSp, %ebx
	addl %eax, %ebx
	fldl (%ebx)
	subl %eax, %ebx
	fnstcw (%ebx)
	movl (%ebx), %ecx	# save NDP control word	
	movl %ecx, %edx
	movb $12, %dh		
	movl %edx, (%ebx)
	fldcw (%ebx)
	addl %eax, %ebx	
	fistpq (%ebx)
	subl %eax, %ebx
	movl %ecx, (%ebx)
	fldcw (%ebx)		# restore NDP control word
	addl %eax, %ebx 
	movl (%ebx), %eax
	xchgl WSIZE(%ebx), %eax
	movl %eax, (%ebx)
	xor %eax, %eax	
	NEXT	
L_degtorad:
	fldl FCONST_180
        movl GlobalSp, %ebx
        addl $WSIZE, %ebx
        fldl (%ebx)
        fdivp %st, %st(1)
        fldpi
        fmulp %st, %st(1)
        fstpl (%ebx)
        NEXT
L_radtodeg:
        movl GlobalSp, %ebx
        addl $WSIZE, %ebx
        fldl (%ebx)
        fldpi
	fxch
        fdivp %st, %st(1)
        fldl FCONST_180
        fmulp %st, %st(1)
        fstpl (%ebx)
        NEXT
L_fne:
	call L_feq
	movl GlobalSp, %ebx
	notl WSIZE(%ebx)
	NEXT
L_feq:
	movl $WSIZE, %eax
	addl %eax, GlobalSp
	movl GlobalSp, %ebx
	fldl (%ebx)
	addl %eax, GlobalSp
	addl %eax, GlobalSp
	movl GlobalSp, %ebx
	fldl (%ebx)
	fucompp
	fnstsw %ax
	andb $69, %ah
	xorb $64, %ah
	jne flt2
	jmp flt1
L_flt:
	movl $WSIZE, %eax
	addl %eax, GlobalSp
	movl GlobalSp, %ebx
	fldl (%ebx)
	addl %eax, GlobalSp
	addl %eax, GlobalSp
	movl GlobalSp, %ebx
	fcompl (%ebx)
	fnstsw %ax
	andb $69, %ah
	jne flt2
flt1:	movl GlobalSp, %ebx
	movl $-1, WSIZE(%ebx)
	jmp fltexit
flt2:   movl GlobalSp, %ebx
	movl $0, WSIZE(%ebx)
fltexit:
	movl $4, %eax
	addl %eax, GlobalTp
	movl GlobalTp, %ebx
	movb $OP_IVAL, (%ebx)
	decl GlobalTp
	xor %eax, %eax
	ret
L_fgt:
	movl $WSIZE, %eax
	addl %eax, GlobalSp
	movl GlobalSp, %ebx
	fldl (%ebx)
	addl %eax, GlobalSp
	addl %eax, GlobalSp
	movl GlobalSp, %ebx
	fcompl (%ebx)
	fnstsw %ax
	andb $69, %ah
	cmpb $1, %ah
	jne flt2
	jmp flt1	
L_fle:
	FOVER
	FOVER
	call L_feq
	_PUSH
	call L_flt
	_POP
	_OR
	ret
L_fge:
	FOVER
	FOVER
	call L_feq
	_PUSH
	call L_fgt
	_POP
	_OR
	ret
L_fzeroeq:
	movl $WSIZE, %eax
	movl GlobalSp, %ebx
	addl %eax, %ebx
	movl (%ebx), %ecx
	movl %ebx, GlobalSp
	addl %eax, %ebx
	movl (%ebx), %eax
	shll $1, %eax
	orl %ecx, %eax
	jnz fzeroeq2
fzeroeq1:	
	movl $-1, (%ebx)
	jmp fzeroeqexit
fzeroeq2:
	movl $0, (%ebx)
fzeroeqexit:	
	movl GlobalTp, %ebx
	incl %ebx
	incl %ebx
	movb $OP_IVAL, (%ebx)
	decl %ebx
	movl %ebx, GlobalTp
	xor %eax, %eax
	ret
L_fzerolt:
	movl $WSIZE, %eax
	addl %eax, GlobalSp
	movl GlobalSp, %ebx
	fldl (%ebx)
	add %eax, %ebx
	fldz
	fcompp	
	fnstsw %ax
	andb $69, %ah
	jne fzeroeq2		
	jmp fzeroeq1
L_fzerogt:
	ret				
L_fadd:
	movl $WSIZE, %eax
        addl %eax, GlobalSp
        movl GlobalSp, %ebx
        fldl (%ebx)
	sall $1, %eax
        addl %eax, %ebx
        faddl (%ebx)
        fstpl (%ebx)
	movl $WSIZE, %eax
        addl %eax, GlobalSp
        incl GlobalTp
	incl GlobalTp
	xor %eax, %eax
        NEXT
L_fsub:
	movl $WSIZE, %eax
	sall $1, %eax
	addl $WSIZE, %eax
        addl %eax, GlobalSp
        movl GlobalSp, %ebx
        fldl (%ebx)
	subl $WSIZE, %eax
        subl %eax, %ebx
        fsubl (%ebx)
        addl %eax, %ebx
        fstpl (%ebx)
	movl $WSIZE, %eax
        subl %eax, GlobalSp
        incl GlobalTp
	incl GlobalTp
	xor %eax, %eax
        NEXT
L_fmul:
	movl $WSIZE, %eax
        addl %eax, GlobalSp
        movl GlobalSp, %ebx
        fldl (%ebx)
	sall $1, %eax
        addl %eax, %ebx
        fmull (%ebx)
        fstpl (%ebx)
	sarl $1, %eax
        addl %eax, GlobalSp
        incl GlobalTp
	incl GlobalTp
	xor %eax, %eax
        NEXT
L_fdiv:
	movl $WSIZE, %eax
        addl %eax, GlobalSp
        movl GlobalSp, %ebx
        fldl (%ebx)
        ftst
        jnz fdiv1
        subl %eax, GlobalSp
        movl $E_DIV_ZERO, %eax
        jmp fdivexit
fdiv1:  addl $WSIZE, %ebx
	addl $WSIZE, %ebx
        fdivrl (%ebx)
        fstpl (%ebx)
        addl %eax, GlobalSp
        incl GlobalTp
	incl GlobalTp
fdivexit:
	xor %eax, %eax
	ret
L_fabs:
        movl GlobalSp, %ebx
        addl $WSIZE, %ebx
        fldl (%ebx)
        fabs
        fstpl (%ebx)
        NEXT
L_fneg:
        movl GlobalSp, %ebx
        addl $WSIZE, %ebx
        fldl (%ebx)
        fchs
        fstpl (%ebx)
        NEXT
L_floor:
	movl GlobalSp, %ebx
	addl $WSIZE, %ebx
	movl WSIZE(%ebx), %eax
	pushl %eax
	movl (%ebx), %eax
	pushl %eax
	call floor
	addl $8, %esp
	fstpl (%ebx)
	xor %eax, %eax		
	NEXT
L_fround:
	movl GlobalSp, %ebx
	addl $WSIZE, %ebx
	fldl (%ebx)
	frndint
	fstpl (%ebx)
	NEXT
L_ftrunc:
	movl GlobalSp, %ebx
	addl $WSIZE, %ebx
	fldl (%ebx)
	fnstcw (%ebx)
	movl (%ebx), %ecx	# save NDP control word
	movl %ecx, %edx
	movb $12, %dh
	movl %edx, (%ebx)
	fldcw (%ebx)
	frndint
	movl %ecx, (%ebx)
	fldcw (%ebx)		# restore NDP control word	
	fstpl (%ebx)		
	xor %eax, %eax
	NEXT
L_fsqrt:
	movl GlobalSp, %ebx
	fldl WSIZE(%ebx)
	fsqrt
	fstpl WSIZE(%ebx)
	NEXT
L_fcos:
	movl GlobalSp, %ebx
	fldl WSIZE(%ebx)
	fcos
	fstpl WSIZE(%ebx)
	NEXT
L_fsin:
	movl GlobalSp, %ebx
	fldl WSIZE(%ebx)
	fsin
	fstpl WSIZE(%ebx)
	NEXT
L_fatan2:
	movl GlobalSp, %ebx
	addl $2*WSIZE, %ebx
	fldl WSIZE(%ebx)
	fldl -WSIZE(%ebx)
	fpatan
	fstpl WSIZE(%ebx)
	movl %ebx, GlobalSp
	incl GlobalTp
	incl GlobalTp
	NEXT
L_fsincos:
	movl GlobalSp, %ebx
	fldl WSIZE(%ebx)
	fsincos
	fstpl -WSIZE(%ebx)
	fstpl WSIZE(%ebx)
	subl $2*WSIZE, %ebx
	movl %ebx, GlobalSp
	movl GlobalTp, %ebx
	movb $OP_IVAL, (%ebx)
	decl %ebx
	movb $OP_IVAL, (%ebx)
	decl %ebx
	movl %ebx, GlobalTp	
	NEXT
#

	.comm GlobalSp,4,4
	.comm GlobalTp,4,4
	.comm GlobalIp,4,4
	.comm GlobalRp,4,4
	.comm GlobalRtp,4,4
	.comm BottomOfStack,4,4
	.comm BottomOfReturnStack,4,4
	.comm BottomOfTypeStack,4,4
	.comm BottomOfReturnTypeStack,4,4
	.comm vmEntryRp,4,4
	.comm Base,4,4
	.comm State,4,4
	.comm pTIB,4,4
	.comm TIB,256,1
	.comm WordBuf,256,1
	.comm NumberCount,4,4
	.comm NumberBuf,256,1

	
