\ ogy.4th
\
\ Demonstrate Ott-Grebogi-Yorke chaos control method for a fixed point
\   on the Henon Map, following OGYs original paper:
\
\   E. Ott, C. Grebogi, and J.A. Yorke, "Controlling Chaos",
\   Physical Review Letters, vol 64, p 1196 (1990)
\
\ Copyright (c) 2002, Krishna Myneni, 01-20-2002
\
\ Requires:
\	matrix.4th
\	quadratic.4th
\	henon.4th
\
\ Revisions:
\
include henon

fvariable p	0e p f!		\ control parameter
fvariable p*    1e-2 p* f!      \ max control parameter
fvariable A0    A f@ A0 f!

: set_control ( f -- | set control parameter )
	fdup p f! A0 f@ f+ A f! ;

: g (  -- dx dy | determine the control vector g near the fixed point)
	0e set_control          \ start with zero control parameter
	solve_fixed_point       \ determine the fixed pt starting somewhere 
	set_fixed_point		\ save the fixed point
	1e-3 set_control	\ change the control parameter by small amount 
	solve_fixed_point       \ find the new fixed point
	yf f@ f- p f@ f/ fswap 
	xf f@ f- p f@ f/ fswap  \ compute vector g
	0e set_control ;

\ OGY's f vectors. These vectors are defined by the relations:
\
\	fs * es = 1    fs * eu = 0
\       fu * es = 0    fu * eu = 1
\
\ where es and eu are the eigenvectors for the stable and unstable
\ directions and '*' indicates a dot product. The f vectors are 
\ ordered such that f_s is in 1st col, f_u is in 2nd col

2 2 fmatrix f  \ OGY's f vectors

: fvectors ( -- | determine the f vectors)

	\ Compute fs

	1 2 e fmat@  2 1 e fmat@  f*  ( eux*esy )
	1 1 e fmat@  2 2 e fmat@  f*  ( esx*euy )
	f- 2>r
	2 2 e fmat@ fnegate 2r@ f/
	1 1 f fmat!                   ( fsx )
	1 2 e fmat@ 2r@ f/
	2 1 f fmat!                   ( fsy )

	\ Compute fu

	2r> fnegate 2>r
	2 1 e fmat@ fnegate 2r@ f/
	1 2 f fmat!                   ( fux )
	1 1 e fmat@ 2r> f/
	2 2 f fmat!                   ( fuy )	
;

: fu ( -- fux fuy | return the components of fu)
	1 2 f fmat@ 2 2 f fmat@ ;

: dot ( x1 y1 x2 y2 -- s | compute dot product of two vectors)
	frot f* 2>r f* 2r> f+ ;
 
fvariable control_scale

: compute_scale_factor ( -- | precompute the control parameter scale factor)
	lambda2 f@ fdup 1e f- f/
	g fu dot f/ 
	control_scale f! ;

: compute_control ( x y -- p | compute the control parameter based on state)
	yf f@ f- fswap xf f@ f- fswap  ( subtract fixed point vector) 
	fu dot 
	control_scale f@ f* ;

\ Starting with the given initial state, apply the OGY control technique
\   to the Henon map and display the state upon each iteration, up to n times

: ogy ( x y n -- | apply the OGY control method )

	\ Preliminaries -- compute relevant quantities of Henon map,
	\   the f vectors, the control vector g, and the control scale factor

	A f@ A0 f!  	\ keep copy of uncontrolled A parameter
	solve_fixed_point 
	Jacobian 
	eigenvalues 
	eigenvectors
	fvectors
	compute_scale_factor

	\ Now we are ready to iterate the map and apply the control

	0 do
	  fover f. 2 spaces fdup f. \ show current x y
	  fover fover compute_control 
	  fdup fabs p* f@ f> if fdrop 0e then 
	  fdup 2 spaces f. cr       \ show parameter perturbation p 
	  set_control
	  H			    \ iterate the map 
	loop
	fdrop fdrop
	A0 f@ A f!                  \ restore original parameter
;

	  
	
