\ notes.4th
\
\ Simple electronic note-keeping system
\
\ Copyright (c) 2001 Krishna Myneni
\ Provided under the GNU General Public License
\ 
\ Required files (only for kForth):
\
\	strings.4th
\	files.4th  ( or filesw.4th under Windows )
\
\ Revisions:
\	9-18-2001 created KM
\	9-20-2001 simplified user commands KM
\	9-21-2001 added modify-note KM
\	4-06-2002 used $constant for string constants and
\	            made mods to allow operation under other Forths;
\		    tested under PFE and gforth  KM	
\ Notes:
\
\ 1. Each notes database file can hold 2560 note records, but there
\      is no size restriction on individual note records.
\ 
\ 2. The structure of the notes database is:
\
\	index		1024 bytes
\	note records	Each record is variable length, but the length 
\			  is always a multiple of 64 bytes
\
\    The structure of a note record is:
\
\	date stamp	4 bytes
\	time stamp	4 bytes
\	user name	8 bytes
\	record length	4 bytes
\	checksum	4 bytes
\	pad area1	pad to 64 bytes
\	keywords list	64 bytes
\	title		64 bytes
\	pad area2	64 bytes
\	body		multiple of 64 bytes
\
\ 3. Compatible definitions for ANS Forth systems are provided below.
\      If not using kForth, uncomment the section "ANS Forth requires"
\      and comment out the section "kForth requires". Methods of
\      passing shell commands vary among ANS Forths. The ANS Forth
\      section defines "shell" for PFE, gforth and iForth. Uncomment the
\      definition appropriate for your system. A shell command is used to 
\      find the name of the user entering a new note into the notes 
\      database in the word "!author". If this feature is not needed, 
\      disable it in the definition of !author.
\
\
\ 4. Change the default notes file name DEF-NOTES-FILE for your use.
\

decimal

\ ====== ANS Forth requires =========
(
: a@         \ a1 -- a2  | fetch address stored at a1
	@ ;

: ?allot     \ n -- a | allot n bytes, return start address
       here swap allot ;

: isdigit    \ n -- flag | return true if n is ascii value of '0' through '9'
	dup [char] / > swap [char] : < and ;  

variable number_sign
variable number_val

: string>s   \ ^str -- n | always interpret in base 10 
	0 number_val !
	false number_sign !
	count
	0 ?do
	  dup c@
	  case
	    [char] -  of true number_sign ! endof 
	    [char] +  of false number_sign ! endof 
	    dup isdigit 
	    if
	      dup [char] 0 - number_val @ 10 * + number_val !
	    then
	  endcase
	  1+
	loop
	drop
	number_val @ number_sign @ if negate then ;

: file-exists  \ ^filename  -- flag | return true if file exists
	count R/O open-file
	if drop false else close-file drop true then ;	

\ pfe shell command
\ : shell  system ; \ c-addr u -- n | execute a shell command in PFE

\ gforth shell command
\ : shell  system  $? ; \ c-addr u -- n | execute a shell command in gforth

\ iforth shell command
\ : shell  system  RETURNCODE @ ;  \ c-addr u -- n | shell command in iForth


10 constant EOL

create EOL_BUF 4 allot
EOL EOL_BUF c!
0 EOL_BUF 1+ c! 

)
\ ====== end of ANS Forth requires ===


\ ====== kForth requires =============
( Comment out this section if not using kForth )

include strings
include files    \ under Windows, include filesw

: shell  \ c-addr u -- n | execute a shell command in kForth
	strpck system ; 

\ ====== end of kForth requires ======


: pack ( a u a2 -- | copy string to counted string at a2)
	2dup c! 1+ swap cmove ;	

: $constant  ( a u -- | create a string constant )
	create  256 ?allot pack
	does>   count ;  \ execution: ( -- a' u )


s" /home/krishna/notes.db" 		$constant  DEF-NOTES-FILE

s" ERROR: unable to open notes file "	$constant  E-OPEN-MSG
s" ERROR: unable to read the index." 	$constant  E-READ-INDEX-MSG
s" ERROR: unable to find first record."	$constant  E-POS-MSG
s" ERROR: unable to read the record." 	$constant  E-READ-MSG
s" ERROR: unable to write the note." 	$constant  E-WRITE-MSG
s" > " 					$constant  NOTE-PROMPT
s" Jan Feb Mar Apr May Jun Jul Aug Sep Oct Nov Dec" $constant MONTHS


	
1024 			constant  INDEX-SIZE
0 			constant  DATE-OFFSET
DATE-OFFSET cell+ 	constant  TIME-OFFSET
TIME-OFFSET cell+ 	constant  USER-OFFSET
USER-OFFSET 2 cells + 	constant  LENGTH-OFFSET
LENGTH-OFFSET cell+ 	constant  CHECKSUM-OFFSET
CHECKSUM-OFFSET cell+ 	constant  PAD1-OFFSET
64 			constant  KEYWORDS-OFFSET
KEYWORDS-OFFSET 64 + 	constant  TITLE-OFFSET
TITLE-OFFSET 64 + 	constant  PAD2-OFFSET
PAD2-OFFSET 64 + 	constant  BODY-OFFSET

create my-notes-file 256 allot
DEF-NOTES-FILE my-notes-file pack
variable my-notes-fd

create notes-index 1024 allot

: use-notes ( ^name -- | select note files from the input stream )
	count my-notes-file pack ;
   
: create-notes ( ^name -- | create a new notes file )
	use-notes
	my-notes-file c@ 0= if
	  cr ." ERROR: no file name specified."
	else
	  my-notes-file file-exists if
	    cr ." ERROR: notes file already exists."
	  else
	    my-notes-file count R/W create-file
	    if 
	      cr ." ERROR: unable to create new notes file "
	    else
	      my-notes-fd !  cr 
	      notes-index INDEX-SIZE 2dup erase 	\ clear the index
	      my-notes-fd @ write-file drop 		\ write the index
	      my-notes-fd @ file-position 2drop INDEX-SIZE <> if 
	        ." ERROR: unable to write the index." cr 
	      then
	      my-notes-fd @ close-file drop
	      ." Successfully created new notes file " 
	    then my-notes-file count type  cr
	  then 
	then ;

: open-notes ( n -- | open the notes file with access method given by n )
	\ n can be either R/O or R/W
	my-notes-file count rot open-file
	if drop E-OPEN-MSG type 
	  my-notes-file count type cr abort 
	then
	my-notes-fd ! 
	notes-index INDEX-SIZE my-notes-fd @ read-file 
	drop INDEX-SIZE <>
	if E-READ-INDEX-MSG type abort then ;

: open-notes-append ( -- )
	my-notes-file count R/W open-file
	if drop E-OPEN-MSG type  
	  my-notes-file count type cr abort
	then
	my-notes-fd ! 
	my-notes-fd @ file-size drop 
	my-notes-fd @ reposition-file  \ position at EOF 
	drop ;


: close-notes ( -- )
	my-notes-fd @ close-file drop ;

\ Time utilities

: dmy>s ( day month year -- n | pack day month year into single cell )
	9 lshift swap 5 lshift or or ;

: s>dmy ( n -- day month year | unpack date )
	dup dup 31 and swap 5 rshift 15 and rot 9 rshift ; 

: smh>s ( sec min hour -- n | pack secs minutes hours into single cell )
	12 lshift swap 6 lshift or or ;

: s>smh ( n -- sec min hour | unpack time )
	dup dup 63 and swap 6 rshift 63 and rot 12 rshift ;


BODY-OFFSET constant HDRSIZE
create note-hdr HDRSIZE allot
64 1024 * constant MAXBODYSIZE
create note-body MAXBODYSIZE allot

: @date-stamp ( -- n ) note-hdr DATE-OFFSET + @ ;
: @time-stamp ( -- n ) note-hdr TIME-OFFSET + @ ;
: @record-length ( -- n ) note-hdr LENGTH-OFFSET + @ ;
: @note-checksum ( -- n ) note-hdr CHECKSUM-OFFSET + @ ;
: @body-length ( -- n ) @record-length HDRSIZE - ;
: !date-stamp ( -- ) time&date dmy>s note-hdr DATE-OFFSET + ! 2drop drop ;
: !time-stamp ( -- ) time&date 2drop drop smh>s note-hdr TIME-OFFSET + ! ;
: !record-length ( n -- ) note-hdr LENGTH-OFFSET + ! ;
: !note-checksum ( n -- ) note-hdr CHECKSUM-OFFSET + ! ;

: !title ( addr count -- )
	note-hdr TITLE-OFFSET + 64 blank
	64 min note-hdr TITLE-OFFSET + swap cmove ;

: !keywords ( addr count -- )
	note-hdr KEYWORDS-OFFSET + 64 blank
	64 min note-hdr KEYWORDS-OFFSET + swap cmove ;

: !author ( -- )
	note-hdr USER-OFFSET + 8 blank
	s" echo -n $USER > author.tmp" shell drop
	c" author.tmp" file-exists if
	  s" author.tmp" R/O open-file 0=
	  if
	    dup note-hdr USER-OFFSET + 8 rot read-line 2drop drop
	    close-file drop
	  else
	    drop
	  then  
	then ;
	
: ?note-checksum ( -- n | compute the checksum for the body of the note )
	@body-length dup 0> swap MAXBODYSIZE <= and
	if
	  0 note-body
	  @body-length 0 do dup >r c@ + r> 1+ loop drop
	else
	  -1
	then ;   
	   
: read-note-hdr ( -- flag )
	note-hdr HDRSIZE my-notes-fd @ read-file
	swap HDRSIZE <> or ; 

: read-note-body ( -- flag )  \ assumes hdr has been read
	note-body @body-length my-notes-fd @ read-file
	swap @body-length <> or ; 

: read-next-note ( -- flag | read a note from the current position )
	read-note-hdr 0= if read-note-body else true then ;

: set-note-position ( n -- flag | set position in file for note n )
	\ origin is n=1; return 0 if successfull.

	1- dup 10 /mod cells notes-index + @
	dup 0= if
	  \ The index is not available; we must search for the record
	  2drop
	  INDEX-SIZE 0 my-notes-fd @ reposition-file drop  	  
	else
	  \ Position at the nearest record for which we have an index
	  0 my-notes-fd @ reposition-file drop
	  nip
	then
	\ Read up to the desired record
	0 ?do read-next-note if unloop true exit then loop 
	false ;

: read-note ( n -- flag | read note n from the database )
	set-note-position 0= if read-next-note else true then ;

: write-note-hdr ( -- n | n = 0 if success )
	my-notes-fd @ file-position drop
	note-hdr HDRSIZE my-notes-fd @ write-file drop
	my-notes-fd @ file-position drop d- dnegate 
	HDRSIZE s>d d= invert ;

: write-note-body ( -- n | n = 0 if success )
	my-notes-fd @ file-position drop
	note-body @body-length my-notes-fd @ write-file drop
	my-notes-fd @ file-position drop d- dnegate
	@body-length s>d d= invert ;

: write-note ( -- flag )
	write-note-hdr 0= if write-note-body else true then ; 

: tab 9 emit ;

: display-count ( n -- )
	s>d <# # # # # #> type ;

: display-note-time-stamp ( -- )
	@date-stamp s>dmy 4 .r bl emit
	12 min 1- 2* 2* MONTHS drop + 3 type 
	bl emit 2 .r 2 spaces
	@time-stamp s>smh 
	s>d <# [char] : hold # # #> type
	s>d <# [char] : hold # # #> type
	s>d <# # # #> type  ;

: display-note-author ( -- )
	note-hdr USER-OFFSET + 8 type ;

: display-note-title ( -- )
	note-hdr TITLE-OFFSET + 64 -trailing type ;

: display-note-keywords ( -- )
	note-hdr KEYWORDS-OFFSET + 64 -trailing type ;

: display-note-header ( -- ) 
	display-note-time-stamp 2 spaces 
	display-note-author 2 spaces
	display-note-title ;

: display-note-body ( -- )
	note-body @body-length type ;

variable note-counter

: list-notes ( n1 n2 -- | display headers for notes n1 through n2 )
	over 0< if nip 1 swap then
	R/O open-notes
	swap dup note-counter !
	set-note-position if 
	  drop close-notes E-POS-MSG type abort
	then
	cr
	begin
	  read-next-note dup
	  0= if
	    note-counter @ display-count [char] : emit 2 spaces
	    display-note-header cr
	    1 note-counter +!
	  then
	  over note-counter @ u< or
	until
	drop
	close-notes ;

: view-notes ( n1 n2 -- | view notes n1 through n2 )
	over 0< if 2drop exit then
	dup 0< if drop dup then 1+
	R/O open-notes
	swap dup set-note-position
	if 2drop close-notes E-POS-MSG type abort then
	cr
	?do
	  read-next-note
	  if
	    close-notes
	    ." Note " i . ." not found" cr
	    unloop exit
	  else
	    i display-count [char] : emit 2 spaces
	    display-note-header cr
	    display-note-body cr 
	    @note-checksum ?note-checksum <> if
	    ." ERROR: checksum does not match." then 
	  then
	loop
	close-notes ;


variable note-body-ptr
create input-line 132 allot
variable input-count

: get-keywords ( -- )
	." KEYWORDS: " input-line 64 accept 
	input-line swap !keywords ;

: pad64 ( n1 -- n2 | n2 is the next multiple of 64 for n1 )
	64 /mod swap 0<> if 1+ then 6 lshift ; 

: take-note ( -- )
	note-hdr HDRSIZE blank
	EOL word count !title
	note-body MAXBODYSIZE blank
	note-body note-body-ptr !
	cr ." Type ':q' on single line when finished." cr
	begin
	  NOTE-PROMPT type
	  input-line 132 accept input-count ! cr
	  input-count @ 2 =
	  input-line c@ [char] : =
	  input-line 1+ c@ 95 and [char] Q = 
	  and and invert
	while
	  input-line note-body-ptr a@ input-count @ cmove
	  input-count @ note-body-ptr +!
	  EOL note-body-ptr a@ !
	  1 note-body-ptr +!
	repeat

	note-body-ptr @ note-body - pad64 \ compute padded body length
	HDRSIZE + !record-length		
	get-keywords cr
	!date-stamp !time-stamp !author	
	?note-checksum !note-checksum	
	open-notes-append
	write-note if E-WRITE-MSG type then
	close-notes ;

: get-number ( -- n | parse input to obtain a numeric argument )
	bl word string>s ;

: validate-index ( n1 -- n1 or -1 | check for n1 <= 0 )
	dup 0 <= if drop -1 then ;

: order-index-pair ( n1 n2 -- n1 n2 or n2 n1 )
	dup 0> if 2dup > if swap then then ;

: get-note-range ( -- n1 n2 | parse input to obtain arguments to commands )
	get-number validate-index
	get-number validate-index
	order-index-pair ;

create modified-hdr HDRSIZE allot

: modify-note ( n -- | allow user to modify the title and keywords of a note )
	dup 0 <= if drop exit then 
	dup R/O open-notes read-note close-notes 
	if drop E-READ-MSG type abort then
	display-note-header cr
	." Current Title: " display-note-title cr
	." New Title    : " input-line 64 blank input-line 64 accept
	dup 0> if input-line swap !title else drop then cr
	." Current Keywords: " display-note-keywords cr
	." New Keywords    : " input-line 64 blank input-line 64 accept
	dup 0> if input-line swap !keywords else drop then cr
	note-hdr modified-hdr HDRSIZE cmove	\ keep copy of the mod hdr
	R/W open-notes 
	set-note-position
	if close-notes E-POS-MSG type abort then
	modified-hdr note-hdr HDRSIZE cmove
	write-note-hdr 
	if E-WRITE-MSG type then 
	close-notes ;
	  

: help-notes ( -- )

	tab ." ln [range]    " tab ." List notes in the specified range" cr
	tab ." vn <range>    " tab ." View notes in the specified range" cr
	tab ." tn [title]    " tab ." Take a new note" cr
	tab ." un <filename> " tab ." Use the specified notes file" cr
	tab ." cn <filename> " tab ." Create a new notes file" cr
	tab ." mn <note>     " tab ." Modify title or keywords of note" cr
	tab ." hn            " tab ." Display this list of commands" cr
	cr 
	." <> is required argument, [] is optional argument" cr
	." 'range' can be either a single note number, e.g. 3, or a range" cr
	."   entered in the form of a pair of numbers, e.g. 3 5" cr
;

\ User commands

: cn ( -- | create specified notes file ) bl word create-notes ;
: un ( -- | use specified notes file ) bl word use-notes ;
: ln ( -- | list specified notes ) get-note-range list-notes ;
: vn ( -- | view specified notes ) get-note-range view-notes ;
: tn ( -- | take a new note ) take-note ;
: mn ( -- | modify title/keys for specified note) get-number modify-note ;
: hn ( -- | display help for notes commands ) help-notes ;

.( Current notes file: ) my-notes-file count type cr 
.( Commands: cn un ln vn tn mn [ hn for help ] )

