#include <unistd.h>
#include <fcntl.h>
#include <glib.h>
#include <string.h>
#include <pthread.h>
#include <signal.h>

#include <asdutil.h>
#include <protocol-asd-spec.h>

#include <thread-manager.h>

#include "protocol-asd.h"
#include "protocol-asd-impl.h"
#include "idgen.h"

gboolean protocol_asd_auth_locked = TRUE;

gboolean protocol_asd_read_request(ProtocolAsdClient *client, ProtocolAsdRequest *request)
{
  g_assert(client && request);

  if (atomic_read(client->fd, request, sizeof(ProtocolAsdRequest)) != sizeof(ProtocolAsdRequest))
    return FALSE;

  request->id = GUINT16_FROM_LE(request->id);
  request->command = GUINT16_FROM_LE(request->command);
  request->length = GUINT32_FROM_LE(request->length);
  return TRUE;
}

void protocol_asd_write_error(ProtocolAsdClient *client, ProtocolAsdRequest *request, guint16 error)
{
  ProtocolAsdRequest r;
  ProtocolAsdError e;
  g_assert(client && request);
  
  r.id = GUINT16_TO_LE(request->id);
  r.version = PROTOCOL_ASD_VERSION;
  r.command = GUINT16_TO_LE(PROTOCOL_ASD_COMMAND_ERROR);
  r.length = GUINT32_TO_LE(sizeof(e));
  
  if (atomic_write(client->fd, &r, sizeof(ProtocolAsdRequest)) != sizeof(ProtocolAsdRequest))
    return;

  e.error = error;

  atomic_write(client->fd, &e, sizeof(ProtocolAsdError));
}

gboolean protocol_asd_write_ack(ProtocolAsdClient *client, ProtocolAsdRequest *request, guint32 l)
{
  ProtocolAsdRequest r;

  g_assert(client && request);

  r.id = GUINT16_TO_LE(request->id);
  r.version = PROTOCOL_ASD_VERSION;
  r.command = GUINT16_TO_LE(PROTOCOL_ASD_COMMAND_ACKNOWLEDGE);
  r.length = GUINT32_TO_LE(l);

  if (atomic_write(client->fd, &r, sizeof(ProtocolAsdRequest)) != sizeof(ProtocolAsdRequest))
    return FALSE;
  
  return TRUE;
}

void protocol_asd_dispatch(int fd)
{
  ProtocolAsdClient client;

  client.fd = fd;
  client.id = generate_id();
  client.auth = FALSE;

  for (;;)
    {
      ProtocolAsdRequest request;
      ProtocolAsdHandler *handler;
      gint r;
      
      if (!protocol_asd_read_request(&client, &request))
	break;
      
      if (request.version != PROTOCOL_ASD_VERSION)
	{
	  protocol_asd_write_error(&client, &request, PROTOCOL_ASD_ERROR_VERSION_NOT_SUPPORTED);
	  break;
	}

      if (!(handler = protocol_asd_find_handler(request.command)))
	{
	  protocol_asd_write_error(&client, &request, PROTOCOL_ASD_ERROR_UNKNOWN_COMMAND);
	  break;
	}

      if (handler->check_auth && !(client.auth || !protocol_asd_auth_locked))
	{
	  protocol_asd_write_error(&client, &request, PROTOCOL_ASD_ERROR_ACCESS_DENIED);
	  break;
	}

      if (handler->check_length)
	if (request.length != handler->length)
	  {
	    protocol_asd_write_error(&client, &request, PROTOCOL_ASD_ERROR_SIZE_MISMATCH);
	    break;
	  }
      
      g_assert(handler->proc);
      r = handler->proc(&client, &request);

      if (r == 1) return;
      if (r == 2) break;
    }

  close(client.fd);
}


static void _protocol_asd_thread_cleanup(gpointer p)
{
  thread_unregister(pthread_self());
  close(*((int*)p));
}

static void* _protocol_asd_thread(void *p)
{
  sigset_t ss;
  int fd;

  g_assert(p);
  fd = *((int*)p);
  g_free(p);

  thread_register(pthread_self());
  pthread_cleanup_push(_protocol_asd_thread_cleanup, p);

  pthread_sigmask(SIG_BLOCK, NULL, &ss);
  sigaddset(&ss, SIGINT);
  sigaddset(&ss, SIGQUIT);
  //  sigaddset(&ss, SIGHUP);
  pthread_sigmask(SIG_BLOCK, &ss, NULL);

  pthread_setcanceltype(PTHREAD_CANCEL_DEFERRED, NULL);
  pthread_setcancelstate(PTHREAD_CANCEL_ENABLE, NULL);

  protocol_asd_dispatch(fd);

  pthread_cleanup_pop(0);
  thread_unregister(pthread_self());

  pthread_detach(pthread_self());

  return NULL;
}

void protocol_asd_new_socket(int fd)
{
  pthread_t t;
  g_assert(fd >= 0);
  g_assert(!pthread_create(&t, NULL, _protocol_asd_thread, g_memdup(&fd, sizeof(int))));
}

