#include <recursive-lock.h>
#include <gc.h>

#include "namespace.h"

static GHashTable *_namespace = NULL;

static GStaticMutex _namespace_mutex = G_STATIC_MUTEX_INIT;
static GStaticPrivate _namespace_private = G_STATIC_PRIVATE_INIT;

typedef struct
{
  gchar* name;
  gpointer ptr;
  guint32 type;
} NamespaceEntry;

#define _namespace_lock() recursive_lock(&_namespace_mutex, &_namespace_private)
#define _namespace_unlock() recursive_unlock(&_namespace_mutex, &_namespace_private)

gboolean namespace_register(gchar *name)
{
  gboolean b;

  g_assert(name);
  _namespace_lock();

  if (!*name)
    b = FALSE;
  else
    {
      if (!_namespace)
        _namespace = g_hash_table_new(g_str_hash, g_str_equal);
      
      if (g_hash_table_lookup(_namespace, name) == NULL)
        {
          NamespaceEntry *n;
          g_assert(n = g_new(NamespaceEntry, 1));
          n->name = g_strdup(name);
          n->ptr = NULL;
          n->type = NAMESPACE_ANY;
          
          g_hash_table_insert(_namespace, n->name, n);
          b = TRUE;
        }
      else
        b = FALSE;
    }

  _namespace_unlock();

  return b;
}

void namespace_unregister(gchar *name)
{
  NamespaceEntry *n;
  g_assert(name && _namespace);

  _namespace_lock();

  g_assert(n = g_hash_table_lookup(_namespace, name));
  g_hash_table_remove(_namespace, n->name);

  g_free(n->name);
  g_free(n);

  if (g_hash_table_size(_namespace) == 0)
    {
      g_hash_table_destroy(_namespace);
      _namespace = NULL;
    }
  
  _namespace_unlock();
}

gboolean namespace_registered(gchar *name)
{
  gboolean b;
  g_assert(name);
  
  _namespace_lock();

  if (!_namespace)
    b = FALSE;
  else
    b = (g_hash_table_lookup(_namespace, name) != NULL);
  
  _namespace_unlock();

  return b;
}

gpointer namespace_lookup_with_type(gchar *name, guint32 type)
{
  gpointer p;
  g_assert(name);

  _namespace_lock();

  if (_namespace)
    p = g_hash_table_lookup(_namespace, name);
  else
    p = NULL;

  if (p)
    {
      if ((type == NAMESPACE_ANY) || (((NamespaceEntry*)p)->type == type))
	p = ((NamespaceEntry*)p)->ptr;
      else
	p = NULL;
    }

  if (p)
    gc_ref_inc(p);

  _namespace_unlock();

  return p;
}

void namespace_set(gchar *name, gpointer ptr, guint32 type)
{
  NamespaceEntry *n;
  g_assert(_namespace && name);

  _namespace_lock();

  g_assert((n = g_hash_table_lookup(_namespace, name)));
  n->ptr = ptr;
  n->type = type;

  _namespace_unlock();
}


static void _proc(gpointer key, gpointer value, gpointer userdata)
{
  GSList **l;
  g_assert(l = (GSList**) userdata);
  g_assert(value);

  *l = g_slist_prepend(*l, value);
}

void namespace_foreach(namespace_foreach_proc proc, guint32 type, gpointer userdata)
{
  GSList *list = NULL, *l;
  _namespace_lock();

  g_hash_table_foreach(_namespace, _proc, &list);

  l = list;
  while (l)
    {
      NamespaceEntry *n;
      g_assert(n = (NamespaceEntry*) l->data);
      if ((type == NAMESPACE_ANY) || (n->type == type))
	proc(n->name, n->ptr, userdata);

       l = g_slist_next(l);
    }

  g_slist_free(list);

  _namespace_unlock();
}
