/*
 * Copyright 1995,96 Thierry Bousch
 * Licensed under the Gnu Public License, Version 2
 *
 * $Id: induce.h,v 2.1 1996/09/18 10:06:29 bousch Exp $
 *
 * Data structures and prototypes for "induce"
 */

typedef struct _vertex {
	int used;		/* How many people use this vertex */
	struct _vertex *hnext;	/* Next vertex in hash table */
	int deps;		/* Number of ancestors */
	struct _vertex **dep;	/* List of ancestors */
	mref_t mr;		/* The mref holding the value */
	int depth;		/* How early will we need it? */
	int score;		/* Is it good to evaluate it? */
	const char *bytecode;	/* How to compute it */
	char name[0];		/* Variable name */
} vertex;

typedef struct _indexed_variable {
	const char *rootname;	/* The name without the indices */
	int nbind;		/* Number of indices */
	const char *ibcode[0];	/* The bytecode for each index */
} idx_var;

typedef struct _eval_rule {
	const char *rootname;	/* Variable rootname */
	int nbind;		/* Number of indices */
	idx_var *conditions;	/* When is this rule valid? */
	int nbdep;		/* Number of dependencies */
	idx_var **dep;		/* List of dependencies */
	const char *bytecode;	/* How to compute it */
} eval_rule;

typedef struct _arglist {
	int arity;		/* Number of arguments, so far */
	char *bytecode;		/* Collected bytecodes */
} alist;

vertex *lookup_vertex (const char *name);
vertex *make_vertex (const char *name);
int number_ancestors (const char *name, char **anc, const char **bcode);
void free_vertex (vertex *v);
void find_depth (vertex *v, int min_depth);
void init_vertex_htable (int initial_size);
void eval_vertices (void);
void depth_statistics(void), hash_statistics(void), usage_statistics(void);
mref_t compute_vertex (vertex *v);
void reset_interpreter_stack (void);
mref_t exec_bytecode (int argc, mref_t *argv, const char *cmd);
int exec_int_bytecode (int argc, int *argv, const char *cmd);
void insert_rule (eval_rule* rule);
eval_rule* first_matching_rule (const char *name, int argc, int *argv);
int register_function (char *func, int arity);
const char* tilde_expand (const char *filename);

/* Memoization */
int register_memo (const char *filename);
int is_saved (const char *name);
void declare_precious (const char *rootname, int arity);
int save_precious (vertex *v);
int retrieve_precious (vertex *v);

/*
 * Global information
 */
extern int floating_precision, no_literals, no_rationals;
extern int trace_mode, quiet;
extern int parsed_poly_type;
extern const char *memo_file, *curr_memo_file;

/*
 * The following functions are for the parser
 */
int yylex(void), yyparse(void);
int new_index (char* ident);
void start_new_rule (const char* ident);
void start_idxvar (const char* rootname);
char* collect_idxvar (void);
void collect_iexpr (const char* bcode);
void collect_condition (const char* bcode);
void add_this_rule (const char* bcode);

/* Paths */

#ifndef _PATH_M4
#define _PATH_M4	"/usr/bin/m4"
#endif
#ifndef _PATH_TMPDIR
#define _PATH_TMPDIR	"/tmp"
#endif
#ifndef _PATH_RULES
#define _PATH_RULES	"Inducefile:Inducefile.m4:~/.inducefile:~/.inducefile.m4"
#endif
#ifndef _PATH_MEMO
#define _PATH_MEMO	"~/.common-memo.gdbm"
#endif
#define _PATH_MEMOLIST	_PATH_TMPDIR "/memolistXXXXXX"
#define _PATH_TMPFILE	_PATH_TMPDIR "/tmprulesXXXXXX"
#define _PATH_TMPMEMO	_PATH_TMPDIR "/tmpmemo.XXXXXX"
