/* Quinn Diff, Determine which packages need recompiled for any given architecture */
/* Copyright (C) 1997-1999, James Troup <james@nocrew.org> */

/* This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <glib.h>
#include "common.h"
#include "error.h"
#include "main.h"
#include "parse_packages.h"
#include "utils.h"
#include "xmalloc.h"

GHashTable *packages_hash_table;

void packages_free_ht_entry (gpointer key, gpointer package, gpointer cruft);

void
determine_packages_architecture (packages_info *packages)  
{

  while (packages->buffer_pos < packages->file_length && !packages_architecture)
    {
      packages_architecture = get_field (packages->buffer, packages->file_length, packages->buffer_pos, "Architecture: ", "unknown");

      if (packages_architecture && strstr (packages_architecture, "all") != NULL)
	{
	  xfree (packages_architecture);
	  packages_architecture = NULL;
	}

      if (!packages_architecture)
	skip_to_next_package (packages->buffer, &packages->buffer_pos, packages->file_length);
    }
  /* Reset the buffer position to the start */
  packages->buffer_pos = 0;

  binary_architecture_match = xmalloc (strlen (packages_architecture) + 2);
  sprintf (binary_architecture_match, " %s", packages_architecture);

}

void 
process_packages (packages_info *packages)
{

  char *source_version;
  int i;

  while (packages->buffer_pos < packages->file_length)
    {
      if (packages->buffer[packages->buffer_pos] == 'P')
	{
	  packages->name = read_line (packages->buffer, packages->file_length, &packages->buffer_pos);

	  debug (debug_packages, "Readline: \"%s\"", packages->name); 
	  
 	  if (!(strncmp ("Package: ", packages->name, 9))) 
	    {
	      strmove (packages->name, packages->name + 9);
	      debug (debug_packages, "main: Searching for version info for package \"%s\" in Packages-packages", 
		    packages->name);
	      packages->version = get_field (packages->buffer, packages->file_length, 
					     packages->buffer_pos, "Version: ", packages->name);
	      packages->source = get_field (packages->buffer, packages->file_length, 
					     packages->buffer_pos, "source: ", packages->name);
	      if (packages->source)
		{
		  source_version = strchr (packages->source, '(');
		  if (source_version)
		    {
		      debug (debug_packages, "naima: before [%s] %s", packages->name, packages->version);
		      source_version++;
		      i = 0;
		      while (*(source_version+i) != '\0' && *(source_version+i) != ')')
			i++;
		      xfree (packages->version);
		      packages->version = xmalloc (i+1);
		      strncpy (packages->version, source_version, i);
		      packages->version[i] = '\0';
		      debug (debug_packages, "naima: after [%s] %s", packages->name, packages->version);
		    }
		}

	      packages_ht_add (packages->name, packages->version);
	      xfree (packages->version);
	      xfree_if_non_null (packages->source);
	    }

	  xfree (packages->name);
	}
      skip_line(packages->buffer, &packages->buffer_pos, packages->file_length);
    }

}

/* Functions for the packages architecture's hash table */

void
packages_ht_create(void)
{

  packages_hash_table = g_hash_table_new (quinn_hash_function, quinn_key_compare_function);
  
  if (packages_hash_table == NULL)
    fubar (SYSERR, "Unable to create hash table for source packages.");

}

void 
packages_ht_add (char *package, char *version)
{

  char *key, *data;

  debug(debug_packages, "Adding \"%s\" [Version %s] to Packages hash table", package, version);

  key = xstrdup (package);
  data = xstrdup (version);

  g_hash_table_insert (packages_hash_table, key, data);

}

char *
packages_ht_lookup (char *package)
{

  char *binary_version;

  binary_version = g_hash_table_lookup (packages_hash_table, package);

  return (binary_version);

}

void
packages_ht_destroy (void)
{

  g_hash_table_foreach (packages_hash_table, packages_free_ht_entry, NULL);
  g_hash_table_destroy (packages_hash_table);
  
}

void
packages_free_ht_entry (gpointer key, 
			gpointer package, 
			gpointer cruft __attribute__ ((unused)))
{
  
  xfree (key);
  xfree (package);

}
