/* Quinn Diff, Determine which packages need recompiled for any given architecture */
/* Copyright (C) 1997-1999, James Troup <james@nocrew.org> */

/* This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <limits.h>
#include <fcntl.h>
#include <string.h>
#include <glib.h>
#include "common.h"
#include "error.h"
#include "main.h"
#include "output.h"
#include "parse_sources.h"
#include "utils.h"
#include "xmalloc.h"

GHashTable *output_hash_table;

void output_display_package (gpointer key, gpointer xpackage, gpointer cruft);
void output_free_ht_entry (gpointer key, gpointer package, gpointer cruft);

void
output_ht_create(void)
{

  output_hash_table = g_hash_table_new (quinn_hash_function, quinn_key_compare_function);
  
  if (output_hash_table == NULL)
    fubar (SYSERR, "Unable to create hash table for output buffer.");

}

void
output_ht_add (const sources_info *source, const char *binary_version,
	       const int compare_versions_result)
{

  output_info *package_to_add;

  debug(debug_output, "output_add: Adding \"%s\" (%s:%s) [Versions %s and %s] to output hash table", 
	source->name, source->priority, source->section, source->version, binary_version);

  /* Set up the output_info node for this package */

  package_to_add = xmalloc (sizeof(output_info));
  package_to_add->name = xstrdup (source->name);  
  package_to_add->source_version = xstrdup (source->version);  
  if (binary_version)
    package_to_add->binary_version = xstrdup (binary_version);
  else
    package_to_add->binary_version = NULL;
  package_to_add->priority = xstrdup (source->priority);  
  package_to_add->section = xstrdup (source->section);  
  package_to_add->compare_versions_result = compare_versions_result;
  package_to_add->errors = 0;
  
  /* And add it (or rather, a pointer to it) to the hash table */

  g_hash_table_insert (output_hash_table, package_to_add->name, package_to_add);

}

output_info *
output_ht_retrieve (char *package) 
{ 

  output_info *output_details; 
   
  output_details = g_hash_table_lookup (output_hash_table, package);

  return (output_details);

} 

void 
output_display_all(void)
{

  g_hash_table_foreach (output_hash_table, output_display_package, NULL);

}
 
void 
output_display_package (gpointer key __attribute__ ((unused)), 
			gpointer xpackage, 
			gpointer cruft __attribute__ ((unused)))
{

  output_info *package;
  char status[15];

  package = (output_info *) xpackage;

  if (!package->binary_version)
    strcpy (status, "uncompiled");
  else
    strcpy (status, "out-of-date");

  if (!package->binary_version || package->compare_versions_result < 0)
    printf ("%s/%s_%s.dsc [%s:%s]\n", 
	    package->section, package->name, 
	    package->source_version, package->priority, status);
  else if (warn_older && package->binary_version && package->compare_versions_result > 0)
    printf ("OLDER: %s/%s_%s.dsc [%s:%s] (Is %s on secondary)\n", 
	    package->section, package->name, package->source_version,
	    package->priority, status, package->binary_version);
  else if (package->compare_versions_result == 0)
    fubar (NONSYS, "[%s] %s/%s: %s and %s are the same? I'm very confused.  Tell James. (%d)\n", 
	   package->priority, package->section, package->name, 
	   package->source_version, package->binary_version, package->compare_versions_result);

}

void
output_ht_destroy(void)
{

  g_hash_table_foreach (output_hash_table, output_free_ht_entry, NULL);
  g_hash_table_destroy (output_hash_table);

}

void
output_free_ht_entry (gpointer key __attribute__ ((unused)), 
		      gpointer package, 
		      gpointer cruft __attribute__ ((unused)))
{
  
  xfree (((output_info *) package)->name);
  xfree (((output_info *) package)->source_version);
  xfree_if_non_null (((output_info *) package)->binary_version);
  xfree (((output_info *) package)->priority);
  xfree (((output_info *) package)->section);
  xfree (package);

}

