"""
	MODULE:		mp3info.py
	COPYRIGHT:	GNU GENERAL PUBLIC LICENSE, Version 2, June 1991
	VERSION:	0.1
	DESCRIPTION:	Allows the manipulation of MP3 TAG data, as well as reading
			MP3 internal information (bits, duration, etc)

	AUTHOR:		S.Venasse (Seamus.Venasse@polaris.ca)
	BASED:		Based upon the Delphi code from Predrag Supurovic (http://www.dv.co.yu/mgscript/mpgtools.htm)

	CHANGE LOG:
		20000920	S.Venasse	Initial coding


	EXAMPLE:

		mp3info = mp3info.MP3Info( 'My Music File.mp3' )
		mp3info.ReadData()

		print "Artist:", mp3info.MPEGData[ 'Artist' ]
		print "Title:", mp3info.MPEGData[ 'Title' ]
		print "Duration:", mp3info.DurationTime()
"""

import string
import os
import stat

class MP3Info:

	# Maximum amount of data to read
	FILE_DETECTION_PRECISION = 1024

	# Genre Definition
	MusicStyle = [
		"Blues", "Classic", "Country", 
		"Dance", "Disco", "Funk", 
		"Grunge", "Hip-Hop", "Jazz", 
		"Metal", "New", "Oldies", 
		"Other", "Pop", "R&B", 
		"Rap", "Reggae", "Rock", 
		"Techno", "Industrial", "Alternative", 
		"Ska", "Death", "Pranks", 
		"Soundtrack", "Euro-Techno", "Ambient", 
		"Trip-Hop", "Vocal", "Jazz+Funk", 
		"Fusion", "Trance", "Classical", 
		"Instrumental", "Acid", "House", 
		"Game", "Sound", "Gospel", 
		"Noise", "AlternRock", "Bass", 
		"Soul", "Punk", "Space", 
		"Meditative", "Instrumental", "Instrumental", 
		"Ethnic", "Gothic", "Darkwave", 
		"Techno-Industrial", "Electronic", "Pop-Folk", 
		"Eurodance", "Dream", "Southern", 
		"Comedy", "Cult", "Gangsta", 
		"Top", "Christian", "Pop/Funk", 
		"Jungle", "Native", "Cabaret", 
		"New", "Psychadelic", "Rave", 
		"Showtunes", "Trailer", "Lo-Fi", 
		"Tribal", "Acid", "Acid", 
		"Polka", "Retro", "Musical", 
		"Rock", "Hard", "Folk", 
		"Folk/Rock", "National", "Swing", 
		"Bebob", "Latin", "Revival", 
		"Celtic", "Bluegrass", "Avantgarde", 
		"Gothic", "Progressive", "Psychedelic", 
		"Symphonic", "Slow", "Big", 
		"Chorus", "Easy", "Acoustic", 
		"Humour", "Speech", "Chanson", 
		"Opera", "Chamber", "Sonata", 
		"Symphony", "Booty", "Primus", 
		"Porn", "Satire", "Slow", 
		"Club", "Tango", "Samba", 
		"Folklore", "Ballad", "Power Ballad",
		"Rhythmic Soul", "Freestyle", "Duet",
		"Punk Rock", "Drum Solo", "A capella",
		"Euro-House", "Dance Hall" ]

	# MPEG version indexes
	MPEG_VERSION_UNKNOWN = 0	# Unknown
	MPEG_VERSION_1 = 1		# Version 1
	MPEG_VERSION_2 = 2		# Version 2
	MPEG_VERSION_25 = 3		# Version 3

	# Description of MPEG version index
	MPEG_VERSIONS = [ 'Unknown', '1.0', '2.0', '2.5' ]

	# Channel mode (number of channels) in MPEG file
	MPEG_MD_STEREO = 0		# Stereo
	MPEG_MD_JOINT_STEREO = 1	# Stereo
	MPEG_MD_DUAL_CHANNEL = 2	# Stereo
	MPEG_MD_MONO = 3		# Mono

	# Description of number of channels
	MPEG_MODES = [ 'Stereo', 'Joint-Stereo', 'Dual-Channel', 'Single-Channel' ]

	# Description of layer value
	MPEG_LAYERS = [ 'Unknown', 'I', 'II', 'III' ]

	# Sampling rates table
	MPEG_SAMPLE_RATES = [
		[ 44100, 48000, 32000, 0 ],
		[ 22050, 24000, 16000, 0 ],
		[ 11025, 12000, 8000, 0 ]
	]

	# Predefined bitrate table
	MPEG_BIT_RATES = [
		[
			[ 0,32,64,96,128,160,192,224,256,288,320,352,384,416,448,0 ],
			[ 0,32,48,56, 64, 80, 96,112,128,160,192,224,256,320,384,0 ],
			[ 0,32,40,48, 56, 64, 80, 96,112,128,160,192,224,256,320,0 ]
		],
		[
			[ 0,32,48, 56, 64, 80, 96,112,128,144,160,176,192,224,256,0 ],
			[ 0, 8,16,24, 32, 40, 48, 56, 64, 80, 96, 112,128,144,160,0 ],
			[ 0, 8,16,24, 32, 40, 48, 56, 64, 80, 96, 112,128,144,160,0 ]
		],
		[
			[ 0,32,48, 56, 64, 80, 96,112,128,144,160,176,192,224,256,0 ],
			[ 0, 8,16,24, 32, 40, 48, 56, 64, 80, 96, 112,128,144,160,0 ],
			[ 0, 8,16,24, 32, 40, 48, 56, 64, 80, 96, 112,128,144,160,0 ]
		]
	]

	# Xing VBR header flags
	XH_FRAMES_FLAG = 1
	XH_BYTES_FLAG = 2
	XH_TOC_FLAG = 4
	XH_VBR_SCALE_FLAG = 8

	# Xing record structure
	tXHeadData = {
		'flags' : 0,			# from Xing header data
		'frames' : 0,			# total bit stream frames from Xing header data
		'bytes' : 0,			# total bit stream bytes from Xing header data
		'vbrscale' : 0			# encoded vbr scale from Xing header data
	}


	def __init__( self, fname ):
		"""
			Initialise the routine and define the filename
		"""

		self.FileName = fname


	def StripWhiteSpace( self, strng ):
		"""
			Removes both nulls and known whitespace characters
		"""

		while ( ( len( strng ) > 0 ) and ( strng[ -1 ] in string.whitespace + '\000' ) ):
			strng = strng[ : -1 ]

		return strng


	def Extract4b( self, inval, block ):
		"""
			Converts four bytes of data in buffer into a long integer
		"""

		buffer = inval[ block : block + 4 ]
		return ord( buffer[ 0 ] ) << 24 | ord( buffer[ 1 ] ) << 16 | ord( buffer[ 2 ] ) << 8 | ord( buffer[ 3 ] )


	def CalcFrameLength( self, Layer, SampleRate, Bitrate, Padding ):
		"""
			Calculate frame length based upon layer type
		"""

		if ( SampleRate > 0 ):
			if ( Layer == 1 ):
				return int( 12 * Bitrate * 1000 / SampleRate + ( Padding * 4 ) )
			else:
				return int( 144 * Bitrate * 1000 / SampleRate + Padding )


	def FrameHeaderValid( self, Data ):
		"""
			Checks several criteria in header to ensure that this is a 
			valid frame header
		"""

		return  ( ( Data[ 'FileLength' ] > 5 ) & ( Data[ 'Version' ] > 0 ) &
			( Data[ 'Layer' ] > 0 ) & ( Data[ 'Bitrate' ] >= -1 ) &
			( Data[ 'Bitrate' ] <> 0 ) & ( Data[ 'SampleRate' ] > 0 ) )


	def DecodeHeader( self, MPEGHeader, MPEGData ):
		"""
			Decode MPEG Frame Header and store data to tMPEGData fields.
		  	Return TRUE if header seems valid
		"""

		if ( ( MPEGHeader & 0xffe00000 ) == 0xffe00000 ):
			VersionIndex = ( MPEGHeader >> 19 ) & 0x3
			if ( VersionIndex == 0 ): MPEGData[ 'Version' ] = self.MPEG_VERSION_25
			elif ( VersionIndex == 1 ): MPEGData[ 'Version' ] = self.MPEG_VERSION_UNKNOWN
			elif ( VersionIndex == 2 ): MPEGData[ 'Version' ] = self.MPEG_VERSION_2
			elif ( VersionIndex == 3 ): MPEGData[ 'Version' ] = self.MPEG_VERSION_1

			# if version is known, read other data
			if ( MPEGData[ 'Version' ] <> self.MPEG_VERSION_UNKNOWN ):
				MPEGData[ 'Layer' ] = 4 - ( ( MPEGHeader >> 17 ) & 0x3 )
				if ( MPEGData[ 'Layer' ] > 3 ): MPEGData[ 'Layer' ] = 0

				BitrateIndex = ( MPEGHeader >> 12 ) & 0xf
				MPEGData[ 'SampleRate' ] = self.MPEG_SAMPLE_RATES[ MPEGData[ 'Version' ] - 1 ][ 
					( ( MPEGHeader >> 10 ) & 0x3)]
				MPEGData[ 'ErrorProtection' ] = ( ( MPEGHeader >> 16 ) & 0x1 ) == 1
				MPEGData[ 'Copyright' ] = ( ( MPEGHeader >> 3 ) & 0x1 ) == 1
				MPEGData[ 'Original' ] = ( ( MPEGHeader >> 2 ) & 0x1 ) == 1
				MPEGData[ 'Mode' ] = ( MPEGHeader >> 6 ) & 0x3
				MPEGData[ 'Padding' ] = ( ( MPEGHeader >> 9 ) & 0x1 ) == 1
				MPEGData[ 'Bitrate' ] = self.MPEG_BIT_RATES[ MPEGData[ 'Version' ] - 1 ][ MPEGData[ 'Layer' ] - 1 ][
					BitrateIndex ]

				if ( MPEGData[ 'Bitrate' ] == 0 ):
					MPEGData[ 'Duration' ] = 0
				else:
					MPEGData[ 'Duration' ] = ( MPEGData[ 'FileLength' ] * 8 ) / (
						( long( MPEGData[ 'Bitrate' ] ) * 1000 ) )

				MPEGData[ 'FrameLength' ] = self.CalcFrameLength( MPEGData[ 'Layer' ], MPEGData[ 'SampleRate' ],
					MPEGData[ 'Bitrate' ], MPEGData[ 'Padding' ] )

			return self.FrameHeaderValid( MPEGData )
		else:
			return 0


	def ResetData( self ):
		"""
			Builds a new MPEGData with all possible tags
		"""

		self.MPEGData = {
			'Header' : '',			# Should contain "TAG" if correct
			'Title'  : '',			# Song name
			'Artist' : '',			# Artist name
			'Album' : '',			# Album
			'Year' : '',			# Year
			'Comment' : '',			# Comment
			'Genre' : 0,			# Genre code
			'Track' : 0,			# Track number on album
			'Duration' : 0,			# Song duration
			'FileLength' : 0,		# File length
			'Version' : 0,			# MPEG audio version index ( 1 - Version 1
								# 2 - Version 2, 3 - Version 2.5,
							# 0 - Unknown
			'Layer' : 0,			# Layer (1, 2, 3, 0 - Unknown
			'SampleRate' : 0,		# Sampling rate in Hz
			'Bitrate' : 0,			# Bit rate
			'Mode' : 0,			# Number of channels (0 - Stereo,
							# 1 - Joint Stereo, 2 - Dual Channel,
							# 3 - Single Channel
			'Copyright' : 0,		# Copyrighted?
			'Original' : 0,			# Original?
			'ErrorProtection' : 0,		# Error protected?
			'Padding' : 0,			# Is frame padded?
			'FrameLength' : 0		# Total frame size, including CRC
		}


	def ReadData( self ):
		"""
			Reads TAG data and internal information
		"""

		self.ResetData()

		result = -1
		location = 0
		fData = self.MPEGData
		XingHeader = self.tXHeadData

		if ( os.path.exists( self.FileName ) ):
			f = open( self.FileName, 'r' )
			fData[ 'FileLength' ] = os.stat( self.FileName )[ stat.ST_SIZE ]
			while 1:
				f.seek( location )
				buffer = f.read( 4 )
				mp3hdr = ord(buffer[0])<<24|ord(buffer[1])<<16|ord(buffer[2])<<8|ord(buffer[3])

				while ( self.DecodeHeader( mp3hdr, fData ) == 0 ) & ( f.tell() <= fData[ 'FileLength' ] - 4 ):
					location = location + 1
					f.seek( location )
					buffer = f.read( 4 )
					mp3hdr = ord(buffer[0])<<24|ord(buffer[1])<<16|ord(buffer[2])<<8|ord(buffer[3])

				self.FirstValidFrameHeaderPosition = f.tell() - 4
				tempLong = fData[ 'FileLength' ] - self.FirstValidFrameHeaderPosition - \
					fData[ 'FrameLength' ] + ( 2 * fData[ 'ErrorProtection' ] )

				if ( self.FrameHeaderValid( fData ) == 0 ) | ( tempLong <= 0 ):
					self.ResetData()
					fData = self.MPEGData
					fData[ 'FileLength' ] = os.stat( self.FileName )[ stat.ST_SIZE ]
					self.FirstValidFrameHeaderPosition = fData[ 'FileLength' ] - 1
					result = -1
				else:
					# BINGO!!! This realy is MPEG audio file so we may proceed
					result = 0

					# check for Xing Variable BitRate info
					if ( fData[ 'Version' ] == 1 ):
						if ( fData[ 'Mode' ] <> 3 ):
							Deviation = 32 + 4
						else:
							Deviation = 17 + 4
					else:
						if ( fData[ 'Mode' ] <> 3 ):
							Deviation = 17 + 4
						else:
							Deviation = 9 + 4
					f.seek( self.FirstValidFrameHeaderPosition + Deviation )
					buffer = f.read( 4 )
					if ( buffer == 'Xing' ):
						fData[ 'Bitrate' ] = -1
						buffer = f.read( 116 )
						block = 0
						XingHeader[ 'flags' ] = self.Extract4b( buffer, block )
						block = block + 4
						if ( ( XingHeader[ 'flags' ] & self.XH_FRAMES_FLAG ) > 0 ):
							XingHeader[ 'frames' ] = self.Extract4b( buffer, block )
							block = block + 4
						else:
							XingHeader[ 'frames' ] = 0
						if ( ( XingHeader[ 'flags' ] & self.XH_BYTES_FLAG ) > 0 ):
							XingHeader[ 'bytes' ] = self.Extract4b( buffer, block )
							block = block + 4
						else:
							XingHeader[ 'bytes' ] = 0
						if ( ( XingHeader[ 'flags' ] & self.XH_TOC_FLAG ) > 0 ):
							block = block + 100
						if ( ( XingHeader[ 'flags' ] & self.XH_VBR_SCALE_FLAG ) > 0 ):
							XingHeader[ 'vbrscale' ] = self.Extract4b( buffer, block )
							block = block + 4
						else:
							XingHeader[ 'vbrscale' ] = 0

						fData[ 'Duration' ] = int(round(( float( 1152 ) / float( fData[ 'SampleRate' ] ) ) \
							 * XingHeader[ 'frames' ] ) )
						fData[ 'FrameLength' ] = divmod( fData[ 'FileLength' ], XingHeader[ 'frames' ] )[0]

					# read TAG if it exists
					if ( fData[ 'FileLength' ] > 128 ):
						f.seek( fData[ 'FileLength' ] - 128 )
						buffer = f.read( 128 )

						if ( string.lower( buffer[ 0 : 3 ] ) == 'tag' ):
							fData[ 'Header' ] = buffer[ 0 : 3 ]
							fData[ 'Title' ] = self.StripWhiteSpace( buffer[ 3 : 33 ] )
							fData[ 'Artist' ] = self.StripWhiteSpace( buffer[ 33 : 63 ] )
							fData[ 'Album' ] = self.StripWhiteSpace( buffer[ 63 : 93 ] )
							fData[ 'Year' ] = self.StripWhiteSpace( buffer[ 93 : 97 ] )
							fData[ 'Comment' ] = buffer[ 97 : 127 ]
							fData[ 'Genre' ] = ord( buffer[ 127 ] )
							if ( ( fData[ 'Comment' ][ -2 ] == '\000' ) and \
								( fData[ 'Comment' ][ -1 ] != '\000' ) ):
								fData[ 'Track' ] = ord( fData[ 'Comment' ][ 29 ] )
								fData[ 'Comment' ] = self.StripWhiteSpace( fData[ 'Comment' ][ :-2])
							else:
								fData[ 'Comment' ] = self.StripWhiteSpace( fData[ 'Comment' ] )

				if ( self.FrameHeaderValid( fData ) ):
					break

			f.close()

		self.MPEGData = fData
		return result


	def WriteTag( self ):
		"""
			Write the MPEGData information as TAG data at the end of the file
		"""

		tmp = '%(Header)s%(Title)-30s%(Artist)-30s%(Album)-30s%(Year)-4s%(Comment)-30s%(Genre)c' % ( self.MPEGData )
		if ( self.MPEGData[ 'Track' ] > 0 ):
			tmp = tmp[ 0 : 125 ] + chr( 0 ) + chr( self.MPEGData[ 'Track' ] ) + tmp[ 127 ]

		if ( os.path.exists( self.FileName ) ):
			f = open( self.FileName, 'a+' )
			f.seek( self.MPEGData[ 'FileLength' ] - 128 )
			f.truncate()
			f.write( tmp )
			f.close()


	def RemoveTag( self ):
		"""
			Removes the tag from the MP3 file
		"""

		if ( os.path.exists( self.FileName ) ):
			f = open( self.FileName, 'a+' )
			f.seek( self.MPEGData[ 'FileLength' ] - 128 )
			f.truncate()
			f.close()

	def GenreStr( self ):
		"""
			Returns the string value of the genre
		"""

		if ( self.MPEGData[ 'Genre' ] < len( self.MusicStyle ) ):
			return self.MusicStyle[ self.MPEGData[ 'Genre' ] ]
		else:
			return 'Unknown'


	def ModeStr( self ):
		"""
			Returns the string value of the mode
		"""

		return self.MPEG_MODES[ self.MPEGData[ 'Mode' ] ]


	def DurationTime( self ):
		"""
			Returns a string value of the duration of the song
		"""

		return '%02d:%02d' % divmod( self.MPEGData[ 'Duration' ], 60 )


	def VersionStr( self ):
		"""
			Returns a string value of the MP3 encoding version
		"""

		return self.MPEG_VERSIONS[ self.MPEGData[ 'Version' ] ]


	def LayerStr( self ):
		"""
			Returns a string value of the MP3 layer type
		"""

		return self.MPEG_LAYERS[ self.MPEGData[ 'Layer' ] ]


	def CopyrightStr( self, CopyrightTrue, CopyrightFalse ):
		"""
			Returns a choice of strings based upon value of copyright
		"""

		if ( self.MPEGData[ 'Copyright' ] ):
			return CopyrightTrue
		else:
			return CopyrightFalse


	def OriginalStr( self, OriginalTrue, OriginalFalse ):
		"""
			Returns a choice of strings based upon value of original
		"""

		if ( self.MPEGData[ 'Original' ] ):
			return OriginalTrue
		else:
			return OriginalFalse


	def ErrorProtectionStr( self, ErrorProtectionTrue, ErrorProtectionFalse ):
		"""
			Returns a choice of strings based upon value of error protection
		"""

		if ( self.MPEGData[ 'ErrorProtection' ] ):
			return ErrorProtectionTrue
		else:
			return ErrorProtectionFalse


