/*
 * pfmon_smpl.c - sampling support for pfmon
 *
 * Copyright (C) 2001-2003 Hewlett-Packard Co
 * Contributed by Stephane Eranian <eranian@hpl.hp.com>
 *
 * This file is part of pfmon, a sample tool to measure performance 
 * of applications on Linux/ia64.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
 * 02111-1307 USA
 */

#include "pfmon.h"

#include <ctype.h>
#include <fcntl.h>
#include <time.h>

extern pfmon_smpl_module_t raw_ia64_smpl_module;
extern pfmon_smpl_module_t compact_ia64_smpl_module;
extern pfmon_smpl_module_t detailed_itanium_smpl_module;
extern pfmon_smpl_module_t detailed_itanium2_smpl_module;
extern pfmon_smpl_module_t btb_ia64_smpl_module;
extern pfmon_smpl_module_t example_ia64_smpl_module;

static pfmon_smpl_module_t *smpl_modules[]={

#ifdef CONFIG_PFMON_SMPL_MOD_DET_ITA
	&detailed_itanium_smpl_module,	/* must be first for Itanium */
#endif
#ifdef CONFIG_PFMON_SMPL_MOD_DET_ITA2
	&detailed_itanium2_smpl_module,	/* must be first for Itanium2 */
#endif
#ifdef CONFIG_PFMON_SMPL_MOD_RAW_IA64
	&raw_ia64_smpl_module,
#endif
#ifdef CONFIG_PFMON_SMPL_MOD_COMPACT_IA64
	&compact_ia64_smpl_module,
#endif
#ifdef CONFIG_PFMON_SMPL_MOD_BTB_IA64
	&btb_ia64_smpl_module,
#endif
#ifdef CONFIG_PFMON_SMPL_MOD_EXAMPLE_IA64
	&example_ia64_smpl_module,
#endif
	NULL
};

void
pfmon_smpl_initialize(void)
{
	pfmon_smpl_module_t *mod;
	/*
	 * pick a default sampling output format
	 */
	if (pfmon_find_smpl_module(NULL, &mod, 0) != 0) {
		warning("no sampling module available for this PMU model\n");
		return;
	}
	if (mod->initialize_module && (*mod->initialize_module)() != 0) {
		fatal_error("failed to intialize sampling module%s\n", mod->name);
	}
	options.smpl_mod = mod;
}

/*
 * sdesc is NULL when aggregation is used
 */
static void
print_smpl_header(pfmon_sdesc_t *sdesc, pfmon_smpl_desc_t *csmpl)
{
	FILE *fp = csmpl->smpl_fp;
	unsigned long msk;
	unsigned int i, j;
	char name[PFMON_MAX_EVTNAME_LEN];

	print_standard_header(fp, csmpl->cpu, sdesc);

	for(i=0; i < options.monitor_count; i++) {

		if ((options.short_rates[i].flags & PFMON_RATE_VAL_SET) == 0) continue;

		pfm_get_event_name(options.events[i].event, name, PFMON_MAX_EVTNAME_LEN);

		fprintf(fp, "# recorded PMDs when %s overflows: ", name);

		msk = options.smpl_pmds[i];

		for(j=0; msk; msk>>=1, j++) if (msk & 0x1) fprintf(fp, "PMD%u ", j);

		fputc('\n', fp);
	}

	fprintf(fp, "#\n# short sampling rates (base/mask/seed):\n");

	for(i=0; i < options.monitor_count; i++) {
		pfm_get_event_name(options.events[i].event, name, PFMON_MAX_EVTNAME_LEN);

		if (options.short_rates[i].flags & PFMON_RATE_VAL_SET) {
			fprintf(fp, "#\t%s %"PRIu64,
				name,
				-options.short_rates[i].value);

			if (options.short_rates[i].flags & PFMON_RATE_MASK_SET) {
				fprintf(fp, "/0x%"PRIx64"/%"PRIu64, 
					options.short_rates[i].mask,
					options.short_rates[i].seed);
			}
			fputc('\n', fp);
		} else {
			fprintf(fp, "#\t%s none\n", name); 
		}
	}

	fprintf(fp, "#\n# long sampling rates (base/mask/seed):\n");
	for(i=0; i < options.monitor_count; i++) {
		pfm_get_event_name(options.events[i].event, name, PFMON_MAX_EVTNAME_LEN);

		if (options.long_rates[i].flags & PFMON_RATE_VAL_SET) {
			fprintf(fp, "#\t%s %"PRIu64,
				name,
				-options.long_rates[i].value);

			if (options.long_rates[i].flags & PFMON_RATE_MASK_SET) {
				fprintf(fp, "/0x%"PRIx64"/%"PRIu64, 
					options.long_rates[i].mask,
					options.long_rates[i].seed);
			}
			fputc('\n', fp);
		} else {
			fprintf(fp, "#\t%s none\n", name); 
		}
	}
	fprintf(fp, "#\n#\n");

	/* 
	 * invoke additional header printing routine if defined
	 */
	if (options.smpl_mod->print_header)
		(*options.smpl_mod->print_header)(csmpl);

	fprintf(fp, "#\n#\n");
}

static int
__pfmon_process_smpl_buffer(pfmon_smpl_desc_t *csmpl)
{
	pfmon_smpl_module_t *mod = options.smpl_mod;
	int ret;
	int need_lock;
	static pthread_mutex_t smpl_results_lock = PTHREAD_MUTEX_INITIALIZER;

	need_lock = csmpl->smpl_fp == stdout ? 1 : 0;
	/*
	 * if necessary, verify we are not parsing the same samples again
	 */
	if (mod->check_new_samples) {
		ret = (*mod->check_new_samples)(csmpl);
		DPRINT(("check_new_samples=%d\n", ret));
		if (ret) return ret;
	}
	/*
	 * in case we output directly to the screen we synchronize
	 * to avoid concurrent printf()
	 */
	if (need_lock) {
		pthread_mutex_lock(&smpl_results_lock);
	}

	ret = (*options.smpl_mod->process_samples)(csmpl);

	if (need_lock) {
		pthread_mutex_unlock(&smpl_results_lock);
	}
	return ret;
}

int
pfmon_setup_common_sampling_output(pfmon_smpl_desc_t *csmpl)
{
        FILE *fp = stdout;
	char *filename;

	filename = options.smpl_outfile;

	if (options.opt_use_smpl == 0 || options.opt_aggr == 0) return 0;

        if (options.smpl_outfile) {
                fp = fopen(filename, "w");
                if (fp == NULL) {
                        warning("cannot create sampling output file %s: %s\n", filename, strerror(errno));
                        return -1;
                }
                if (options.opt_syst_wide)
			vbprintf("CPU%-3u sampling results will be in file \"%s\"\n", csmpl->cpu, filename);
		else
			vbprintf("sampling results will be in file \"%s\"\n", filename);
        }

        csmpl->smpl_fp    = fp;
	csmpl->last_ovfl  = ~0ULL;
	csmpl->last_count = ~0ULL;

	if (options.opt_with_header) print_smpl_header(NULL, csmpl);

        return 0;
}

int
pfmon_setup_private_sampling_output(pfmon_sdesc_t *sdesc)
{
        FILE *fp = stdout;
	pfmon_smpl_desc_t *csmpl = &sdesc->csmpl;
        char filename[PFMON_MAX_FILENAME_LEN];

	if (options.opt_use_smpl == 0) return 0;
	
	/*
	 * We must initialize last_ovfl which is private
	 * to each session. the rest (fp, entry_count) is
	 * shared from the aggregate structure.
	 */
	if (options.opt_aggr == 1) {
		csmpl->last_ovfl  = ~0ULL;
		csmpl->last_count = ~0ULL;
		return 0;
	}

	/*
	 * check version
	 */
	if (options.smpl_mod->check_version) {
		if ((*options.smpl_mod->check_version)(csmpl) == -1) return -1;
	} 

	if ( options.opt_addr2sym 
	  && pfmon_syms_hash_alloc(PFMON_DFL_SYM_HASH_SIZE, PFMON_DFL_SYM_ENTRIES, &csmpl->sym_hash)) {
		warning("cannot allocate sym hash table\n");
		return -1;
	}

        if (options.smpl_outfile) {
                if (is_regular_file(options.smpl_outfile)) {
                    	if (options.opt_syst_wide) {
				sprintf(filename, "%s.cpu%u", options.smpl_outfile, csmpl->cpu);
			} else {
				if (options.opt_follows) {
					if (options.opt_split_exec) {
						sprintf(filename, "%s.%d.%d.%u", 
							options.smpl_outfile, 
							sdesc->pid, 
							sdesc->tid, 
							sdesc->exec_count);
					} else {
						sprintf(filename, "%s.%d.%d", 
							options.smpl_outfile, 
							sdesc->pid, 
							sdesc->tid);
					}
				} else {
					sprintf(filename, "%s", options.smpl_outfile);
				}
			}
                } else {
                        strcpy(filename, options.smpl_outfile);
                }

                fp = fopen(filename, "w");
                if (fp == NULL) {
                        warning("cannot create private sampling output file %s: %s\n", filename, strerror(errno));
			if (csmpl->sym_hash) pfmon_syms_hash_free(csmpl->sym_hash);
                        return -1;
                }
                if (options.opt_syst_wide) 
			vbprintf("CPU%-3u results will be in file \"%s\"\n", csmpl->cpu, filename);
		else
			vbprintf("[%d] sampling results will be in file \"%s\"\n", sdesc->tid, filename);
        }

        csmpl->smpl_fp    = fp;
	csmpl->last_ovfl  = ~0ULL;
	csmpl->last_count = ~0ULL;

	if (options.opt_with_header) print_smpl_header(sdesc, csmpl);

        return 0;
}

void
pfmon_close_sampling_output(pfmon_smpl_desc_t *csmpl, pid_t tid, unsigned int cpu)
{
	uint64_t count;

	if (options.opt_use_smpl == 0) return;

	if (options.smpl_mod->terminate_session) {
		options.smpl_mod->terminate_session(csmpl);
	}

	if (csmpl->smpl_fp && csmpl->smpl_fp != stdout) {
		count  = csmpl->entry_count;
		if (options.opt_aggr) {
			vbprintf("%"PRIu64" samples collected\n", count);
		} else if (options.opt_syst_wide) {
			vbprintf("CPU%u %"PRIu64" samples collected\n", cpu, count);
		} else {
			vbprintf("[%d] %"PRIu64" samples collected (%"PRIu64 " buffer overflows)\n", 
				tid, 
				count, 
				csmpl->last_ovfl);
		}
		fclose(csmpl->smpl_fp);
	}

	if (csmpl->sym_hash) pfmon_syms_hash_free(csmpl->sym_hash);
}

static int
pfmon_check_smpl_module(pfm_uuid_t uuid)
{
	FILE *fp;
	char *p;
	size_t len;
	char uuid_str[64];
	char buffer[256];

	pfm_uuid2str(uuid, sizeof(uuid_str), uuid_str);

	len = strlen(uuid_str);

	fp = fopen("/proc/perfmon", "r");
	if (fp == NULL) {
		warning("unable to open /proc/perfmon\n");
		return -1; /* will fail later on if module not present */
	}
	for (;;) {
		p  = fgets(buffer, sizeof(buffer)-1, fp);

		if (p == NULL) break;

		if (strncmp("format", buffer, 6)) continue;
		p = strchr(buffer, ':');
		if (p == NULL) continue;
		DPRINT(("uuid_str:%s:\nuuid_p:%s:\n", uuid_str, p+2));
		if (!strncmp(p+2, uuid_str, len)) {
			fclose(fp);
			return 0;
		}

	}
	fclose(fp);
	warning("cannot find requested kernel sampling format : %s\n", uuid_str);
	return -1;
}

void 
pfmon_setup_sampling_rates(char *long_smpl_rates, char *short_smpl_rates, char *smpl_periods_random)
{
	unsigned int cnt = 0;
	int ret;

	if (smpl_periods_random && long_smpl_rates == NULL && short_smpl_rates == NULL)
		fatal_error("no short or long periods provided to apply randomization\n");

	if (long_smpl_rates) {
		/*
		 * in case not all rates are specified, they will default to zero, i.e. no sampling
		 * on this counter
		 */
		ret = gen_smpl_rates(long_smpl_rates, options.monitor_count, options.long_rates, &cnt);
		if (ret == -1) fatal_error("cannot set long sampling rates\n");

		/*
		 * in case the short period rates were not specified, we copy them from the long period rates
		 */
		if (short_smpl_rates == NULL) {
			memcpy(options.short_rates, options.long_rates, cnt*sizeof(pfmon_smpl_rate_t));
		}
		if (cnt) options.opt_use_smpl = 1;
	}

	if (short_smpl_rates) {
		/*
		 * in case not all rates are specified, they will default to zero, i.e. no sampling
		 * on this counter
		 */
		ret = gen_smpl_rates(short_smpl_rates, options.monitor_count, options.short_rates, &cnt);
		if (ret == -1) fatal_error("cannot set short sampling rates\n");


		/*
		 * in case the long period rates were not specified, we copy them from the short period rates
		 */
		if (long_smpl_rates == NULL) {
			memcpy(options.long_rates, options.short_rates, cnt*sizeof(pfmon_smpl_rate_t));
		}
		if (cnt) options.opt_use_smpl = 1;
	}

	/* 
	 * nothing else to do, we are not sampling
	 */
	if (options.opt_use_smpl == 0) return;
	
	/*
	 * some extra sanity checks now that we know we are sampling
	 */
	if (options.opt_addr2sym && options.opt_syst_wide)
		warning("only kernel symbols are resolved in system-wide mode\n");

	if (options.opt_follow_exec && options.opt_addr2sym)
		fatal_error("this version of pfmon cannot resolve symbols if follow-exec is enabled\n");

	/*
	 * some sanity checks
	 */
	if (options.smpl_mod == NULL) 
		fatal_error("error, no sampling module select\n");

	if (pfmon_check_smpl_module(options.smpl_mod->uuid)) fatal_error("cannot proceed");

	if (options.smpl_mod->process_samples == NULL)
		fatal_error("sampling module %s without sampling process entry point\n", options.smpl_mod->name);

	if (smpl_periods_random) {
		/*
		 * we place the masks/seeds into the long rates table. It is always defined
		 */
		ret = gen_smpl_randomization(smpl_periods_random, options.monitor_count, options.long_rates, &cnt);
		if (ret == -1) fatal_error("cannot setup randomization parameters\n");
		
		/* propagate  mask/seed to short rates */
		for(cnt = 0; cnt < options.monitor_count; cnt++) {
			options.short_rates[cnt].mask  = options.long_rates[cnt].mask;
			options.short_rates[cnt].seed  = options.long_rates[cnt].seed;
			options.short_rates[cnt].flags = options.long_rates[cnt].flags;

			if (options.long_rates[cnt].flags & PFMON_RATE_MASK_SET) {
				options.events[cnt].flags |= PFMON_MONITOR_FL_RANDOMIZE;
			}
		}

	}

	vbprintf("using %s sampling module\n", options.smpl_mod->name);

	if (options.opt_verbose) {
		unsigned int i;

		vbprintf("long  sampling periods(val/mask/seed): ");
		for(i=0; i < options.monitor_count; i++) {
			vbprintf("%"PRIu64"/0x%"PRIx64"/%"PRIu64, 
				-options.long_rates[i].value,
				options.long_rates[i].mask,
				options.long_rates[i].seed);
		}
		vbprintf("\nshort sampling periods(val/mask/seed): ");
		for(i=0; i < options.monitor_count; i++) {
			vbprintf("%"PRIu64"/0x%"PRIx64"/%"PRIu64, 
				-options.short_rates[i].value,
				options.short_rates[i].mask,
				options.short_rates[i].seed);
		}
		vbprintf("\n");
	}
}

/*
 * look for a matching sampling format.
 * The name and CPU model must match.
 *
 * if ignore_cpu is true, then we don't check if the host CPU matches
 */
int
pfmon_find_smpl_module(char *name, pfmon_smpl_module_t **mod, int ignore_cpu)
{
	pfmon_smpl_module_t **p;
	unsigned long mask, gen_mask;
	int type, ret;

	ret = pfm_get_pmu_type(&type);
	if (ret != PFMLIB_SUCCESS) {
		DPRINT(("library not initialized\n"));
		return -1;
	}

	mask     = PFMON_PMU_MASK(type);
	gen_mask = PFMON_PMU_MASK(PFMLIB_GENERIC_IA64_PMU);

	for(p = smpl_modules; *p ; p++) {

		if (name == NULL || !strcmp(name, (*p)->name)) {
			if (ignore_cpu == 0 && (*p)->pmu_mask != gen_mask && ((*p)->pmu_mask & mask) == 0) {
				continue;
			}
			*mod = *p;
			return 0;
		}
	}
printf("name=%p ret=-1\n", name);
	return -1;
}

void
pfmon_list_smpl_modules(void)
{
	pfmon_smpl_module_t **p = smpl_modules;
	unsigned long mask, gen_mask;
	int type;

	pfm_get_pmu_type(&type);
	mask     = PFMON_PMU_MASK(type);
	gen_mask = PFMON_PMU_MASK(PFMLIB_GENERIC_IA64_PMU);

	printf("supported sampling modules: ");
	while (*p) {	
		if ((*p)->pmu_mask == gen_mask || ((*p)->pmu_mask & mask)) printf("[%s] ", (*p)->name);
		p++;
	}
	printf("\n");
}

void
pfmon_smpl_module_info(pfmon_smpl_module_t *fmt)
{
	unsigned long m;
	int i;
	char name[PFMON_MAX_EVTNAME_LEN];

	printf("name        : %s\n"
	       "description : %s\n",
		fmt->name,
		fmt->description);

	printf("PMU models  : ");
	m = fmt->pmu_mask;
	for(i=0; m; i++, m >>=1) {
		if (m & 0x1) {
			pfm_get_pmu_name_bytype(i, name,PFMON_MAX_EVTNAME_LEN);
			printf("[%s] ", name);
		}
	}
	printf("\n");
	if (fmt->show_options) {
		printf("options     :\n");
		fmt->show_options();
	}
}

int
pfmon_reset_sampling(pfmon_sdesc_t *sdesc)
{
	int ret;

	/* for debug purposes */
	if (options.opt_block_restart) {
		printf("<press a key to resume monitoring>\n");
		getchar();
	}
	LOCK_SDESC(sdesc);

	if (sdesc->csmpl.processing) {
		vbprintf("[%d] race conditdion with process_samples in reset_sampling\n", sdesc->tid);
	}

	sdesc->csmpl.entry_count = 0;
	sdesc->csmpl.last_ovfl   = ~0ULL;
	sdesc->csmpl.last_count  = ~0ULL;

	ret = perfmonctl(sdesc->ctxid, PFM_RESTART, NULL, 0);
	if (ret == -1) {
		warning("perfmonctl error PFM_RESTART [%d] %s \n", sdesc->tid, strerror(errno));
 	}
	UNLOCK_SDESC(sdesc);
 	return ret;
}

int
pfmon_process_smpl_buf(pfmon_sdesc_t *sdesc, int need_restart)
{
	int ret;

	LOCK_SDESC(sdesc);

	sdesc->csmpl.processing = 1;
	ret = __pfmon_process_smpl_buffer(&sdesc->csmpl);

	DPRINT(("process_smpl_buf: pid=%d restart=%d ret=%d\n", getpid(), need_restart, ret));

	if (need_restart) {
		if (options.opt_block_restart) {
			printf("<press a key to resume monitoring>\n");
			getchar();
		}
		ret = perfmonctl(sdesc->ctxid, PFM_RESTART, NULL, 0);
		if (ret == -1) {
			warning("perfmonctl error PFM_RESTART [%d] %s\n", sdesc->tid, strerror(errno));
		}
	}
	sdesc->csmpl.processing = 0;
	UNLOCK_SDESC(sdesc);
	return ret;
}


void
pfmon_smpl_mod_usage(void)
{
	pfmon_smpl_module_t **mod;
	unsigned long mask, gen_mask;
	int type, ret;

	ret = pfm_get_pmu_type(&type);
	if (ret != PFMLIB_SUCCESS) {
		fatal_error("library not initialized\n");
	}
	mask     = PFMON_PMU_MASK(type);
	gen_mask = PFMON_PMU_MASK(PFMLIB_GENERIC_IA64_PMU);

	for(mod = smpl_modules; *mod; mod++) {

		if ((*mod)->pmu_mask != gen_mask && ((*mod)->pmu_mask & mask) == 0) continue;

		if ((*mod)->show_options) {
			printf("options for \"%s\" sampling format:\n", (*mod)->name);
			(*(*mod)->show_options)();
		}
	}
}

