//  This file is part of ff3d - http://www.freefem.org/ff3d
//  Copyright (C) 2001, 2002, 2003 Stphane Del Pino

//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2, or (at your option)
//  any later version.

//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.

//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software Foundation,
//  Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  

//  $Id: BoundaryConditionExpression.cpp,v 1.3 2004/07/11 22:37:07 delpinux Exp $

#include <BoundaryConditionExpression.hpp>
#include <FunctionExpression.hpp>
#include <BoundaryCondition.hpp>

#include <UserFunction.hpp>
#include <Dirichlet.hpp>
#include <Fourrier.hpp>
#include <Neumann.hpp>

#include <UnknownListExpression.hpp>
#include <Information.hpp>

ReferenceCounting<BoundaryCondition>
BoundaryConditionExpression::boundaryCondition()
{
  return __boundaryCondition;
}


BoundaryConditionExpression::
BoundaryConditionExpression(const BoundaryConditionExpression& e)
  : Expression(e),
    __boundaryCondition(e.__boundaryCondition),
    __boundary(e.__boundary),
    __unknown(e.__unknown),
    __boundaryConditionType(e.__boundaryConditionType)
{
  ;
}

BoundaryConditionExpression::
BoundaryConditionExpression(ReferenceCounting<BoundaryExpression> boundary,
			    ReferenceCounting<FunctionVariable> unknown,
			    const BoundaryConditionExpression::BoundaryConditionType& t)
  : Expression(Expression::boundaryCondition),
    __boundary(boundary),
    __unknown(unknown),
    __boundaryConditionType(t)
{
  ;
}

BoundaryConditionExpression::~BoundaryConditionExpression()
{
  ;
}



std::ostream& BoundaryConditionExpressionDirichlet::put(std::ostream& os) const
{
  os << "\t\t" << (*__unknown).name() << " = " << (*__g)
     << " on " << (*__boundary) << '\n';
  return os;
}

void BoundaryConditionExpressionDirichlet::execute()
{
  (*__boundary).execute();
  (*__g).execute();

  ReferenceCounting<UserFunction> g = new UserFunctionLanguage(__g);

  ReferenceCounting<UnknownListExpression> L
    = Information::instance().getUnknownList();

  size_t n = (*L).number(__unknown);

  ReferenceCounting<PDECondition> D = new Dirichlet(g,n);

  __boundaryCondition
      = new BoundaryCondition(D, (*__boundary).boundary());
}

BoundaryConditionExpressionDirichlet::
BoundaryConditionExpressionDirichlet(ReferenceCounting<FunctionVariable> unknown,
				     ReferenceCounting<FunctionExpression> g,
				     ReferenceCounting<BoundaryExpression> boundary)
  : BoundaryConditionExpression(boundary,
				unknown,
				BoundaryConditionExpression::dirichlet),
    __g(g)
{
  ;
}

BoundaryConditionExpressionDirichlet::
BoundaryConditionExpressionDirichlet(const BoundaryConditionExpressionDirichlet& d)
  : BoundaryConditionExpression(d),
    __g(d.__g)
{
  ;
}

BoundaryConditionExpressionDirichlet::
~BoundaryConditionExpressionDirichlet()
{
  ;
}


std::ostream& BoundaryConditionExpressionNeumann::put(std::ostream& os) const
{
  os << "\t\tdnu(" << (*__unknown).name() << ") = "<< (*__g)
     << " on " << (*__boundary) << '\n';
  return os;
}

void BoundaryConditionExpressionNeumann::execute()
{
  (*__boundary).execute();
  (*__g).execute();
  ReferenceCounting<UserFunction> g = new UserFunctionLanguage(__g);

  ReferenceCounting<UnknownListExpression> L
    = Information::instance().getUnknownList();

  size_t n = (*L).number(__unknown);

  ReferenceCounting<PDECondition> N = new Neumann(g,n);

  __boundaryCondition
      = new BoundaryCondition(N, (*__boundary).boundary());
}

BoundaryConditionExpressionNeumann::
BoundaryConditionExpressionNeumann(ReferenceCounting<FunctionVariable> unknown,
				   ReferenceCounting<FunctionExpression> g,
				   ReferenceCounting<BoundaryExpression> boundary)
  : BoundaryConditionExpression(boundary,
				unknown,
				BoundaryConditionExpression::neumann),
    __g(g)
{
  ;
}

BoundaryConditionExpressionNeumann::
BoundaryConditionExpressionNeumann(const BoundaryConditionExpressionNeumann& d)
  : BoundaryConditionExpression(d),
    __g(d.__g)
{
  ;
}

BoundaryConditionExpressionNeumann::
~BoundaryConditionExpressionNeumann()
{
  ;
}

std::ostream& BoundaryConditionExpressionFourrier::put(std::ostream& os) const
{
  os << "\t\t" << (*__Alpha) << '*' << (*__unknown).name()
     <<" + dnu(" << (*__unknown).name() << ") = " << (*__g)
     << " on " << (*__boundary) << '\n';
  return os;
}

void BoundaryConditionExpressionFourrier::execute()
{
  (*__boundary).execute();
  (*__Alpha).execute();
  (*__g).execute();
  ReferenceCounting<UserFunction> alpha = new UserFunctionLanguage(__Alpha);
  ReferenceCounting<UserFunction> g = new UserFunctionLanguage(__g);

  ReferenceCounting<UnknownListExpression> L
    = Information::instance().getUnknownList();

  size_t n = (*L).number(__unknown);

  ReferenceCounting<PDECondition> F = new Fourrier(alpha,g,n);

  __boundaryCondition
      = new BoundaryCondition(F, (*__boundary).boundary());
}

BoundaryConditionExpressionFourrier::
BoundaryConditionExpressionFourrier(ReferenceCounting<FunctionVariable> unknown,
				    ReferenceCounting<FunctionExpression> alpha,
				    ReferenceCounting<FunctionExpression> g,
				    ReferenceCounting<BoundaryExpression> boundary)
  : BoundaryConditionExpression(boundary,
				unknown,
				BoundaryConditionExpression::neumann),
    __Alpha(alpha),
    __g(g)
{
  ;
}

BoundaryConditionExpressionFourrier::
BoundaryConditionExpressionFourrier(const BoundaryConditionExpressionFourrier& f)
  : BoundaryConditionExpression(f),
    __Alpha(f.__Alpha),
    __g(f.__g)
{
  ;
}

BoundaryConditionExpressionFourrier::~BoundaryConditionExpressionFourrier()
{
  ;
}

