/*
 * Copyright (c) 2002-2004 Kungliga Tekniska Hgskolan
 * (Royal Institute of Technology, Stockholm, Sweden).
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * 3. Neither the name of the Institute nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE INSTITUTE AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE INSTITUTE OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */
/*
 * NNPFS operations.
 */

#include <nnpfs/nnpfs_locl.h>
#include <nnpfs/nnpfs_message.h>
#include <nnpfs/nnpfs_common.h>
#include <nnpfs/nnpfs_fs.h>
#include <nnpfs/nnpfs_dev.h>
#include <nnpfs/nnpfs_deb.h>
#include <nnpfs/nnpfs_syscalls.h>
#include <nnpfs/nnpfs_vnodeops.h>

RCSID("$Id: nnpfs_vnodeops-netbsd.c,v 1.15 2004/02/07 23:25:41 lha Exp $");

/*
 * This is the UBC version of the io function (read/write/getpages/putpages)
 * for NetBSD 1.6 (conditioned on later then 1.5AB).
 */

#if __NetBSD_Version__ >= 105280000

extern vop_t **nnpfs_vnodeop_p;

#if __NetBSD_Version__ >= 106150000
#define NNPFS_GOP_SIZE(f) \
	f(struct vnode *vp, off_t size, off_t *eobp, int flag)
#else
#define NNPFS_GOP_SIZE(f) \
	f(struct vnode *vp, off_t size, off_t *eobp)
#endif

static void NNPFS_GOP_SIZE(nnpfs_gop_size);
static int nnpfs_gop_alloc(struct vnode *, off_t, off_t, int, struct ucred *);

struct genfs_ops nnpfs_genfsops = {
    nnpfs_gop_size,
    nnpfs_gop_alloc,
    genfs_gop_write,
};

static void
NNPFS_GOP_SIZE(nnpfs_gop_size)
{
    NNPFSDEB(XDEBVNOPS, ("nnpfs_gop_size: %p\n", vp));

    *eobp = MAX(size, vp->v_size);
}

static int
nnpfs_gop_alloc(struct vnode *vp, off_t off, off_t len, int flags,
	      struct ucred *cred)
{
    return 0;
}

/*
 *
 */

static int
nnpfs_netbsd_bmap(struct vop_bmap_args *ap)
{
    /* {
	struct vnode *a_vp;
	daddr_t a_bn;
	struct vnode **a_vpp;
	daddr_t *a_bnp;
	int *a_runp;
    } */
    struct vnode *vp = ap->a_vp;

    NNPFSDEB(XDEBVNOPS, ("nnpfs_netbsd_bmap: %p\n", vp));

    if (DATA_FROM_VNODE(vp) == NULL)
	*ap->a_bnp = -1;
    else
	*ap->a_bnp = ap->a_bn;

    if (ap->a_vpp)
	*ap->a_vpp = DATA_FROM_VNODE(vp);
    if (ap->a_runp)
	*ap->a_runp = 0;
    return 0;
}

static int
nnpfs_netbsd_strategy(struct vop_strategy_args *ap)
{
    /* {
        struct buf *a_bp;
    } */
    struct buf *bp = ap->a_bp;
    struct vnode *vp = bp->b_vp;
    struct nnpfs_node *xn = VNODE_TO_XNODE(vp);
    struct iovec iov;
    struct uio uio;
    struct proc *p = nnpfs_curproc();
    int error, s, rw, nnpfsflags;
    struct ucred *cred;
    
    NNPFSDEB(XDEBVNOPS, ("nnpfs_netbsd_strategy: %p\n", vp));
    NNPFSDEB(XDEBVNOPS, ("nnpfs_netbsd_strategy: from %lld with %d\n", 
			 (long long)dbtob(bp->b_blkno), (int)bp->b_resid));

    bp = ap->a_bp;

    rw = (bp->b_flags & B_READ) == 0;

    nnpfsflags = rw ? NNPFS_DATA_W : NNPFS_DATA_R;
    cred = rw ? xn->wr_cred : xn->rd_cred;

    /* XXX */
    xn->flags |= NNPFS_VMOPEN;

    error = nnpfs_data_valid(vp, cred, p, nnpfsflags, 
			   dbtob(bp->b_blkno) + bp->b_resid);

    xn->flags &= ~NNPFS_VMOPEN;

    if (error) {
	bp->b_error = error;
	bp->b_flags |= B_ERROR;
	biodone(bp);
	NNPFSDEB(XDEBVNOPS, ("nnpfs_netbsd_strategy: failed returns %d\n", error));
	return error;
    }
	
    iov.iov_base = (void *)bp->b_data;
    iov.iov_len = bp->b_resid;
    uio.uio_iov = &iov;
    uio.uio_iovcnt = 1;
    uio.uio_offset = dbtob(bp->b_blkno);
    uio.uio_segflg = UIO_SYSSPACE;
    uio.uio_rw = rw ? UIO_WRITE : UIO_READ;
    uio.uio_resid = bp->b_resid;
    uio.uio_procp = p;
    
    error = VOP_OPEN(DATA_FROM_VNODE(vp), rw ? FWRITE : FREAD, cred, p);

    if (rw) {
	error = VOP_WRITE(DATA_FROM_VNODE(vp), &uio, 0, cred);
	if (error == 0)
	    xn->flags |= NNPFS_DATA_DIRTY;
    } else
	error = VOP_READ(DATA_FROM_VNODE(vp), &uio, 0, cred);

    if (error) {
	bp->b_error = error;
	bp->b_flags |= B_ERROR;
    }

    (void)VOP_CLOSE(DATA_FROM_VNODE(vp), rw ? FWRITE : FREAD, cred, p);

    if (error == 0 && rw) {
	xn->attr.va_size = vp->v_size;
	xn->attr.va_bytes = vp->v_size;
	xn->offset = vp->v_size;
    }

    bp->b_resid = uio.uio_resid;
 done:
    biodone(bp);

    NNPFSDEB(XDEBVNOPS, ("nnpfs_netbsd_strategy: returns %d\n", error));
    return error;
}

static int
nnpfs_netbsd_read(struct vop_read_args *ap)
{
    /* {
       struct vnode *a_vp;
       struct uio *a_uio;
       int a_ioflag;
       struct ucred *a_cred;
    } */
    struct nnpfs_node *xn;
    struct vnode *vp;
    struct uio *uio;
    void *win;
    vsize_t bytelen;
    struct buf *bp;
    long size;
    int error;
    
    vp = ap->a_vp;
    uio = ap->a_uio;
    error = 0;
    xn = VNODE_TO_XNODE(vp);
    
    NNPFSDEB(XDEBVNOPS, ("nnpfs_netbsd_read: %p\n", vp));

    if (vp->v_type != VREG && vp->v_type != VLNK)
	return EISDIR;
    
    if (uio->uio_resid == 0)
	return (0);
    if (uio->uio_offset < 0)
	return EFBIG;
    if (uio->uio_offset > vp->v_size)
	return 0;
    
    nnpfs_update_read_cred(VNODE_TO_XNODE(vp), ap->a_cred);

    error = nnpfs_data_valid(vp, ap->a_cred, nnpfs_uio_to_proc(uio),
			     NNPFS_DATA_R,
			     nnpfs_uio_end_length(uio));

    while (uio->uio_resid > 0) {
	NNPFSDEB(XDEBVNOPS, ("nnpfs_netbsd_read: need to copy %ld\n",
	    (long)uio->uio_resid));

	bytelen = MIN(xn->attr.va_size - uio->uio_offset,
		      uio->uio_resid);
	if (bytelen == 0)
	    break;
	
	NNPFSDEB(XDEBVNOPS, ("nnpfs_netbsd_read: allocating window\n"));

	win = ubc_alloc(&vp->v_uobj, uio->uio_offset,
			&bytelen, UBC_READ);
	NNPFSDEB(XDEBVNOPS, ("nnpfs_netbsd_read: copy data\n"));
	error = uiomove(win, bytelen, uio);
	NNPFSDEB(XDEBVNOPS, ("nnpfs_netbsd_read: release window\n"));
	ubc_release(win, 0);
	if (error)
	    break;
    }
    NNPFSDEB(XDEBVNOPS, ("nnpfs_netbsd_read: done, error = %d\n", error));
    
    return (error);
}

static int
nnpfs_netbsd_write(struct vop_write_args *ap)
{
    struct vnode *vp;
    struct uio *uio;
    struct ucred *cred;
    off_t osize, origoff, oldoff;
    int error, flags, ioflag, resid;
    void *win;
    vsize_t bytelen;
    struct nnpfs_node *xn;

    NNPFSDEB(XDEBVNOPS, ("nnpfs_netbsd_write: %p\n", vp));

    cred = ap->a_cred;
    ioflag = ap->a_ioflag;
    uio = ap->a_uio;
    vp = ap->a_vp;

#ifdef DIAGNOSTIC
    if (uio->uio_rw != UIO_WRITE)
	panic("nnpfs_netbsd_write: mode");
#endif
    if (uio->uio_offset < 0)
	return (EFBIG);

    if (vp->v_type != VREG && vp->v_type != VLNK)
	return EISDIR;

    nnpfs_update_write_cred(VNODE_TO_XNODE(vp), ap->a_cred);

    error = nnpfs_data_valid(vp, ap->a_cred, uio->uio_procp, NNPFS_DATA_W,
	nnpfs_uio_end_length(uio));
    if (error)
	return (error);
    
    xn = VNODE_TO_XNODE(vp);

    if (ioflag & IO_APPEND)
	uio->uio_offset = xn->attr.va_size;

    flags = ioflag & IO_SYNC ? B_SYNC : 0;
    resid = uio->uio_resid;
    osize = xn->attr.va_size;
    error = 0;

    origoff = uio->uio_offset;
    while (uio->uio_resid > 0) {
	oldoff = uio->uio_offset;

	bytelen = MIN(DEV_BSIZE - (uio->uio_offset & (DEV_BSIZE - 1)),
		      uio->uio_resid);

	/*
	 * copy the data.
	 */

	win = ubc_alloc(&vp->v_uobj, uio->uio_offset, &bytelen,
	    UBC_WRITE);
	error = uiomove(win, bytelen, uio);
	ubc_release(win, 0);
	if (error) {
	    break;
	}

	/*
	 * update UVM's notion of the size now that we've
	 * copied the data into the vnode's pages.
	 */

	if (vp->v_size < uio->uio_offset) {
	    uvm_vnp_setsize(vp, uio->uio_offset);
	}

	if (oldoff >> 16 != uio->uio_offset >> 16) {
	    simple_lock(&vp->v_interlock);
	    error = VOP_PUTPAGES(vp, (oldoff >> 16) << 16,
		(uio->uio_offset >> 16) << 16, PGO_CLEANIT);
	    if (error) {
		break;
	    }
	}
    }

    if (error) {
	(void) VOP_TRUNCATE(vp, osize, ioflag & IO_SYNC, ap->a_cred,
	    uio->uio_procp);
	uio->uio_offset -= resid - uio->uio_resid;
	uio->uio_resid = resid;
    } else if (resid > uio->uio_resid && (ioflag & IO_SYNC) == IO_SYNC) {
	error = VOP_UPDATE(vp, NULL, NULL, UPDATE_WAIT);
    }
    if (!error) {
	struct timespec ts;

	xn->attr.va_size = vp->v_size;
	xn->attr.va_bytes = vp->v_size;
	TIMEVAL_TO_TIMESPEC(&time, &ts);
	xn->attr.va_mtime = ts;
	xn->offset = vp->v_size;
	xn->flags |= NNPFS_DATA_DIRTY;
    }

    return (error);
}


/*
 * vnode functions
 */

static struct vnodeopv_entry_desc nnpfs_vnodeop_entries[] = {
    {&vop_default_desc,		(vop_t *) nnpfs_eopnotsupp},
    {&vop_lookup_desc,		(vop_t *) nnpfs_lookup },
    {&vop_open_desc,		(vop_t *) nnpfs_open },
    {&vop_fsync_desc,		(vop_t *) nnpfs_fsync },
    {&vop_close_desc,		(vop_t *) nnpfs_close },
    {&vop_read_desc,		(vop_t *) nnpfs_netbsd_read },
    {&vop_write_desc,		(vop_t *) nnpfs_netbsd_write },
    {&vop_mmap_desc,		(vop_t *) nnpfs_mmap },
    {&vop_ioctl_desc,		(vop_t *) nnpfs_ioctl },
    {&vop_seek_desc,		(vop_t *) nnpfs_seek },
    {&vop_poll_desc,		(vop_t *) nnpfs_poll },
    {&vop_getattr_desc,		(vop_t *) nnpfs_getattr },
    {&vop_setattr_desc,		(vop_t *) nnpfs_setattr },
    {&vop_access_desc,		(vop_t *) nnpfs_access },
    {&vop_create_desc,		(vop_t *) nnpfs_create },
    {&vop_remove_desc,		(vop_t *) nnpfs_remove },
    {&vop_link_desc,		(vop_t *) nnpfs_link },
    {&vop_rename_desc,		(vop_t *) nnpfs_rename },
    {&vop_mkdir_desc,		(vop_t *) nnpfs_mkdir },
    {&vop_rmdir_desc,		(vop_t *) nnpfs_rmdir },
    {&vop_readdir_desc,		(vop_t *) nnpfs_readdir },
    {&vop_symlink_desc,		(vop_t *) nnpfs_symlink },
    {&vop_readlink_desc,	(vop_t *) nnpfs_readlink },
    {&vop_inactive_desc,	(vop_t *) nnpfs_inactive },
    {&vop_reclaim_desc,		(vop_t *) nnpfs_reclaim },
    {&vop_lock_desc,		(vop_t *) nnpfs_lock },
    {&vop_unlock_desc,		(vop_t *) nnpfs_unlock },
    {&vop_islocked_desc,	(vop_t *) nnpfs_islocked },
    {&vop_abortop_desc,		(vop_t *) nnpfs_abortop },
    {&vop_getpages_desc,	(vop_t *) genfs_getpages },
    {&vop_putpages_desc,	(vop_t *) genfs_putpages },
    {&vop_revoke_desc,		(vop_t *) genfs_revoke },
    {&vop_bmap_desc,		(vop_t *) nnpfs_netbsd_bmap },
    {&vop_strategy_desc,	(vop_t *) nnpfs_netbsd_strategy },
    {&vop_print_desc,		(vop_t *) nnpfs_print}, 
    {&vop_advlock_desc,		(vop_t *) nnpfs_advlock },
    {(struct vnodeop_desc *) NULL, (int (*) (void *)) NULL}
};

struct vnodeopv_desc nnpfs_netbsd_vnodeop_opv_desc =
{&nnpfs_vnodeop_p, nnpfs_vnodeop_entries};

#endif
